<?php

App::uses('BuzzVouchersAppModel', 'BuzzVouchers.Model');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('GiftVoucherPurchaseState', 'BuzzVouchers.Model');
App::uses('Discount', 'BuzzDiscount.Lib');

class GiftVoucherPurchase extends BuzzVouchersAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

/**
 * Default order
 *
 * @var array|string
 */
	public $order = 'GiftVoucherPurchase.id DESC';

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'sales_ref';

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		),
		'DeliveryAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress',
			'foreignKey' => 'delivery_address_id'
		),
		'GiftVoucherDeliveryOption' => array(
			'className' => 'BuzzVouchers.GiftVoucherDeliveryOption'
		),
		'GiftVoucherPurchaseState' => array(
			'className' => 'BuzzVouchers.GiftVoucherPurchaseState'
		)
	);

/**
 * Has many associations
 *
 * @var array
 */
	public $hasMany = array(
		'GiftVoucherPurchaseItem' => array(
			'className' => 'BuzzVouchers.GiftVoucherPurchaseItem'
		)
	);

/**
 * Read for edit
 *
 * @param int $id Record ID
 * @param array $params
 * @return array
 */
	public function readForEdit($id, $params = []) {
		$defaults = array(
			'contain' => array(
				'CustomerAddress' => array(
					'Country',
					'UsState'
				),
				'DeliveryAddress' => array(
					'Country',
					'UsState'
				),
				'GiftVoucherPurchaseItem' => array(
					'GiftVoucher',
					'GiftVoucherPurchaseItemVoucher'
				),
				'GiftVoucherDeliveryOption'
			)
		);
		$params = Hash::merge($defaults, $params);

		return parent::readForEdit($id, $params);
	}

/**
 * Read for view
 *
 * @param int $id Record ID
 * @param array $params
 * @return array
 */
	public function readForView($id, $params = []) {
		$defaults = array(
			'contain' => array(
				'CustomerAddress' => array(
					'Country',
					'UsState'
				),
				'DeliveryAddress' => array(
					'Country',
					'UsState'
				),
				'GiftVoucherPurchaseItem',
				'GiftVoucherDeliveryOption'
			)
		);
		$params = Hash::merge($defaults, $params);

		$data = parent::readForEdit($id, $params);

		// We need to separately add on the gift vouchers to ensure we get all
		// the translated data. This would get missed in the first query as we
		// are querying to deep for the Translable behaviour.
		if (!empty($data['GiftVoucherPurchaseItem'])) {
			$this->GiftVoucher = $this->GiftVoucherPurchaseItem->GiftVoucher;
			$giftVouchers = $this->GiftVoucherPurchaseItem->GiftVoucher->find(
				'all',
				array(
					'contain' => array(
						'VoucherActivity'
					),
					'conditions' => array(
						'GiftVoucher.id' => Hash::extract($data, 'GiftVoucherPurchaseItem.{n}.gift_voucher_id')
					)
				)
			);
			foreach ($giftVouchers as &$value) {
				$value['GiftVoucher']['VoucherActivity'] = $value['VoucherActivity'];
			}
			$giftVouchers = Hash::combine($giftVouchers, '{n}.GiftVoucher.id', '{n}.GiftVoucher');
			foreach ($data['GiftVoucherPurchaseItem'] as &$value) {
				$giftVoucherId = $value['gift_voucher_id'];
				$value['GiftVoucher'] = $giftVouchers[$giftVoucherId];
			}
		}

		return $data;
	}

/**
 * Get the current basket
 *
 * @return array
 */
	public function getBasket() {
		$basketId = $this->getBasketId();
		if (empty($basketId)) {
			return [];
		}
		$params = array(
			'conditions' => array(
				'GiftVoucherPurchase.gift_voucher_purchase_state_id' => GiftVoucherPurchaseState::UNPAID
			)
		);
		$data = $this->readForView($basketId, $params);

		if (!empty($data)) {
			$data['GiftVoucherPurchase']['total_items'] = array_sum(
				Hash::extract(
					$data,
					'GiftVoucherPurchaseItem.{n}.quantity'
				)
			);
		}

		return $data;
	}

/**
 * Get a gift voucher purchase to display to the customer.
 *
 * @param int $id Purchase ID
 * @return array
 */
	public function getPurchase($id) {
		$params = array(
			'contain' => array(
				'GiftVoucherPurchaseItem' => array(
					// Need to make sure we retrieve the purchased vouchers.
					'GiftVoucherPurchaseItemVoucher'
				)
			),
			'conditions' => array(
				'GiftVoucherPurchase.gift_voucher_purchase_state_id' => array(
					GiftVoucherPurchaseState::COMPLETE,
					GiftVoucherPurchaseState::API_FAILED
				)
			)
		);
		return $this->readForView($id, $params);
	}

/**
 * Adds items to the basket
 *
 * @param array $items
 * @return bool
 */
	public function addItems(array $items) {
		$basketId = $this->getBasketId();

		// Make sure we don't save items with zero quantity to the database.
		$removeItems = [];
		foreach ($items as $key => $val) {
			if ((int)$val['quantity'] === 0) {
				unset($items[$key]);
			}
			// Flag existing items that are getting removed.
			if (!empty($val['id'])) {
				$removeItems[] = $val['id'];
			}
		}

		// Remove unwanted items.
		if (!empty($removeItems)) {
			$this->GiftVoucherPurchaseItem->deleteAll(['GiftVoucherPurchaseItem.id' => $removeItems]);
			unset($removeItems);
		}

		// Get the details of the gift vouchers being purchased so that we can
		// calculate costs.
		$vouchers = $this->GiftVoucherPurchaseItem->GiftVoucher->find(
			'all',
			array(
				'conditions' => array(
					'GiftVoucher.id' => Hash::extract($items, '{n}.gift_voucher_id')
				)
			)
		);
		$vouchers = Hash::combine($vouchers, '{n}.GiftVoucher.id', '{n}.GiftVoucher');

		foreach ($items as $key => &$item) {

			$giftVoucherId = $item['gift_voucher_id'];

			if (!empty($vouchers[$giftVoucherId])) {
				$item['item_unit_cost'] = $vouchers[$giftVoucherId]['price'];
				$item['total'] = $item['quantity'] * $vouchers[$giftVoucherId]['price'];
			} else {
				unset($items[$key]);
			}

		}

		if ($basketId === null) {
			// Create a new basket.
			$data = array(
				'GiftVoucherPurchase' => array(
					'id' => $basketId,
					'gift_voucher_purchase_state_id' => GiftVoucherPurchaseState::UNPAID
				),
				'GiftVoucherPurchaseItem' => $items
			);

			$result = $this->saveAssociated($data) !== false;
			$basketId = $result === true ? $this->id : null;

		} else {
			// Make sure we save to the current basket.
			$items = Hash::insert($items, '{n}.gift_voucher_purchase_id', $basketId);
			$result = $this->GiftVoucherPurchaseItem->saveMany($items) !== false;

		}

		// Update the basket total and session.
		if ($result === true) {
			CakeSession::write('GiftVoucherPurchase.id', $basketId);
			$this->_updateBasketTotal();
		} else {
			$this->clearBasketId();
		}

		return $result;
	}

/**
 * Update items in the basket
 *
 * @param array $items
 * @return bool
 */
	public function updateItems(array $items) {
		$basketId = $this->getBasketId();

		// Get the current basket items.
		$basketItems = $this->GiftVoucherPurchaseItem->find(
			'all',
			['conditions' => ['GiftVoucherPurchaseItem.gift_voucher_purchase_id' => $basketId]]
		);
		$basketItems = Hash::combine($basketItems, '{n}.GiftVoucherPurchaseItem.id', '{n}.GiftVoucherPurchaseItem');

		// Make sure we don't save items with zero quantity to the database.
		$data = [];
		$removeItems = [];
		foreach ($items as $key => $val) {
			if ((int)$val['quantity'] > 0 && !empty($basketItems[$key])) {
				// Update item
				$val['total'] = $basketItems[$key]['item_unit_cost'] * $val['quantity'];
				$data[] = $val;
			} else {
				// Remove item
				$removeItems[] = $val['id'];
				unset($items[$key]);
			}
		}

		// Remove unwanted items.
		if (!empty($removeItems)) {
			$this->GiftVoucherPurchaseItem->deleteAll(['GiftVoucherPurchaseItem.id' => $removeItems]);
			unset($removeItems);
		}

		// Save the updated basket items.
		$result = $this->GiftVoucherPurchaseItem->saveMany($data) !== false;

		if ($result === true) {
			$this->_updateBasketTotal();
		}

		return $result;
	}

/**
 * Remove an item from the basket
 *
 * @param int $itemId
 * @return bool
 */
	public function removeItem($itemId) {
		$result = $this->GiftVoucherPurchaseItem->deleteAll([
			'GiftVoucherPurchaseItem.id' => $itemId,
			'GiftVoucherPurchaseItem.gift_voucher_purchase_id' => $this->getBasketId()
		]);
		if ($result === true) {
			$this->_updateBasketTotal();
		}
		return $result;
	}

/**
 * Update the current basket total.
 *
 * @return bool
 */
	protected function _updateBasketTotal() {
		$basket = $this->getBasket();

		if (empty($basket)) {
			return false;
		}

		// Calculate the subtotal.
		$subtotal = array_sum(Hash::extract($basket, 'GiftVoucherPurchaseItem.{n}.total'));

		// Calculate the delivery cost.
		if (!empty($basket['GiftVoucherDeliveryOption']['price'])) {
			$deliveryCost = $basket['GiftVoucherPurchase']['total_items'] * $basket['GiftVoucherDeliveryOption']['price'];
		} else {
			$deliveryCost = 0;
		}

		$total = $subtotal + $deliveryCost;

		// Deduct any discount on the booking
		if (! empty($basket['GiftVoucherPurchase']['discount'])) {
			$total -= $basket['GiftVoucherPurchase']['discount'];
		}

		if ($total < 0) {
			$total = 0;
		}

		// Save the update total.
		$data = array(
			'id' => $basket['GiftVoucherPurchase']['id'],
			'subtotal' => $subtotal,
			'delivery_cost' => $deliveryCost,
			'total_cost' => $total
		);
		return $this->save($data) !== false;
	}

/**
 * Add the customer delivery details to the order
 *
 * @param int $deliveryOptionId
 * @param array $address
 * @param string $giftMessage
 * @return bool
 */
	public function addDeliveryDetails($deliveryOptionId, array $address, $giftMessage = null) {
		$delivery = $this->GiftVoucherDeliveryOption->findById($deliveryOptionId);
		if (empty($delivery)) {
			$this->invalidate('gift_voucher_delivery_option_id', 'Please select a delivery option');
			return false;
		}
		$data = array(
			'GiftVoucherPurchase' => array(
				'id' => $this->getBasketId(),
				'gift_voucher_delivery_option_id' => $deliveryOptionId,
			)
		);
		if ($delivery['GiftVoucherDeliveryOption']['online'] !== true) {
			$data['GiftVoucherPurchase']['gift_message'] = $giftMessage;
			$data['DeliveryAddress'] = $address;
		}
		$result = $this->saveAssociated($data) !== false;
		$this->_updateBasketTotal();
		return $result;
	}

/**
 * Add the customer billing details to the order
 *
 * @param array $customer
 * @return bool
 */
	public function addBillingDetails(array $customer) {
		$data = array(
			'GiftVoucherPurchase' => array(
				'id' => $this->getBasketId()
			),
			'CustomerAddress' => $customer
		);
		return $this->saveAssociated($data) !== false;
	}

/**
 * Complete the purchase
 *
 * @param bool $paid True if payment successfully made
 * @return bool
 */
	public function completePurchase($paid) {
		$id = $this->getBasketId();

		$basket = $this->getBasket($id);

		$response = true;

		if ((int)$basket['GiftVoucherPurchase']['gift_voucher_purchase_state_id'] === GiftVoucherPurchaseState::UNPAID) {
			if ($paid === true) {
				// Create the vouchers
				$result = $this->_createVouchers($basket);

				$data = array(
					'id' => $id,
					'gift_voucher_purchase_state_id' => $result === true ? GiftVoucherPurchaseState::COMPLETE : GiftVoucherPurchaseState::API_FAILED,
					'completed_date' => gmdate('Y-m-d H:i:s')
				);

				if ($result === false) {
					$data['api_log'] = json_encode(ClassRegistry::init('BuzzVouchers.VoucherApi')->getApiResponses());
				}

				$response = $this->save($data);

				if (! empty($basket['GiftVoucherPurchase']['discount_code']) && $basket['GiftVoucherPurchase']['discount'] > 0) {
					$discountInfo = ClassRegistry::init('BuzzBookings.BookingApi')->checkDiscountCode($basket['GiftVoucherPurchase']['discount_code']);

					$type = 'F';

					if (! empty($discountInfo['PercentageDiscount']) && $discountInfo['PercentageDiscount'] > 0) {
						$type = 'P';
					}

					ClassRegistry::init('BuzzBookings.BookingApi')->applyDiscountToBooking(
						'GiftVoucherPurchase',
						$basket['GiftVoucherPurchase']['id'],
						$basket['GiftVoucherPurchase']['sales_ref'],
						$basket['GiftVoucherPurchase']['discount_code'],
						$basket['GiftVoucherPurchase']['discount'],
						$type
					);
				}




			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'gift_voucher_purchase_state_id' => GiftVoucherPurchaseState::PAYMENT_FAILED
				);
				$response = $this->save($data);

			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.GiftVoucherPurchase.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);

		}

		return $response;
	}

/**
 * Create vouchers
 *
 * @param array $basket
 * @return bool
 */
	protected function _createVouchers(array $basket) {
		$data = [];
		$response = true;

		// Set sales ref ready to pass to API (this will get set by the API
		// after the creation of the first voucher).
		$salesRef = null;

		$deliveryCost = $basket['GiftVoucherPurchase']['delivery_cost'] / $basket['GiftVoucherPurchase']['total_items'];
		foreach ($basket['GiftVoucherPurchaseItem'] as $item) {

			$totalPaid = $item['total'];
			// We include the delivery cost in the amount paid passed to the API.
			if ($deliveryCost > 0) {
				$totalPaid += $deliveryCost * $item['quantity'];
			}

			$result = ClassRegistry::init('BuzzVouchers.VoucherApi')->createVoucher(
				$this->name,
				$this->getBasketId(),
				$basket['CustomerAddress']['first_name'],
				$basket['CustomerAddress']['last_name'],
				$basket['CustomerAddress']['email'],
				$basket['CustomerAddress']['telephone'],
				CustomerAddress::generateAddressArray(
					// For postal delivery we want to pass the delivery address
					// to the API; for evouchers the customer's address.
					$basket['GiftVoucherDeliveryOption']['online'] === true ? $basket['CustomerAddress'] : $basket['DeliveryAddress']
				),
				$basket['GiftVoucherDeliveryOption']['online'] === false ? $basket['GiftVoucherPurchase']['gift_message'] : null,
				$item['GiftVoucher']['api_reference'],
				$item['quantity'],
				$basket['GiftVoucherDeliveryOption']['online'],
				null,
				$this->getPaymentDetails($basket['GiftVoucherPurchase']['id'], 'BuzzVouchers.GiftVoucherPurchase'),
				// If the basket total is zero pass 0.01 to the API to avoid
				// API errors. This is a temporary fix until the API gets
				// fixed.
				$totalPaid > 0 ? $totalPaid : 0.01,
				$salesRef
			);

			if ($result !== false) {
				$salesRef = $result['Voucher'][0]['sales_order_id'];
				foreach ($result['Voucher'] as $voucher) {
					$data[] = array(
						'gift_voucher_purchase_item_id' => $item['id'],
						'description' => $voucher['description'],
						'code' => $voucher['code'],
						'expiry_date' => $voucher['expiry_date']
					);
				}
			} else {
				$response = false;
			}

		}

		if (!empty($data)) {
			$this->id = $this->getBasketId();
			$this->saveField('sales_ref', $salesRef);
			$this->GiftVoucherPurchaseItem->GiftVoucherPurchaseItemVoucher->saveMany($data);
		}

		return $response;
	}

	/**
	 * API call to check discount codes (to enable unit test mocking).
	 *
	 * @param string $discountCode
	 * @return array
	 */
	protected function _checkDiscountCodeApiCall($discountCode) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->checkDiscountCode(
			$discountCode
		);
	}

	/**
	 * Add a discount to the current order. The process for this is to first check if
	 * the discount code entered is valid, and if it is, we then need to drill down
	 * through the basket items to see which ones can have the discount applied.
	 *
	 * @param  integer $bookingId    id of the booking to apply this to
	 * @param  string $discountCode the discount code to apply
	 * @return boolean returns true if we're able to apply the discount to anything.
	 */
		public function applyDiscountCode($bookingId, $discountCode) {
			$basket = $this->getBasket($bookingId);

			// Validate discount
			$discount = $this->_checkDiscountCodeApiCall($discountCode);

			if (! $discount) {
				return false;
			}

			// At this point we know the discount code is valid, now we need to go
			// through the items in the basket to see which ones it can be applied
			// to.

			$discountLib = new Discount($discount, $basket);

			$basketData = $discountLib->applyDiscount('voucher');

			if (! empty($basketData['discount'])) {
				$this->save(array(
					'id' => $bookingId,
					'discount' => $basketData['discount'],
					'discount_code' => $discountCode
				));
			}

			$this->_updateBasketTotal();

			return true;


		}


}
