<?php

App::uses('BuzzVouchersAppController', 'BuzzVouchers.Controller');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class GiftVoucherPurchasesController extends BuzzVouchersAppController implements PurchasableInterface {

	/**
	 * List of admin actions the controller supports
	 *
	 * @var array
	 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_email_confirmation'
	);

	/**
	 * Helpers
	 */
	public $helpers = array(
		'Address.Address'
	);

	/**
	 * Components
	 *
	 * @var array
	 */
	public $components = array(
		'BuzzPurchase.Purchase',
		'BuzzPurchase.GaEcommerce',
		'Transactions.Transactions'
	);

	/**
	 * Before filter
	 *
	 * @return void
	 */
	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'vouchers',
			'basket',
			'remove',
			'delivery',
			'checkout',
			'confirmation',
			// Payment gateway callback
			'payment_callback'
		));

		return;
	}

	/**
	 * Gift vouchers landing page - will mostly get bypassed by homepage/activity pages
	 *
	 * @return void
	 */
	public function index() {
		if (!empty($this->request->data)) {

			if (!empty($this->request->data['GiftVoucherPurchase']['voucher_activity_id'])) {
				// Forward to the correct vouchers step.
				return $this->redirect([
					'action' => 'vouchers',
					$this->request->data['GiftVoucherPurchase']['voucher_activity_id']
				]);
			} else {
				$this->Session->setFlash(
					__d('buzz_vouchers', 'Please select an activity'),
					'flash_fail'
				);
			}

		}

		$this->loadModel('BuzzVouchers.VoucherActivity');
		$voucherActivities = $this->VoucherActivity->getListed();

		if (count($voucherActivities) === 1) {
			return $this->redirect([
				'action' => 'vouchers'
			]);
		} elseif (empty($voucherActivities)) {
			// No activities have been setup or created so throw an exception
			throw new NotFoundException();
		}

		$this->set(compact('voucherActivities'));
		$this->view = 'BuzzVouchers./GiftVoucherPurchases/index';

		return;
	}

	/**
	 * Vouchers step - add gift vouchers to the basket
	 *
	 * @param int $voucherActivityId Voucher activity ID
	 * @return void
	 */
	public function vouchers($voucherActivityId = null) {
		$GiftVoucherPurchase = $this->{$this->modelClass};
		$this->loadModel('BuzzVouchers.VoucherActivity');

		if (!empty($this->request->data)) {

			$result = $GiftVoucherPurchase->addItems($this->request->data['GiftVoucherPurchaseItem']);
			if ($result === true) {
				return $this->redirect(['action' => 'basket']);
			} else {
				$this->Session->setFlash(
					__d('buzz_vouchers', 'Please select a voucher'),
					'flash_fail'
				);
			}

		}

		// Get the basket.
		$basket = $GiftVoucherPurchase->getBasket();

		if (empty($voucherActivityId)) {
			// Check if the site is configured to only have one listed
			// voucher activity. This is needed to prevent a redirect
			// loop with this and the previous step.
			$voucherActivityId = $this->VoucherActivity->onlyAvailableActivity();
		}

		if ($voucherActivityId === false) {
			throw new NotFoundException();
		}

		// Get the vouchers and activity details for generating the form.
		$data = $this->VoucherActivity->getPurchasableVoucherActivity($voucherActivityId);

		if (empty($data)) {
			throw new NotFoundException();
		}

		if (empty($this->request->data['GiftVoucherPurchaseItem']) && !empty($basket['GiftVoucherPurchaseItem'])) {
			// Populate the form with existing basket items.
			$this->request->data['GiftVoucherPurchaseItem'] = Hash::combine(
				$basket['GiftVoucherPurchaseItem'],
				'{n}.gift_voucher_id',
				'{n}'
			);
		}

		$previousStep = ['action' => 'index'];

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($data['VoucherActivity']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->VoucherActivity->Gallery->readForView($data['VoucherActivity']['gallery_id'])
			);
		}

		$this->loadModel('BuzzVouchers.VoucherActivity');
		$voucherActivities = $this->VoucherActivity->getListed();

		$this->set(compact('data', 'basket', 'previousStep', 'voucherActivities'));
		$this->view = 'BuzzVouchers./GiftVoucherPurchases/vouchers_full';

		return;
	}

	/**
	 * Basket step - view the basket
	 *
	 * @return void
	 */
	public function basket() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		if (!empty($this->request->data['GiftVoucherPurchase']['voucher_activity_id'])) {
			// Redirect to the activity so that the user can add more vouchers.
			return $this->redirect([
				'action' => 'vouchers',
				$this->request->data['GiftVoucherPurchase']['voucher_activity_id']
			]);
		} elseif (!empty($this->request->data)) {
			// Update the basket.
			$GiftVoucherPurchase->updateItems($this->request->data['GiftVoucherPurchaseItem']);

			if (! empty($this->request->data['GiftVoucherPurchase']['discount_code'])) {
				if (
					$GiftVoucherPurchase->applyDiscountCode($GiftVoucherPurchase->id,
					$this->request->data['GiftVoucherPurchase']['discount_code'])) {

					$this->Session->setFlash(
						__d('buzz_bookings', 'Your discount has been applied to your basket.'),
						'flash_success'
					);
				} else {
					$this->Session->setFlash(
						__d('buzz_bookings', 'The discount code entered was not valid for any of the items in your basket.'),
						'flash_fail'
					);
				}

				return $this->redirect($this->here);
			}
		}

		// Get the basket.
		$basket = $GiftVoucherPurchase->getBasket();

		// Populate the form.
		if (empty($this->request->data)) {
			$this->request->data = $basket;
		}

		if (empty($basket['GiftVoucherPurchaseItem'])) {
			$this->Session->setFlash(__d('buzz_vouchers', 'Your basket is empty'), 'flash_fail');
			return $this->redirect(['action' => 'index']);
		}

		$previousStep = ['action' => 'index'];
		if (!empty($basket['GiftVoucherPurchaseItem'])) {
			$items = Hash::sort($basket['GiftVoucherPurchaseItem'], '{n}.modified', 'desc');
			$previousStep = array(
				'action' => 'vouchers',
				$items[0]['GiftVoucher']['voucher_activity_id']
			);
		}
		$nextStep = ['action' => 'delivery'];

		// Get the voucher activities to populate the form.
		$voucherActivities = ClassRegistry::init('BuzzVouchers.VoucherActivity')->getListed();

		$this->set(compact('basket', 'previousStep', 'nextStep', 'voucherActivities'));
		$this->view = 'BuzzVouchers./GiftVoucherPurchases/basket';

		$this->set('nextStep', array(
			'admin' => false,
			'plugin' => 'buzz_vouchers',
			'controller' => 'gift_voucher_purchase',
			'action' => 'checkout'
		));

		if (Configure::read('BuzzDiscount.disableDiscountForm') != true) {
			$this->set('showDiscountForm', true);
		}

		return;
	}

	/**
	 * Remove item from the basket.
	 *
	 * @param int $id GiftVoucherPurchaseItem record ID
	 * @return void
	 */
	public function remove($id) {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		if ($GiftVoucherPurchase->removeItem($id) === false) {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'Unable to remove item'),
				'flash_fail'
			);
		}

		return $this->redirect(['action' => 'basket']);
	}

	/**
	 * Delivery step
	 *
	 * @return void
	 */
	public function delivery() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		// Get the basket.
		$basket = $GiftVoucherPurchase->getBasket();

		if (empty($basket['GiftVoucherPurchaseItem'])) {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'index'
			]);
		}

		if (!empty($this->request->data['GiftVoucherPurchase']['gift_voucher_delivery_option_id'])) {

			$deliveryOption = (int)$this->request->data['GiftVoucherPurchase']['gift_voucher_delivery_option_id'];
			if (!empty($this->request->data['GiftVoucherPurchase'][$deliveryOption]['gift_message'])) {
				$giftMessage = $this->request->data['GiftVoucherPurchase'][$deliveryOption]['gift_message'];
			} else {
				$giftMessage = null;
			}

			$result = $GiftVoucherPurchase->addDeliveryDetails(
				$deliveryOption,
				empty($this->request->data['CustomerAddress'][$deliveryOption]) ? [] : $this->request->data['CustomerAddress'][$deliveryOption],
				$giftMessage
			);
			if ($result === true) {
				return $this->redirect(['action' => 'checkout']);
			} else {
				$this->Session->setFlash(
					__d('buzz_vouchers', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} elseif (!empty($this->request->data)) {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'Please select a delivery option'),
				'flash_fail'
			);
		}

		$deliveryOptions = $GiftVoucherPurchase->GiftVoucherDeliveryOption->getOptions();

		if (empty($this->request->data)) {

			$this->request->data = $basket;

			$deliveryOptionIds = Hash::extract($deliveryOptions, '{n}.GiftVoucherDeliveryOption.id');

			if (empty($basket['GiftVoucherPurchase']['delivery_address_id'])) {
				// Set the address field defaults (these are configured by the
				// CustomerAddress plugin to make it easier to override for each site).
				foreach ($deliveryOptionIds as $deliveryOptionId) {
					$this->request->data['CustomerAddress'][$deliveryOptionId]['country_id'] = CustomerAddress::getDefaultCountry();
					$this->request->data['CustomerAddress'][$deliveryOptionId]['us_state_id'] = CustomerAddress::getDefaultUsState();
				}
			} else {
				// Need to set the correct values for the address fields and gift message.
				foreach ($deliveryOptionIds as $deliveryOptionId) {
					$this->request->data['CustomerAddress'][$deliveryOptionId] = $basket['DeliveryAddress'];
					$this->request->data['GiftVoucherPurchase'][$deliveryOptionId]['gift_message'] = $basket['GiftVoucherPurchase']['gift_message'];
				}
			}

		}

		$previousStep = ['action' => 'basket'];

		$this->set(compact('basket', 'deliveryOptions', 'previousStep'));
		$this->_populateLookups();
		$this->view = 'BuzzVouchers./GiftVoucherPurchases/delivery';

		return;
	}

	/**
	 * Checkout step
	 *
	 * @return void
	 */
	public function checkout() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		// Get the basket.
		$basket = $GiftVoucherPurchase->getBasket();

		if (empty($basket['GiftVoucherPurchaseItem'])) {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'index'
			]);
		} elseif (empty($basket['GiftVoucherPurchase']['gift_voucher_delivery_option_id'])) {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'Please select a delivery option'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'delivery'
			]);
		}

		if (!empty($this->request->data)) {

			if (!empty($this->request->data['GiftVoucherPurchase']['use_delivery_address'])) {
				$address = $basket['DeliveryAddress'];
				// Make sure we still save some of the booking details to the new customer address.
				// Most importantly we use the billing address' ID (otherwise we overwrite the
				// delivery address details).
				$address['id'] = $this->request->data['CustomerAddress']['id'];
				$address['first_name'] = $this->request->data['CustomerAddress']['first_name'];
				$address['last_name'] = $this->request->data['CustomerAddress']['last_name'];
				$address['telephone'] = $this->request->data['CustomerAddress']['telephone'];
				$address['email'] = $this->request->data['CustomerAddress']['email'];
			} else {
				$address = $this->request->data['CustomerAddress'];
			}

			$result = $GiftVoucherPurchase->addBillingDetails($address);
			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();
			if ($result === true && $paymentValidates === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_vouchers', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
			$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();

		}

		// Get the basket.
		$basket = $GiftVoucherPurchase->getBasket();

		// Get gift voucher purchase conditions
		$this->loadModel('BuzzConditions.Condition');
		$conditions = $this->Condition->getConditions('GiftVoucherPurchase');

		$previousStep = ['action' => 'delivery'];

		$this->set(compact('basket', 'conditions', 'previousStep'));
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();
		$this->view = 'BuzzVouchers./GiftVoucherPurchases/checkout';

		return;
	}

	/**
	 * Processes the payment and triggers the completion of the payment on success
	 *
	 * @return void
	 */
	protected function _pay() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		// Get the latest basket.
		$basket = $GiftVoucherPurchase->getBasket();

		if ($basket['GiftVoucherPurchase']['total_cost'] > 0) {

			if (Configure::read('BuzzPurchase.onsite') === true) {
				$payment = $this->request->data['Payment'];

				// Split name into first and second names for payment gateways.
				$fullName = trim($payment['card_name']);
				$splitName = explode(' ', $fullName);
				$payment['last_name'] = array_pop($splitName);
				$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
			} else {
				$payment = [];
			}

			$transaction = $this->Transactions->takePayment(
				// Return URLs
				array(
					'return' => Router::url(['action' => 'payment_callback'], true),
					'cancel' => Router::url(['action' => 'checkout'], true)
				),
				// Calling record
				array(
					'model' => 'BuzzVouchers.GiftVoucherPurchase',
					'model_id' => $basket['GiftVoucherPurchase']['id']
				),
				// Amount to be paid
				$basket['GiftVoucherPurchase']['total_cost'],
				// Items
				array(),
				// Extras
				array(
					'language' => Configure::read('Config.language'),
					'card' => $payment,
					'address' => $basket['CustomerAddress'],
					'user' => array('email' => $basket['CustomerAddress']['email']),
					'description' => __d(
						'buzz_vouchers',
						'%s Gift Voucher Purchase %d',
						[Configure::read('SiteSetting.site_title'), $basket['GiftVoucherPurchase']['id']]
					)
				),
				0,
				Configure::read('Transactions.gateway')
			);

			if (!empty($transaction['result'])) {
				// Payment taken successfully, complete purchase.
				return $this->_completePurchase();
			} else {
				$this->Session->setFlash(
					__d('buzz_vouchers', 'There was a problem processing your payment, please check your details and try again'),
					'flash_fail'
				);
			}

		} else {
			// Nothing to pay, complete purchase.
			return $this->_completePurchase();
		}


		return;
	}

	/**
	 * Confirmation page
	 *
	 * @param int $id Basket ID
	 * @return void
	 */
	public function confirmation($id) {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		if ($GiftVoucherPurchase->hashBasketId($id) === $this->Session->read('GiftVoucherPurchase.hash')) {

			$this->Session->delete('GiftVoucherPurchase.hash');

			$basket = $GiftVoucherPurchase->getPurchase($id);

			// Check that a sale has been generated by the API before
			// proceeding. This should never happen if the API is up and
			// running and the site is correctly configured.
			if (empty($basket['GiftVoucherPurchase']['sales_ref'])) {
				throw new InternalErrorException(__d('buzz_vouchers', 'Something went wrong'));
			}

			if (Configure::check('BuzzVouchers.confirmation_page_id') === true) {
				$this->assignPage(Configure::read('BuzzVouchers.confirmation_page_id'));
			}

			// Set up the GA ecommerce tracking variables.
			$gaTransaction = $this->GaEcommerce->transaction(
				$basket['GiftVoucherPurchase']['sales_ref'],
				$basket['GiftVoucherPurchase']['total_cost'],
				$basket['GiftVoucherPurchase']['delivery_cost']
			);
			foreach ($basket['GiftVoucherPurchaseItem'] as $item) {
				$this->GaEcommerce->addItem(
					$basket['GiftVoucherPurchase']['sales_ref'],
					$item['GiftVoucher']['name'],
					'V' . $item['gift_voucher_id'],
					$item['item_unit_cost'],
					$item['quantity'],
					'Voucher'
				);
			}
			$gaItems = $this->GaEcommerce->items();
			$this->set(compact('gaTransaction', 'gaItems'));

			$this->set(compact('basket'));
			$this->view = 'BuzzVouchers./GiftVoucherPurchases/confirmation';

		} else {

			throw new ForbiddenException();

		}

		return;
	}

	/**
	 * Complete the purchase (after payment received)
	 *
	 * @return void Redirects to confirmation page
	 */
	protected function _completePurchase() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		$basketId = $GiftVoucherPurchase->getBasketId();

		if (empty($basketId)) {
			throw new ForbiddenException();
		}

		$this->Session->write('GiftVoucherPurchase.hash', $GiftVoucherPurchase->hashBasketId());

		$GiftVoucherPurchase->completePurchase(true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	/**
	 * Populate lookups for address fields.
	 *
	 * @return void
	 */
	protected function _populateLookups() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$usStates = $GiftVoucherPurchase->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));

		return;
	}

	/**
	 * Generic payment gateway callback.
	 *
	 * @return void
	 */
	public function payment_callback() {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completePurchase();
		} else {
			$this->Session->setFlash(
				__d('buzz_vouchers', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'checkout'
			]);
		}
	}

	/**
	 * Admin index paginate
	 *
	 * @return array
	 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'CustomerAddress',
				'GiftVoucherPurchaseState'
			)
		);

		return $paginate;
	}

	/**
	 * Admin index columns
	 *
	 * @return array
	 */
	protected function _adminIndexColumns() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		$columns = parent::_adminIndexColumns();

		// Remove created/modified columns.
		unset($columns['GiftVoucherPurchase.created']);
		unset($columns['GiftVoucherPurchase.modified']);

		$columns[$GiftVoucherPurchase->alias . '.total_cost']['type'] = 'currency';

		$newColumns = array(
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_vouchers', 'Customer'),
				'type' => 'string'
			),
			'GiftVoucherPurchaseState.name' => array(
				'label' => __d('buzz_vouchers', 'Status'),
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'GiftVoucherPurchase.sales_ref', $newColumns);
	}

	/**
	 * Admin columns whitelist
	 *
	 * @return array
	 */
	protected function _adminIndexColumnsWhitelist() {
		$GiftVoucherPurchase = $this->{$this->modelClass};

		$whitelist = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $GiftVoucherPurchase->alias . '.sales_ref';
		$whitelist[] = $GiftVoucherPurchase->alias . '.total_cost';
		$whitelist[] = $GiftVoucherPurchase->alias . '.completed_date';

		return $whitelist;
	}

	/**
	 * Filters
	 *
	 * @return array
	 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['GiftVoucherPurchase.name']);
		unset($filters['GiftVoucherPurchase.created']);
		unset($filters['GiftVoucherPurchase.modified']);

		$newFilters = array(
			'GiftVoucherPurchase.sales_ref' => array(
				'label' => __d('buzz_vouchers', 'Sales Ref'),
				'type' => 'string',
				'compare' => array('GiftVoucherPurchase.sales_ref' => "%s")
			),
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_vouchers', 'Customer'),
				'type' => 'string',
				'compare' => array('CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%")
			),
			'GiftVoucherPurchase.gift_voucher_purchase_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => GiftVoucherPurchaseState::COMPLETE,
				'compare' => array('GiftVoucherPurchase.gift_voucher_purchase_state_id' => '%s')
			),
			'GiftVoucherPurchase.completed_date' => array(
				'label' => __d('buzz_vouchers', 'Completed Date'),
				'type' => 'date',
				'compare' => array('GiftVoucherPurchase.completed_date' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'GiftVoucherPurchase.sales_ref', $newFilters);
	}

	/**
	 * Used to populate form drop down selects
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$this->set('giftVoucherPurchaseStates', $this->GiftVoucherPurchase->GiftVoucherPurchaseState->find('list'));
		$this->set('giftVoucherDeliveryOptions', $this->GiftVoucherPurchase->GiftVoucherDeliveryOption->find('list'));
		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);

		if ((int)$this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('GiftVoucherPurchase', $id));
		}

		// We're overriding the scaffolding template as we want to customise
		// the tabs.
		$this->view = 'BuzzVouchers.admin_form';

		return;
	}

	protected function _adminIndexToolbar($id = null) {
		return [];
	}

	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);

		$actions['Resend Email'] = array(
			'url' => array('action' => 'email_confirmation', $id),
			'icon' => 'envelope'
		);

		return $actions;
	}

	/**
	 * Re-sends the confirmation email to the customer and redirects to the
	 * admin edit form.
	 *
	 * @param int $id Purchase ID
	 */
	public function admin_email_confirmation($id) {
		$GiftVoucherPurchase = $this->{$this->modelClass}->alias;

		// Raise an event when purchase is complete. This will be used for
		// triggering the sending of confirmation emails (etc.).
		$Event = new CakeEvent('Model.GiftVoucherPurchase.completed', $this, array(
			'id' => $id
		));
		$this->getEventManager()->dispatch($Event);

		$this->Session->setFlash(
			array(
				'title' => __d('buzz_vouchers', '%s confirmation sent', [InflectorExt::humanize($this->$GiftVoucherPurchase->displayName)]),
				'description' => __d('buzz_vouchers', 'The confirmation email has been resent to the customer!')
			),
			'flash_success'
		);

		return $this->redirect(['action' => 'edit', $id]);
	}

}
