<?php

App::uses('ModelBehavior', 'Behavior');
App::uses('TranslateUtil', 'BuzzTranslate.Lib');

class TranslatableBehavior extends ModelBehavior {

/**
 * Setup behavior
 *
 * @param Model $Model
 * @param array $settings
 * @return void
 */
	public function setup(Model $Model, $settings = array()) {
		parent::setup($Model, $settings);

		$this->settings[$Model->alias] = array_merge(
			$settings
		);

		return;
	}

/**
 * After save
 *
 * @param Model $Model
 * @param array $data Translations to save
 * @return void
 */
	public function afterSave(Model $Model, $created, $options = []) {
		if (parent::afterSave($Model, $created, $options) === true && !empty($Model->data[$Model->alias]['__i18n'])) {
			$this->saveTranslations($Model, $Model->data[$Model->alias]['__i18n']);
		}
		return;
	}

/**
 * Save translations.
 *
 * @param Model $Model
 * @param array $data Translations to save
 * @return bool
 */
	public function saveTranslations(Model $Model, $data) {
		foreach ($data as $lang => &$row) {
			$row['content'] = json_encode(Hash::filter($row['content']));
			$row['locale'] = $lang;
			$row['model'] = $Model->name;
			$row['foreign_id'] = $Model->id;
		}
		return ClassRegistry::init('BuzzTranslate.Translation')->saveMany($data) !== false;
	}

/**
 * After find attach translations.
 *
 * @param Model $Model
 * @param array $data
 * @param bool $primary
 * @return array
 */
	public function afterFind(Model $Model, $data, $primary = false) {
		$Translation = ClassRegistry::init('BuzzTranslate.Translation');

		if ($Translation::getLocales() > 1) {

			$aliases = [];
			if (!empty($data[0])) {
				// Get all the associated models
				$aliases = array_keys($data[0]);
				if (($key = array_search($Model->alias, $aliases)) !== false) {
					unset($aliases[$key]);
				}

				// We need to get out all of the primary keys so that we can
				// fetch the relevant translations.
				$hasMany = array_keys($Model->hasMany);
				$hasOne = array_keys($Model->hasOne);
				$belongsTo = array_keys($Model->belongsTo);
				$hasAndBelongsToMany = array_keys($Model->hasAndBelongsToMany);
				foreach ($aliases as $alias) {
					// We use the model's actual name rather than the alias or className here to
					// ensure we can easily extend models.
					if (in_array($alias, $hasMany)) {
						$model = $Model->$alias->name;
						$keys[$model] = Hash::extract($data, '{n}.' . $alias . '.{n}.id');
					} elseif (in_array($alias, $hasOne)) {
						$model = $Model->$alias->name;
						$keys[$model] = Hash::extract($data, '{n}.' . $alias . '.id');
					} elseif (in_array($alias, $belongsTo)) {
						$model = $Model->$alias->name;
						$keys[$model] = Hash::extract($data, '{n}.' . $alias . '.id');
					} elseif (in_array($alias, $hasAndBelongsToMany)) {
						$model = $Model->$alias->name;
						$keys[$model] = Hash::extract($data, '{n}.' . $alias . '.{n}.id');
					}
				}
				if (empty($keys[$Model->alias])) {
					$keys[$Model->alias] = Hash::extract($data, '{n}.' . $Model->alias . '.id');
				} else {
					$keys[$Model->alias] += Hash::extract($data, '{n}.' . $Model->alias . '.id');
				}

				// Get out all the relevant translations.
				$params = [];
				foreach ($keys as $model => $ids) {
					$params['conditions']['or'][] = array(
						'Translation.model' => $model,
						'Translation.foreign_id' => $ids
					);
				}
				$translations = $Translation->find('all', $params);

				$modelTranslations = Hash::combine(
					$translations,
					'{n}.Translation.id',
					'{n}.Translation',
					'{n}.Translation.model'
				);

				$modelTranslations = $this->stripPluginFromKeys($modelTranslations);

				foreach ($data as &$value) {
					foreach ($value as $alias => &$val) {
						if (!empty($modelTranslations[$alias])) {
							$val = $this->_attachTranslations(
								$val,
								Hash::combine(
									$modelTranslations[$alias],
									'{n}.locale',
									'{n}',
									'{n}.foreign_id'
								)
							);
						}
					}
				}

			}

		}

		return $data;
	}

/**
 * Removes the plugin syntax from the array keys.
 *
 * @param array $data
 * @return array
 */
	public function stripPluginFromKeys(array $data) {
		$result = [];
		foreach ($data as $key => $value) {
			list($plugin, $key) = pluginSplit($key);
			$result[$key] = $value;
		}
		return $result;
	}

/**
 * Attach translations to the passed data.
 *
 * @param array $data
 * @param array $translations
 * @param bool $recursive
 * @return array
 */
	protected function _attachTranslations(array $data, array $translations, $recursive = true) {
		if ((bool)count(array_filter(array_keys($data), 'is_string')) === true) {
			if (!empty($translations[$data['id']])) {
				$data['__i18n'] = $translations[$data['id']];
			}
		} else {
			foreach ($data as &$value) {
				$value = $this->_attachTranslations($value, $translations, false);
			}
		}

		return $data;
	}

/**
 * Returns a translated list. This is a replacement for $this->find('list').
 *
 * @param Model $Model
 * @param array $params
 * @param string $locale
 * @return array
 */
	public function translatedList(Model $Model, array $params = [], $locale = null) {
		// Specify the default fields required for the list (can be overridden
		// using the passed $params parameter).
		$defaults = array(
			'fields' => array(
				$Model->alias . '.' . $Model->primaryKey,
				$Model->alias . '.' . $Model->displayField
			)
		);
		$params = Hash::merge($params, $defaults);

		// Get the results including translations.
		$results = $Model->find('all', $params);

		// Build our return data replacing array values with translations.
		$data = [];
		foreach ($results as $result) {
			$data[$result[$Model->alias][$Model->primaryKey]] = TranslateUtil::translate(
				$result,
				$params['fields'][1],
				$locale
			);
		}

		return $data;
	}

}
