<?php

App::uses('BuzzSubscribeAppModel', 'BuzzSubscribe.Model');

class Subscriber extends BuzzSubscribeAppModel {

	public $displayField = 'email';

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 100),
				'message' => 'No more than 100 characters long'
			)
		),
		'first_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'last_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			)
		)
	);

/**
 * Constructor
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);
		$this->virtualFields['full_name'] = 'CONCAT(`' . $this->alias . '`.`first_name`, \' \', IFNULL(`' . $this->alias . '`.`last_name`, \'\'))';
	}

/**
 * Before save logic
 *
 * @param array $options Options passed from Model::save().
 * @return bool True if the operation should continue, false if it should abort
 */
	public function beforeSave($options = []) {
		if (!empty($this->data[$this->alias]['name'])) {
			if (preg_match('/^(\w+)\s(.*)/', $this->data[$this->alias]['name'], $matches) === 1) {
				$this->data[$this->alias]['first_name'] = $matches[1];
				$this->data[$this->alias]['last_name'] = $matches[2];
			} else {
				$this->data[$this->alias]['first_name'] = $this->data[$this->alias]['name'];
			}
		}
		return true;
	}

/**
 * After save logic
 *
 * @param bool $created True if this save created a new record
 * @param array $options Options passed from Model::save().
 * @return void
 */
	public function afterSave($created, $options = []) {
		parent::afterSave($created, $options);
		if ($created === true) {
			$Event = new CakeEvent('Model.Subscriber.created', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}
	}

/**
 * Helper method for adding a subscriber
 *
 * @param string $email     Email address
 * @param string $firstName First name
 * @param string $lastName  Last name
 * @param string $source    Source name of subscriber, e.g. 'Booking'
 * @return bool
 */
	public function addSubscriber($email, $firstName, $lastName, $source = null) {
		return (bool)$this->save([
			'email' => $email,
			'first_name' => $firstName,
			'last_name' => $lastName,
			'source' => $source,
		]);
	}

}
