<?php
App::uses('HttpSocket', 'Network/Http');
App::uses('DataSource', 'Model/Datasource');
App::Uses('Site', 'BuzzSites.Model');

class BuzzSource extends DataSource {

	protected $_schema = [];

	public function __construct($config) {
		parent::__construct($config);
		$this->Http = new HttpSocket();
	}

	public function describe($model) {
		return $this->_schema;
	}

	public function read(Model $model, $queryData = array(), $recursive = null) {
		$json = $this->Http->get(
			$this->config['host'] . '/' . $queryData['method'],
			$queryData['conditions']
		);

		$response = json_decode($json, true);

		// Log the API call.
		$this->_logApiCall(
			$this->config['host'],
			$queryData['method'],
			!empty($queryData['conditions']) ? $queryData['conditions'] : [],
			$json,
			isset($queryData['debug']) ? $queryData['debug'] : 1,
			!empty($queryData['apiLog']) ? $queryData['apiLog'] : null
		);

		if (isset($response['ExceptionDetail'])) {
			// Log and report the API error.
			$this->_logApiError(
				$this->config['host'] . '/' . $queryData['method'],
				!empty($queryData['conditions']) ? $queryData['conditions'] : [],
				$json
			);
		}

		if (is_null($response)) {
			$error = json_last_error();
			throw new CakeException($error);
		}

		return [$model->alias => $response];
	}

	/**
	 * Log API failure when an exception is thrown.
	 *
	 * @param string $url
	 * @param array $params
	 * @param string $data JSON response from API
	 * @return void
	 */
	protected function _logApiError($url, array $params, $data) {
		$url .= '?' . http_build_query($params);

		// Log in the error logs.
		$this->log($url, 'error');
		$this->log($params, 'error');
		$this->log($data, 'error');

		// See if we need to queue an API failure email.
		$email = Configure::read('BuzzSource.api_error_report_email');
		if (!empty($email)) {
			$subject = Configure::read('BuzzSource.api_error_report_subject');
			$message = '<p>' . date('d/m/Y H:i:s') . ' - ' . $url . "</p>";
			$message .= '<p>Site: ' . $_SERVER['HTTP_HOST'] . '</p>';
			$message .= '<p>IP Address: ' . $_SERVER['REMOTE_ADDR'] . '</p>';
			$message .= '<pre>' . print_r($data, true) . '</pre>';

			// Send email.
			ClassRegistry::init('BuzzEmails.Email')->queueEmail(
				$subject ?: 'API Failure',
				$message,
				$email,
				(array)Configure::read('SiteSetting.admin_email')
			);
		}

		return;
	}

	/**
	 * Logs API calls
	 *
	 * @param string $url
	 * @param string $method
	 * @param array $params
	 * @param string $data
	 * @param int $debugLevel Level for debugging use 1 to always log and 2 to only log when full debug enabled
	 * @param array $log Array contain the model and foreign ID for logging to the API log
	 * @return void
	 */
	protected function _logApiCall($url, $method, array $params, $data, $debugLevel = 1, $log = null) {
		$url .= '/' . $method;
		// Check if we need to write to the log files.
		if ($debugLevel > 0) {
			$debug = Configure::read('api.debug');
			if (!empty($debug) && $debug >= $debugLevel) {
				$this->log($url, 'api');
				$this->log($params, 'api');
				$this->log($data, 'api');
			}
		}
		// Check if we need to record the API call in the API Log.
		if (!empty($log['model']) && !empty($log['foreign_id'])) {
			ClassRegistry::init('BuzzSource.ApiLog')->add(
				$method,
				$url,
				$params,
				$data,
				$log['model'],
				$log['foreign_id']
			);
		}
		return;
	}

}
