<?php
App::uses('BuzzSitesAppModel', 'BuzzSites.Model');

class Site extends BuzzSitesAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzTranslate.Translatable'
	);

	public $imageSlots = [
		'logo' => [
			'slots' => 1,
			'fields' => [
				'alt'
			]
		]
	];

	/**
	 * Overridden to reset the default option on save if we're changing it.
	 *
	 * @param array $options
	 *
	 * @return bool
	 */
	public function beforeSave($options = []) {
		if (! empty($this->data['Site']) && $this->data['Site']['is_default'] == '1') {
			// Reset default
			$this->updateAll(['is_default' => 0]);
		}

		return parent::beforeSave($options);
	}

	public function getAll() {
		return $this->find('all', [
			'conditions' => ['is_active' => 1],
			'contain' => ['LogoImage']
		]);
	}

	public function getList($sites = null) {
		$sites = ! empty($sites) ? $sites : $this->getAll();

		$siteList = [];

		foreach ($sites as $site) {
			$siteList[$site['Site']['id']] = $site['Site']['name'];
		}
		return $siteList;
	}

	public function getTranslatedList() {
		return $this->translatedList([
			'conditions' => [
				'Site.is_active' => true
			],
			'order' => [
				'Site.name' => 'ASC'
			]
		]);
	}

	public function getListByApiId() {
		$this->primaryKey = 'api_site_id';
		return $this->translatedList([
			'conditions' => [
				'Site.is_active' => true
			],
			'order' => [
				'Site.name' => 'ASC'
			]
		], Configure::read('Config.language'));
	}

	public function getDefaultSite($sites = null) {
		if (empty($sites)) {

			return $this->find('first', [
				'conditions' => ['is_default' => true]
			]);
		}

		$default = null;

		foreach ($sites as $site) {
			if ($site['Site']['is_default']) {
				$default = $site;
				break;
			}
		}

		if (! empty($default)) {

			if (! CakeSession::check('activeSite') || empty(CakeSession::read('activeSite'))) {
				CakeSession::write('activeSite', $default);
				Configure::write('api.site_id', $default['Site']['api_site_id']);
			}

			return $default;
		}

		throw new Execption('Unable to load default site');
	}

	public function setActiveSite($site, $isApiId = false) {
		if (is_numeric($site)) {
			if ($isApiId) {
				$site = $this->find('first', ['conditions' => ['api_site_id' => $site]]);
			} else {
				$site = $this->findById($site);
			}
		}
		CakeSession::write('activeSite', $site);
		Configure::write('api.site_id', $site['Site']['api_site_id']);
	}

	public function getActiveSite() {
		// Work out the active site
		if (CakeSession::check('activeSite') && ! empty(CakeSession::read('activeSite'))) {
			$site = CakeSession::read('activeSite');
		}
		if (! isset($site) && ! isset($site['Site'])) {
			// No active site set, set it to default and return that.
			$site = $this->getDefaultSite();
			$this->setActiveSite($site);
		}

		// Now we need to work out our environment so we return the correct api details.

		if (Configure::read('app.enviroment') != 'PRODUCTION') {
			// Staging
			$site['Site']['api'] = $site['Site']['staging_api_url'];
			$site['Site']['diaryApi'] = $site['Site']['staging_diary_api_url'];
			$site['Site']['imageApi'] = $site['Site']['staging_image_api_url'];
			$site['Site']['gateway'] = $site['Site']['staging_gateway_credentials'];

		} else {
			// Production
			$site['Site']['api'] = $site['Site']['production_api_url'];
			$site['Site']['diaryApi'] = $site['Site']['production_diary_api_url'];
			$site['Site']['imageApi'] = $site['Site']['production_image_api_url'];
			$site['Site']['gateway'] = $site['Site']['production_gateway_credentials'];
		}

		return $site;
	}

	public function getActiveSiteConfig($type = null) {
		$site = $this->getActiveSite();

		Configure::write('api.site_id', $site['Site']['api_site_id']);

		return [
			'datasource' => 'BuzzSource.BuzzSource',
			'database' => false,
			'host' => ! empty($type) ? $site['Site'][$type] : $site['Site']['api']
		];
	}

}
