<?php

App::uses('BuzzPurchaseAppModel', 'BuzzPurchase.Model');

class Payment extends BuzzPurchaseAppModel {

/**
 * Payment does not use a table, it is used for validating credit card details.
 *
 * @var string|bool
 */
	public $useTable = false;

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'card_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'card_number' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'cc_number' => array(
				'rule' => array('cc', 'fast', true, null),
				'message' => 'The card number you provided was invalid'
			),
			'numeric' => array(
				'rule' => 'naturalNumber',
				'message' => 'Numbers only'
			)
		),
		'expiry_date' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Please provide a valid expiry date'
			)
		),
		// We do not define validation error messages here as these rules will
		// use the error message for 'expiry_date'.
		'expiry_month' => array(
			'required' => array(
				'rule' => 'notEmpty'
			),
			'month1' => array(
				'rule' => array('comparison', '<=', 12)
			),
			'month2' => array(
				'rule' => array('comparison', '>', 0)
			)
		),
		// We do not define validation error messages here as these rules will
		// use the error message for 'expiry_date'.
		'expiry_year' => array(
			'required' => array(
				'rule' => 'notEmpty'
			),
			'year' => array(
				'rule' => "/^[0-9]{2}$/"
			),
			'future' => array(
				'rule' => array('validateYear', 'future')
			)
		),
		'start_date' => array(
			'required' => array(
				'message' => 'Please provide a valid start date'
			)
		),
		// We do not define validation error messages here as these rules will
		// use the error message for 'start_date'.
		'start_month' => array(
			'month1' => array(
				'rule' => array('comparison', '<=', 12),
				'allowEmpty' => true
			),
			'month2' => array(
				'rule' => array('comparison', '>', 0),
				'allowEmpty' => true
			)
		),
		// We do not define validation error messages here as these rules will
		// use the error message for 'start_date'.
		'start_year' => array(
			'year' => array(
				'rule' => "/^[0-9]{2}$/",
				'allowEmpty' => true
			)
		),
		'issue_number' => array(
			'valid' => array(
				'rule' => '|^\\d{0,3}$|',
				'message' => 'Please provide a valid issue number',
				'allowEmpty' => true
			)
		),
		'security_number' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Please provide the card security number'
			),
			'between' => array(
				'rule' => array('between', 3, 4),
				'message' => 'Please provide a valid security number'
			)
		)
	);

/**
 * Card types configured for the site.
 *
 * @var array
 */
	public $cardTypes = null;

/**
 * Validate the year.
 *
 * @param array $check Data to validate
 * @param string $dir Either 'past' or 'future'
 * @return bool
 */
	public function validateYear($check, $dir = 'past') {
		$year = array_shift($check);
		if ($dir === 'future') {
			return $year >= date('y');
		} else {
			return $year <= date('y');
		}
	}

/**
 * Before validation logic.
 *
 * @param array $options
 * @return bool
 */
	public function beforeValidate($options = []) {
		// Update the credit card number rule to use the cards configured for
		// the current site.
		$cardTypes = $this->getCardTypes();
		if (!empty($cardTypes)) {
			$this->validator()->getField('card_number')->getRule('cc_number')->rule = [
				'cc',
				array_keys($cardTypes),
				true,
				null
			];
		}
		return parent::beforeValidate($options);
	}

/**
 * After validation logic. We check if we need to flag the expiry or start date
 * fields as invalid here.
 *
 * @return void
 */
	public function afterValidate() {
		// Check if we need to flag the expiry date fields as invalid.
		if (
			!empty($this->validationErrors['expiry_month'])
			|| !empty($this->validationErrors['expiry_year'])
		) {
			$this->invalidate('expiry_date', $this->validate['expiry_date']['required']['message']);
		}

		// Check if we need to flag the start date fields as invalid.
		if (
			!empty($this->validationErrors['start_month'])
			|| !empty($this->validationErrors['start_year'])
		) {
			$this->invalidate('start_date', $this->validate['start_date']['required']['message']);
		}

		return;
	}

/**
 * Returns the available card types.
 *
 * @return array
 */
	public function getCardTypes() {
		if (empty($this->cardTypes)) {
			$this->cardTypes = Configure::read('BuzzPurchase.card_types');
		}
		return $this->cardTypes;
	}

}
