<?php

App::uses('ModelBehavior', 'Behavior');

class PayableBehavior extends ModelBehavior {

/**
 * Basket ID
 *
 * @var int
 */
	protected $_basketId = null;

/**
 * Setup behavior
 *
 * @param Model $Model
 * @param array $settings
 * @return void
 */
	public function setup(Model $Model, $settings = array()) {
		parent::setup($Model, $settings);

		if (!isset($this->settings[$Model->alias])) {
			$this->settings[$Model->alias] = array(
				'basket_session_key' => $Model->alias . '.id'
			);
		}

		$this->settings[$Model->alias] = array_merge(
			$this->settings[$Model->alias],
			$settings
		);

		return;
	}

/**
 * Set the basket ID from the session
 *
 * @param Model $Model
 * @param int $id
 * @return void
 */
	public function setBasketId(Model $Model, $id) {
		CakeSession::write($this->settings[$Model->alias]['basket_session_key'], $id);
		$this->_basketId = $id;
		return;
	}

/**
 * Get the current basket ID from the session
 *
 * @param Model $Model
 * @return int
 */
	public function getBasketId(Model $Model) {
		if (empty($this->_basketId)) {
			$this->_basketId = CakeSession::read($this->settings[$Model->alias]['basket_session_key']);
		}
		return $this->_basketId;
	}

/**
 * Clear the basket ID from the session
 *
 * @param Model $Model
 * @return bool
 */
	public function clearBasketId(Model $Model) {
		$this->_basketId = null;
		CakeSession::delete($this->settings[$Model->alias]['basket_session_key']);
		return true;
	}

/**
 * Returns a hashed basket ID
 *
 * @param Model $Model
 * @param int $id (Optional) basket ID
 * @return string
 */
	public function hashBasketId(Model $Model, $id = null) {
		$id = $id ?: $this->getBasketId($Model);
		return md5($id);
	}

/**
 * Returns payment details for passing to the API
 *
 * @param Model $Model
 * @param int $id Model ID
 * @param string $model
 * @return array
 */
	public function getPaymentDetails(Model $Model, $id, $model = null) {
		$data = [
			'vendorTxCode' => '',
			'vpstxId' => '',
			'status' => 'OK',
			'txauthno' => ''
		];
		if ($model === null) {
			$model = (!empty($Model->plugin) ? $Model->plugin . '.' : '') . $Model->alias;
		}
		switch (Configure::read('Transactions.gateway')) {
			case 'epay':
				$this->_generateEpayPaymentDetails($id, $model, $data);
				break;
			case 'redsys':
				$this->_generateRedSysPaymentDetails($id, $model, $data);
				break;
			case 'sagepay_direct':
				$this->_generateSagePayDirectPaymentDetails($id, $model, $data);
				break;
		}

		return $data;
	}

/**
 * Get the ePay payment details.
 *
 * @param int $id
 * @param string $model
 * @param array &$data
 */
	protected function _generateEpayPaymentDetails($id, $model, array &$data) {
		$result = ClassRegistry::init('Transactions.Transaction')->find(
			'first',
			[
				'fields' => ['Epay.*'],
				'joins' => [
					[
						'table' => 'transactions_epay',
						'alias' => 'Epay',
						'type' => 'INNER',
						'conditions' => 'Epay.transaction_id = Transaction.id'
					]
				],
				'conditions' => [
					'Transaction.model_id' => $id,
					'Transaction.model' => $model
				],
				'order' => ['Epay.created' => 'DESC']
			]
		);

		if (!empty($result)) {
			$data = [
				'vendorTxCode' => $result['Epay']['txnid'],
				'vpstxId' => '',
				'status' => 'OK',
				'txauthno' => ''
			];
		}
	}

/**
 * Get the Redsys payment details.
 *
 * @param int $id
 * @param string $model
 * @param array &$data
 */
	protected function _generateRedSysPaymentDetails($id, $model, array &$data) {
		$result = ClassRegistry::init('Transactions.Transaction')->find(
			'first',
			[
				'fields' => ['Redsys.*'],
				'joins' => [
					[
						'table' => 'transactions_redsys',
						'alias' => 'Redsys',
						'type' => 'INNER',
						'conditions' => 'Redsys.transaction_id = Transaction.id'
					]
				],
				'conditions' => [
					'Transaction.model_id' => $id,
					'Transaction.model' => $model
				],
				'order' => ['Redsys.created' => 'DESC']
			]
		);

		if (!empty($result)) {
			$data = [
				'vendorTxCode' => $result['Redsys']['order'],
				'vpstxId' => $result['Redsys']['signature'],
				'status' => 'OK',
				'txauthno' => $result['Redsys']['authorisation_code']
			];
		}
	}

/**
 * Get the Redsys payment details.
 *
 * @param int $id
 * @param string $model
 * @param array &$data
 */
	protected function _generateSagePayDirectPaymentDetails($id, $model, array &$data) {
		$result = ClassRegistry::init('Transactions.Transaction')->find(
			'first',
			[
				'conditions' => [
					'Transaction.model_id' => $id,
					'Transaction.model' => $model
				]
			]
		);

		if (!empty($result['Transaction']['payment_token'])) {
			$transaction = json_decode($result['Transaction']['payment_token']);
			$data = [
				'vendorTxCode' => !empty($transaction->VendorTxCode) ? $transaction->VendorTxCode : '',
				'vpstxId' => !empty($transaction->VPSTxId) ? $transaction->VPSTxId : '',
				'status' => 'OK',
				'txauthno' => !empty($transaction->TxAuthNo) ? $transaction->TxAuthNo : ''
			];
		}
	}

}
