<?php

class ActivityGroupsController extends BuzzPhotosAppController {

	public function __construct($request = null, $response = null) {
		$this->components['BuzzTranslate.Translation'] = array(
			'fields' => array(
				'name'
			),
			'meta' => true
		);

		parent::__construct($request, $response);
	}

	public function beforeFilter() {
		parent::beforeFilter();
		$this->Auth->allow(array(
			'index',
			'location',
			'view',
			'ajax_get_times',
			'album',
			'share'
		));
	}

	protected function _adminPopulateLookups() {
		parent::_adminPopulateLookups();
		$this->set('themes', array(1, 2, 3, 4));
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();
		$fields['ActivityGroup.theme_id']['empty'] = false;

		return $fields;
	}

	public function admin_edit($id = null) {
		if ($this->request->is('post') || $this->request->is('put')) {
			// Delete the existing activity group sites
			$this->loadModel('BuzzPhotos.ActivityGroupSite');
			$this->ActivityGroupSite->deleteAll(['activity_group_id' => $id]);
		}

		parent::admin_edit($id);

		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->getAll();
		$this->set(compact('sites'));

		$this->Translation->adminTranslationFormFields();

		$this->view = 'admin_edit';
	}

	public function index() {
		if (Configure::read('BuzzPhotos.photos_page_id') > 0) {
			$this->assignPage(Configure::read('BuzzPhotos.photos_page_id'));
		}

		$activities = $this->{$this->modelClass}->find('all', array(
			'conditions' => array(
				'is_active' => 1
			),
			'contain' => array(
				'Image'
			)
		));
		$this->set('activities', $activities);
	}

	public function location($activityId) {
		if (
			($this->request->is('post') || $this->request->is('put')) &&
			! empty($this->request->data['siteId'])
		) {
			$siteId = $this->request->data['siteId'];

			// Update active site
			$this->loadModel('BuzzSites.Site');
			$this->Site->setActiveSite($siteId);

			return $this->redirect([
				'action' => 'view',
				$siteId,
				$activityId
			]);
		}

		$this->loadModel('BuzzPhotos.ActivityGroupSite');
		$activitySiteIds = $this->ActivityGroupSite->find('all', [
			'fields' => ['site_id'],
			'conditions' => [
				'not' => [
					'api_key' => '',
					'stock_code' => ''
				]
			]
		]);

		$activitySiteIds = Hash::extract($activitySiteIds, '{n}.ActivityGroupSite.site_id');

		if (empty($activitySiteIds)) {
			return $this->redirect($this->referer());
		}

		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->find('all', [
			'conditions' => ['id' => $activitySiteIds],
			'contain' => ['LogoImage']
		]);
		$activeSite = $this->Site->getActiveSite();
		$activeSiteId = $activeSite['Site']['id'];

		if (count($sites) == 1) {
			return $this->redirect([
				'action' => 'view',
				$sites[0]['Site']['id'],
				$activityId
			]);
		}

		$this->set(compact('sites', 'activityId', 'activeSiteId'));
	}

	public function view($siteId, $activityId) {
		if ($this->request->is('post')) {
			$this->Session->write('BuzzPhotos.email', $this->request->data[$this->{$this->modelClass}->alias]['email']);

			return $this->redirect(['action' => 'album', $siteId, $activityId]);
		}

		$activity = $this->{$this->modelClass}->find('first', array(
			'conditions' => array(
				'id' => $activityId
			),
			'contain' => array(
				'FeatureImage',
				'ActivityGroupSite'
			)
		));

		if (empty($activity)) {
			throw new NotFoundException;
		}
		$this->set(compact('siteId'));
		$this->set('data', $activity);
	}

/**
 * Display an album
 * @param int $activityGroupId Activity group ID
 * @param int $id Album ID
 * @return void
 */
	public function album($siteId, $activityGroupId, $id = null) {
		$this->loadModel('BuzzPhotos.PhotoApi');
		$album = null;
		if (! empty($id) && is_numeric($activityGroupId) && $id !== 'undefined') {
			$album = $this->PhotoApi->getAlbum($id);
		} elseif ($this->Session->check('BuzzPhotos.email')) {
			$activity = $this->{$this->modelClass}->getActivityOrFail($activityGroupId, $siteId);
			$album = $this->PhotoApi->getCustomerPhotos(
				$activity['ActivityGroupSite'][0]['api_key'],
				$this->Session->read('BuzzPhotos.email')
			);

			if (empty($album)) {
				$this->Session->setFlash(__d(
					'buzz_photos',
					'We were unable to find any photos for %s',
					'<strong>' . $this->Session->read('BuzzPhotos.email') . '</strong>'
				), 'flash_fail');

				return $this->redirect(['action' => 'view', $siteId, $activityGroupId]);
			}
		}
		if (empty($album['Images'])) {
			throw new NotFoundException();
		}

		$this->loadModel('BuzzSites.Site');
		$site = $this->Site->findById($siteId);
		$this->Site->setActiveSite($siteId);

		$date = ! empty($album['AlbumDate']) ? $this->_extractDate($album['AlbumDate']) : null;

		$activities = array();

		foreach ($album['Images'] as $key => $image) {
			$activities[$key]['Image'] = array(
				'filename' => 'image.jpg',
				'filepath' => 'http://' . $image['ThumbKey'],
				'id' => $image['Id'],
				'type' => ''
			);

			$activities[$key]['PreviewImage'] = array(
				'filename' => 'image.jpg',
				'filepath' => 'http://' . $image['MediumKey'],
				'id' => $image['Id'],
				'type' => ''
			);
		}

		$this->set(compact('activities', 'date', 'activityGroupId', 'site'));
	}

	public function ajax_get_times($siteId, $activityId, $date) {
		$activity = $this->{$this->modelClass}->getActivityOrFail($activityId, $siteId);
		$this->loadModel('BuzzPhotos.PhotoApi');
		$data = $this->PhotoApi->getTimes($activity['ActivityGroupSite'][0]['api_key'], $date);

		$this->set(compact('data'));
		$this->layout = 'ajax';
	}

/**
 * Display an image shared on social media with a link back to the album.
 * @param int $activityGroupId Activity group ID
 * @param string $uuid Album UUID
 * @return void
 */
	public function share($activityGroupId, $uuid) {
		$image = null;
		if (! empty($uuid) && is_numeric($activityGroupId)) {
			$this->loadModel('BuzzPhotos.PhotoApi');
			$image = $this->PhotoApi->getImage($uuid);
		}

		if (empty($image)) {
			throw new NotFoundException;
		}

		$album = $this->PhotoApi->getAlbum($image['AlbumId']);

		// Extract and format the date
		$date = $this->_extractDate($album['AlbumDate']);

		$this->set(compact('image', 'album', 'date', 'activityGroupId'));
	}

	private function _extractDate($date) {
		$date = ltrim($date, '/Date(');
		$date = substr($date, 0, strpos($date, "+"));
		$date = $date / 1000;
		$date = date('jS F Y', $date);

		return $date;
	}

	protected function _adminIndexActions() {
		$actions = parent::_adminIndexActions();
		unset($actions[2]); // unset view

		$actions[] = 'Admin View';
		return $actions;
	}

	public function admin_view($id) {
		return $this->redirect([
			'admin' => false,
			'action' => 'location',
			$id
		]);
	}

}
