<?php
namespace Evoluted\BuzzPhotos\DiscountModifiers;

use Evoluted\PriceModifier\Interfaces\DiscountModifierInterface;
use Evoluted\PriceModifier\Interfaces\BasketInterface;

/**
 * A percentage discount handler for the PriceModifer package. (Overridden for
 * twinwoods to handle quantities)
 *
 * @package 	PriceModifier
 * @author 		Rick Mills <rick@evoluted.net>
 * @author		Evoluted New Media <developers@evoluted.net>
 *
 */
class PercentageModifier implements DiscountModifierInterface
{
	/**
     * @var Evoluted\PriceModifier\Interfaces\BasketInterface
     */
	protected $basket;

	/**
	 * @var Mixed A unique identifier for the discount. This can be a string or a number.
	 */
	protected $id;

	/**
	 * @var Array This contains any settings/data needed for this handler.
	 */
	protected $params = [
		'percent' => 0,
		'applyToItems' => false // set to true to individually apply the discount to items instead of the basket total
	];

	/**
	 * Construct the discount handler
	 *
	 * @param mixed $id A unique identifier for this discount
	 * @param array $discountData Any data needed for the discount
	 * @param BasketInterface $basket The current basket to modify
	 */
	public function __construct($id, $discountData, BasketInterface $basket)
	{
		$this->id = $id;
		$this->params = array_merge($this->params, $discountData);
		$this->basket = $basket;
	}

	/**
	 * Applies the discount to the loaded basket.
	 *
	 * @return BasketInterface Returns the modified basket
	 */
	public function applyDiscount()
	{
		if ($this->params['applyToItems']) {

			foreach ($this->basket->items() as $key => $basketItem) {

				if (empty($this->basket->validDiscounts[$this->id]) || in_array($basketItem->reference, $this->basket->validDiscounts[$this->id])) {
					$totalToApplyTo = $basketItem->unit_price;

					$discount = $this->__getDiscount($totalToApplyTo);

					$this->basket->discount += $discount;

					$this->basket->item($key)->discount = $discount;
					$this->basket->item($key)->discount_type = 'P';
				}
			}
		} else {
			if ($this->params['amount'] == '100') {
				$this->basket->discount = $this->basket->getBasketTotal();
			} else {
				$this->basket->discount += $this->__getDiscount($this->basket->getBasketTotal());
			}
		}

		return $this->basket;
	}

	/**
	 * Works out the percentage discount from the stored params.
	 *
	 * @param  double $amount the amount to apply the discount to
	 *
	 * @return double the discount amount
	 */
	private function __getDiscount($amount)
	{
		if ($this->params['amount'] == '100') {
			return $amount;
		}

		return ($this->params['amount'] / 100) * $amount;
	}

}
