<?php
App::uses('BuzzPhotosAppModel', 'BuzzPhotos.Model');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('OrderState', 'BuzzPhotos.Model');

class Order extends BuzzPhotosAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

	public $hasMany = array(
		'OrderItem' => array(
			'className' => 'BuzzPhotos.OrderItem',
			'foreignKey' => 'order_id',
			'conditions' => '',
			'fields' => '',
			'order' => '',
			'dependent' => true
		)
	);

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		),
		'OrderState' => array(
			'className' => 'BuzzPhotos.OrderState'
		)
	);

	public $virtualFields = array(
		'full_name' => 'CONCAT(Order.first_name, " ", Order.last_name)'
	);

/**
 * Default order
 *
 * @var array|string
 */
	public $order = 'Order.id DESC';

	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'OrderItem';

		return parent::readForEdit($id, $query);
	}

/**
 * Clear old incomplete orders from the db
 *
 * @return void
 */
	public function clearOldOrders() {
		// Only delete if they have been left unpaid and untouched for over 2 weeks
		$date = new DateTime();
		$date->sub(new DateInterval('P14D'));
		$date = $date->format('Y-m-d H:i:s');

		$this->deleteAll(array(
			'Order.is_paid' => 0,
			'Order.modified < ' => $date
		));
	}

/**
 * Add the customer billing details to the order
 *
 * @param int $id Order ID
 * @param array $customer
 * @return bool
 */
	public function addBillingDetails($id, array $customer) {
		$data = array(
			'Order' => array(
				'id' => $id
			),
			'CustomerAddress' => $customer
		);
		return $this->saveAssociated($data) !== false;
	}

/**
 * Complete the purchase
 *
 * @param bool $paid True if payment successfully made
 * @return bool
 */
	public function completePurchase($id, $paid) {
		$order = $this->find('first', [
			'contain' => [
				'OrderItem.ActivityGroup',
				'CustomerAddress'
			],
			'conditions' => [
				'Order.id' => $id
			]
		]);

		$response = true;

		if ((int)$order['Order']['order_state_id'] === OrderState::UNPAID) {
			if ($paid === true) {
				$result = $this->_completeSale($order);
				$order['Order']['is_paid'] = true;
				$data = [
					'id' => $id,
					'is_paid' => true,
					'order_state_id' => $result === true ? OrderState::COMPLETE : OrderState::API_FAILED
				];

				$response = $this->save($data);
			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'order_state_id' => OrderState::PAYMENT_FAILED
				);
				$response = $this->save($data);
			}

			// Clear the basket session.
			$this->clearOrderId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('BuzzPhotos.Model.Order.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}

		return $response;
	}

/**
 * Complete a sale by adding items via the API and generating a sales ref.
 * @param  array  $order Order data
 * @return bool
 */
	protected function _completeSale(array $order) {
		$BookingApi = ClassRegistry::init('BuzzBookings.BookingApi');
		if (empty($order['OrderItem'])) {
			return false;
		}
		$salesRef = null;
		foreach ($order['OrderItem'] as &$item) {
			$result = $BookingApi->addItemToSalesOrder(
				'OrderItem',
				$item['id'],
				Configure::read('api.site_id'),
				$salesRef,
				$item['ActivityGroup']['stock_code'],
				1,
				$item['unit_price']
			);

			if (! empty($result['sales_ref'])) {
				$item['sales_ref'] = $result['sales_ref'];
				$item['api_ref'] = $result['api_ref'];
				// Store teh sales ref so that we use it on the next iteration.
				$salesRef = $item['sales_ref'];
			}
		}
		// Update the order items in the database.
		if (! empty($order['OrderItem'])) {
			$this->OrderItem->saveMany($order['OrderItem']);
		}
		return $BookingApi->completeSale(
			'Order',
			$order['Order']['id'],
			$salesRef,
			$order['Order']['first_name'],
			$order['Order']['last_name'],
			$order['Order']['email'],
			$order['Order']['phone'],
			CustomerAddress::generateAddressArray($order['CustomerAddress']),
			$order['Order']['total_price'],
			$this->getPaymentDetails($order['Order']['id'], 'BuzzPhotos.Order')
		);
	}

/**
 * Clear the sessions
 * @return void
 */
	public function clearOrderId() {
		CakeSession::delete('photoOrderId');
		CakeSession::delete('photoOrderUuid');
	}
}
