<?php
App::uses('BuzzPhotosAppController', 'BuzzPhotos.Controller');
App::uses('String', 'Utility');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('Discount', 'BuzzPhotos.Lib');
App::uses('OrderState', 'BuzzPhotos.Model');

class OrdersController extends BuzzPhotosAppController {

/**
 * List of admin actions the controller supports
 *
 * @var array
 */
	public $adminActions = array(
		'admin_index',
		'admin_view',
		'admin_email_confirmation'
	);

/**
 * Components
 *
 * @var array
 */
	public $components = [
		'BuzzPurchase.FbPixel',
		'BuzzPurchase.GaEcommerce',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

	/**
	 * Paginator defaults for admin.
	 *
	 * @var array
	 */
	public $adminPaginate = array(
		'limit' => 25,
		'conditions' => array(
			'Order.is_paid' => '1'
		),
		'sort' => 'Order.modified',
		'direction' => 'desc'
	);

	public function beforeFilter() {
		parent::beforeFilter();
		$this->Auth->allow(array(
			'add',
			'checkout',
			'confirmation',
			'download_photo',
			'payment_callback'
		));

		$this->{$this->modelClass}->clearOldOrders();
	}

	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();
		$newColumns = array(
			'Order.full_name' => array(
				'type' => 'string',
				'label' => 'Customer Name'
			),
			'Order.email' => array(
				'type' => 'string',
				'label' => 'Customer Email',
			),
			'Order.phone' => array(
				'type' => 'string',
				'label' => 'Customer Phone'
			),
			'OrderState.name' => array(
				'type' => 'string',
				'label' => 'Status'
			),
			'Order.total_price' => array(
				'type' => 'currency',
				'label' => 'Order Total'
			)
		);

		return ArrayUtil::addAfter($columns, 'Order.id', $newColumns);
	}

	protected function _adminIndexActions() {
		return array('Admin View');
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormfields();
		unset($fields['Order.id']);

		return $fields;
	}

/**
 * Filters
 *
 * @return array
 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		$newFilters = array(
			'Order.sales_ref' => array(
				'label' => 'Sales Ref',
				'type' => 'string',
				'compare' => array('GiftVoucherPurchase.sales_ref' => "%s")
			),
			'Order.name' => array(
				'label' => 'Customer',
				'type' => 'string',
				'compare' => array('CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%")
			),
			'Order.order_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => OrderState::COMPLETE,
				'compare' => array('Order.order_state_id' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'GiftVoucherPurchase.sales_ref', $newFilters);
	}

	public function admin_view($id) {
		parent::admin_view($id);

		$this->loadModel('Transactions.Transaction');
		$transaction = $this->Transaction->find('first', array(
			'conditions' => array(
				'model' => 'BuzzPhotos.Order',
				'model_id' => $id
			),
			'contain' => 'TransactionsItem'
		));

		$this->viewVars['fields']['Order.sales_ref'] = array(
			'type' => 'string',
			'null' => 1,
			'label' => 'Sales Ref'
		);

		$this->request->data['Order']['sales_ref'] = (! empty($this->request->data['OrderItem'][0]['sales_ref']) ? $this->request->data['OrderItem'][0]['sales_ref'] : null);

		if (! empty($transaction)) {
			$this->set(compact('transaction'));
		}

		$this->view = 'admin_view';
	}

/**
 * Admin index toolbar
 * @return array Toolbar actions
 */
	protected function _adminIndexToolbar() {
		return [];
	}

/**
 * Admin form toolbar
 * @param int $id Order ID
 * @return array Toolbar actions
 */
	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);
		unset($actions['Delete']);

		$actions['Resend Email'] = array(
			'url' => array('action' => 'email_confirmation', $id),
			'icon' => 'envelope'
		);

		return $actions;
	}

/**
 * Re-sends the confirmation email to the customer and redirects to the
 * admin view.
 *
 * @param int $id Purchase ID
 */
	public function admin_email_confirmation($id) {
		$Order = $this->{$this->modelClass}->alias;

		// Raise an event when purchase is complete. This will be used for
		// triggering the sending of confirmation emails (etc.).
		$Event = new CakeEvent('BuzzPhotos.Model.Order.completed', $this, array(
			'id' => $id
		));
		$this->getEventManager()->dispatch($Event);

		$this->Session->setFlash(
			array(
				'title' => __d('buzz_photos', '%s confirmation sent', [InflectorExt::humanize($this->$Order->displayName)]),
				'description' => __d('buzz_photos', 'The confirmation email has been resent to the customer!')
			),
			'flash_success'
		);

		return $this->redirect(['action' => 'view', $id]);
	}

/**
 * Adds the photos to the customers order, and then redirects to the checkout.
 *
 * @return  Redirect
 */
	public function add() {
		if ($this->request->is('post') || $this->request->is('put')) {

			// Extract image ids to purchase
			$imageCheckboxes = $this->request->data['ActivityGroup']['image_checkbox'];
			$activityGroupId = $this->request->data['ActivityGroup']['id'];

			$imageIds = array();

			foreach ($imageCheckboxes as $val) {
				if ($val != '0') {
					$imageIds[] = $val;
				}
			}

			if (! empty($imageIds)) {

				$orderUuid = String::uuid();

				$unitPrice = Configure::read('SiteSetting.photo_price');
				$totalPrice = $unitPrice * count($imageIds);

				$orderData = array(
					array(
					'Order' => array(
							'uuid' => $orderUuid,
							'first_name' => null,
							'last_name' => null,
							'phone' => null,
							'email' => null,
							'is_paid' => 0,
							'billing_address' => null,
							'sub_total' => $totalPrice,
							'order_state_id' => OrderState::UNPAID
						),
						'OrderItem' => array()
					)
				);

				$this->loadModel('BuzzPhotos.PhotoApi');

				foreach ($imageIds as $imageId) {
					$imageData = $this->PhotoApi->getImage($imageId);
					$thumbUrl = 'http://' . $imageData['MediumKey']; // There is a thumb but its really tiny so we'll grab medium instead

					$orderData[0]['OrderItem'][] = array(
						'uuid' => $imageId,
						'thumb_url' => $thumbUrl,
						'unit_price' => $unitPrice,
						'activity_group_id' => $this->request->data['ActivityGroup']['id']
					);
				}

				if ($this->{$this->modelClass}->saveMany($orderData, array('deep' => true))) {
					$this->Session->write('photoOrderUuid', $orderUuid);

					return $this->redirect(array(
						'plugin' => 'buzz_photos',
						'controller' => 'orders',
						'action' => 'checkout'
					));
				}

				$this->Session->setFlash(__d('buzz_photos', 'There was a problem selecting your photos. Please try again.'), 'flash_fail');

				return $this->redirect(array(
					'plugin' => 'buzz_photos',
					'controller' => 'activity_groups',
					'action' => 'index'
				));
			}
		}
	}

/**
 * Runs the checkout functionality, using BuzzPurchase.
 *
 * @return View
 */
	public function checkout() {
		$Order = $this->{$this->modelClass};
		$orderUuid = $this->Session->read('photoOrderUuid');
		if (empty($orderUuid)) {
			throw new NotFoundException;
		}

		$order = $this->{$this->modelClass}->find('first', array(
			'conditions' => array(
				'uuid' => $orderUuid,
				'is_paid' => 0
			),
			'contain' => array(
				'OrderItem'
			)
		));

		if ($order['Order']['total_price'] <= 0) {
			$order['Order']['total_price'] = $order['Order']['sub_total'] - $order['Order']['discount'];
			$this->{$this->modelClass}->save($order, array('deep' => true));
		}

		if (!empty($this->request->data)) {

			if (! empty($this->request->data['Order']['discount_code'])) {
				// Apply discount

				$Discount = new Discount($this->request->data['Order']['discount_code'], $order);
				$discount = $Discount->applyDiscount();

				if (! $discount) {
					$this->Session->setFlash(
						__d('buzz_photos', 'The discount code entered was invalid or expired.'),
						'flash_fail'
					);
					return $this->redirect($this->here);
				}

				// We've now got our discount amount, apply it to the order and redirect back.
				$order['Order']['discount'] = $discount;
				$order['Order']['discount_code'] = $this->request->data['Order']['discount_code'];
				$order['Order']['total_price'] = $order['Order']['sub_total'] - $discount;
				$this->{$this->modelClass}->save($order, array('deep' => true));

				$this->Session->setFlash(
					__d('buzz_photos', 'Your discount code has been applied.'),
					'flash_success'
				);

				return $this->redirect($this->here);
			}

			$order['Order']['first_name'] = $this->request->data['CustomerAddress']['first_name'];
			$order['Order']['last_name'] = $this->request->data['CustomerAddress']['last_name'];
			$order['Order']['phone'] = $this->request->data['CustomerAddress']['telephone'];
			$order['Order']['email'] = $this->request->data['CustomerAddress']['email'];

			$result = (bool)$Order->save($order, true, ['first_name', 'last_name', 'phone', 'email']);

			$result = $result === true ? $Order->addBillingDetails($order['Order']['id'], $this->request->data['CustomerAddress']) : false;

			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();

			if ($result === true && $paymentValidates === true) {
				// Take payment.
				$this->_pay($order);
			} else {
				$this->Session->setFlash(
					__d('buzz_photos', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {
			$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
			$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();
		}

		$previousStep = array(
			'plugin' => 'buzz_photos',
			'controller' => 'activity_groups',
			'action' => 'index'
		);

		if (empty($order)) {
			return $this->redirect($previousStep);
		}

		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();

		$this->set(compact('order', 'previousStep'));
	}

/**
 * Processes the payment and triggers the completion of the payment on success.
 *
 * @return void
 */
	protected function _pay($order) {
		$Order = $this->{$this->modelClass};

		if ($order['Order']['total_price'] > 0) {

			if (Configure::read('BuzzPurchase.onsite') === true) {
				$payment = $this->request->data['Payment'];

				// Split name into first and second names for payment gateways.
				$fullName = trim($payment['card_name']);
				$splitName = explode(' ', $fullName);
				$payment['last_name'] = array_pop($splitName);
				$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
			} else {
				$payment = [];
			}

			$items = [];

			foreach ($order['OrderItem'] as $item) {
				$items[] = [
					'description' => 'Photo ID ' . $item['uuid'],
					'amount' => $item['unit_price'],
					'model' => 'OrderItem',
					'model_id' => $item['id']
				];
			}

			$this->loadModel('BuzzCustomers.Country');
			$country = $this->Country->findById($this->request->data['CustomerAddress']['country_id']);
			$this->request->data['CustomerAddress']['Country'] = $country['Country'];

			$this->Session->write('photoOrderId', $order['Order']['id']);

			$transaction = $this->Transactions->takePayment(
				// Return URLs
				array(
					'return' => Router::url(['action' => 'payment_callback'], true),
					'cancel' => Router::url(['action' => 'checkout'], true)
				),
				// Calling record
				array(
					'model' => 'BuzzPhotos.Order',
					'model_id' => $order['Order']['id']
				),
				// Amount to be paid
				$order['Order']['total_price'],
				// Items
				$items,
				// Extras
				array(
					'language' => Configure::read('Config.language'),
					'card' => $payment,
					'address' => $this->request->data['CustomerAddress'],
					'user' => array('email' => $this->request->data['CustomerAddress']['email']),
					'description' => __d(
						'buzz_photos',
						'%s Photo Purchase %d',
						[Configure::read('SiteSetting.site_title'), $order['Order']['id']]
					)
				),
				0,
				Configure::read('Transactions.gateway')
			);

			if (!empty($transaction['result'])) {
				// Payment taken successfully, complete purchase.
				return $this->_completePurchase();
			} else {
				$this->Session->setFlash(
					__d(
						'buzz_photos',
						'There was a problem processing your payment, please check your details and try again'
					),
					'flash_fail'
				);
			}

		} else {
			// Nothing to pay, complete purchase.
			return $this->_completePurchase();
		}

		return;
	}

/**
 * Confirmation page.
 *
 * @param int $id Basket ID
 * @return void
 */
	public function confirmation($uuid) {
		$Order = $this->{$this->modelClass};

		if (md5($uuid) === $this->Session->read('PhotoOrder.hash') || Configure::read('app.enviroment') == 'DEVELOPMENT') {

			$this->Session->delete('PhotoOrder.hash');
			$this->set('showOrder', true);

		}

		$order = $Order->find('first', array(
			'conditions' => array(
				'uuid' => $uuid
			),
			'contain' => array(
				'OrderItem'
			)
		));

		if (Configure::check('BuzzPhotos.confirmation_page_id') === true) {
			$this->assignPage(Configure::read('BuzzPhotos.confirmation_page_id'));
		}

		// Set up the GA ecommerce tracking variables.
		$gaTransaction = $this->GaEcommerce->transaction(
			$order['Order']['id'],
			$order['Order']['total_price']
		);

		foreach ($order['OrderItem'] as $item) {
			$this->GaEcommerce->addItem(
				$order['Order']['id'],
				'Photo Purchase',
				$item['uuid'],
				$item['unit_price'],
				1,
				'Order'
			);
			$this->FbPixel->addItem('P' . $item['id']);
		}

		$gaItems = $this->GaEcommerce->items();
		// Set up the Facebook Pixel tracking variables.
		$fbPixelEvent = $this->FbPixel->transaction(
			$order['Order']['total_price']
		);
		$this->set(compact('fbPixelEvent', 'gaTransaction', 'gaItems', 'order'));

		return;
	}

/**
 * Formats the address data for use in a single line string.
 *
 * @param  array $data Address data array
 * @return string Reconstructed address as a string
 */
	private function __formatAddress($data) {
		$this->loadModel('BuzzCustomers.Country');
		$countries = $this->Country->find('list');

		$address = array(
			$data['address_line_1'],
			$data['address_line_2'],
			$data['city'],
			$countries[$data['country_id']]
		);

		if (! empty($data['us_state_id'])) {
			$this->loadModel('BuzzCustomers.UsState');
			$usStates = $this->UsState->find('list');

			$address[] = $usStates[$data['us_state_id']];
		}

		$address[] = $data['postcode'];

		return implode(',', $address);
	}

/**
 * Generic payment gateway callback.
 *
 * @return void
 */
	public function payment_callback() {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completePurchase();
		} else {
			$this->Session->setFlash(
				__d('buzz_photos', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'checkout'
			]);
		}
	}

/**
 * Retrieves a download link from the api and redirects to that. Intened to reduce
 * the chance of people guessing urls to full images.
 *
 * @param  string $uuid uuid of the photo to look up
 * @return Redirect Redirects to the full photo url
 */
	public function download_photo($uuid) {
		$this->loadModel('BuzzPhotos.PhotoApi');

		$photo = $this->PhotoApi->getSignedImage($uuid);

		if (! empty($photo)) {
			return $this->redirect($photo);
		} else {
			$this->Session->setFlash(__d('buzz_photos', 'Your photo download link has expired.'), 'flash_fail');
			return $this->redirect('/');
		}
	}

/**
 * Complete the purchase (after payment received)
 *
 * @return void Redirects to confirmation page
 */
	protected function _completePurchase() {
		$orderId = $this->Session->read('photoOrderId');

		$Order = $this->{$this->modelClass};
		$order = $Order->findById($orderId);

		if (empty($order)) {
			throw new ForbiddenException();
		}

		$Order->completePurchase($orderId, true);

		$this->loadModel('BuzzPhotos.PhotoApi');

		// The hash *must* be set after completing the purchase as
		// completePurchase() will delete the session. The idea being that we only
		// show the order details once, on the final page after coming back from
		// the payment provider. Subsequent visits will only show the downloadable
		// photos on a simplified page.
		$this->Session->write('PhotoOrder.hash', md5($order['Order']['uuid']));

		return $this->redirect([
			'action' => 'confirmation',
			$order['Order']['uuid']
		]);
	}

/**
 * Populate lookups for address fields.
 *
 * @return void
 */
	protected function _populateLookups() {
		$Booking = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$UsStatesModel = ClassRegistry::init('BuzzCustomers.UsState');
		$usStates = $UsStatesModel->find('list');

		$this->set(compact('countries', 'usStates'));

		return;
	}

/**
 * Used to populate form drop down selects
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$this->set('orderStates', $this->Order->OrderState->find('list'));
		return;
	}

/**
 * Admin index paginate
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'CustomerAddress',
				'OrderState'
			)
		);

		return $paginate;
	}

}
