<?php

App::uses('AppHelper', 'View/Helper');

App::uses('Menu', 'Navigation.Model');

/**
 * Navigation Helper
 *
 * To create a formatted menu pass the return from Menu::getMenu($id):-
 *
 * $this->Navigation->menu($menu, $attr = array());
 *
 * You can check the active state of an individual menu item:-
 *
 * $this->Navigation->checkActive($item);
 *
 * @author  Andy Carter
 * @package Navigation
 */
class NavigationHelper extends AppHelper {

/**
 * Helper dependencies
 *
 * @var array
 * @access public
 */
	public $helpers = array('Html', 'Image', 'Session', 'BuzzTranslate.I18n');

	public $settings = array(
		'tag' => 'li',
		'wrapTag' => 'ul',
		'parentWrapTag' => null,
		'childWrapTag' => 'div',
		'parentWrapClass' => null,
		'childWrapClass' => null,
		'childrenClass' => 'menu-item--has-children',
		'activeClass' => 'menu-item--active',
		'activeTrailClass' => 'menu-item--active-trail',
		'itemClassPrefix' => 'menu-item--',
		'model' => 'Menu',
		'childrenImageClass' => 'menu-item--has-children-image',
		'depth' => null,
		'disableParentLinks' => false
	);

	public function __construct(View $View, $settings = array()) {
		parent::__construct($View, $settings);

		$this->settings = array_merge($this->settings, (array)$settings);
		$this->user_group_id = $this->Session->read('Auth.User.User.user_group_id');

		return;
	}

/**
 * Returns a formatted menu
 * @param  [type] $data [description]
 * @param  array  $attr [description]
 * @return [type]       [description]
 */
	public function menu($data, $attr = array()) {
		// Set the depth of menu to return.
		$attr = array_merge($this->settings, (array)$attr);
		$depth = $attr['depth'];

		// Build up the menu with the active and active_trail flags set.
		$this->buildMenu($data, $depth, $attr);
		Cache::write('MenuData', $data);
		// Return the marked up menu.
		return $this->_formatMenu($data, $attr, true);
	}

/**
 * Build up the menu array with the correct active/active-trail flags set.
 * @param  array 	$data  menu data
 * @param  integer 	$depth depth to recursively run the method for
 *                         reduces by 1 each run (set to null to
 *                         build the entire menu)
 * @param 	array 	$attr  settings data merged with passed overwritten parameters
 * @param  integer  $activeParent	ID of parent of active item
 * @return array
 */
	public function buildMenu(&$data, $depth = null, $attr = array(), &$activeParent = false) {
		foreach ($data as &$item) {

			$hasChildren = isset($item['children']) && !empty($item['children']);

			if ($hasChildren && ($depth === null | $depth > 1)) {

				$this->buildMenu(
						$item['children'], ($depth === null ? null : $depth - 1), $attr, $activeParent
				);
			} else {

				// We've reached the maximum depth of the current menu so
				// get rid of the children.
				unset($item['children']);
			}

			$item[$attr['model']]['active_trail'] = false;
			// Check for active states

			if ($this->checkActive($item, $attr['model'])) {

				$item[$attr['model']]['active'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			} elseif ($activeParent === $item[$attr['model']]['id']) {

				$item[$attr['model']]['active_trail'] = true;
				$activeParent = $item[$attr['model']]['parent_id'];
			}
		}

		return $data;
	}

/**
 * Build up the menu recursively.
 * @param  array   $data    array containing menu data
 * @param  array   $attr
 * @param  boolean $primary true on the first call, false thereafter to
 *                          indicate child menu
 * @return string  formatted menu
 */
	protected function _formatMenu($data, $attr, $primary = false) {
		$return = '';

		if ($primary === false && !empty($attr['childWrapTag'])) {
			$return .= $this->Html->tag($attr['childWrapTag'], null, ['class' => $attr['childWrapClass']]);
		}

		$id = '';
		$classes = array();

		if ($primary && isset($attr['class']) && !empty($attr['class'])) {

			$classes[] = $attr['class'];
		}

		if ($primary && isset($attr['id']) && !empty($attr['id'])) {

			$id = " id='{$attr['id']}'";
		}

		$class = implode(' ', $classes);

		$dataAttr = '';
		if ($primary && isset($attr['data']) && !empty($attr['data'])) {
			foreach ($attr['data'] as $k => $v) {
				$dataAttr .= ' data-' . $k . '="' . $v . '"';
			}
		}

		$return .= "<{$attr['wrapTag']}$id class='$class'{$dataAttr}>";

		$allHidden = true;
		foreach ($data as $item) {
			if (!empty($item['Menu']['user_group_id']) && $item['Menu']['user_group_id'] < $this->user_group_id) {
				continue;
			}
			$allHidden = false;

			if (isset($item['children'][0]['Menu']['is_menu_hidden'])) {
				$hasChildren = isset($item['children']) && !empty($item['children']) && !$item['children'][0]['Menu']['is_menu_hidden'];
			} else {
				$hasChildren = false;
			}

			$hasChildrenImage = false;
			if (isset($item['children'][0]['MenuImage'][0]['id'])) {
				$hasChildrenImage = true;
			}
			$children = $hasChildren ? $this->_formatMenu($item['children'], $attr, false) : null;

			// Work out the classes to apply to the current menu item.

			$classes = array();

			if (isset($item[$attr['model']]['id']) && !empty($attr['itemClassPrefix'])) {

				$classes[] = $attr['itemClassPrefix'] . $item[$attr['model']]['id'];
			}

			if (isset($item[$attr['model']]['class']) && !empty($item[$attr['model']]['class'])) {

				$classes[] = $item[$attr['model']]['class'];
			}

			if ($hasChildren && !empty($attr['childrenClass'])) {

				$classes[] = $attr['childrenClass'];
			}

			if ($hasChildrenImage && !empty($attr['childrenImageClass'])) {

				$classes[] = $attr['childrenImageClass'];
			}

			if ($item[$attr['model']]['active'] && !empty($attr['activeClass'])) {

				$classes[] = $attr['activeClass'];
			}

			if ($item[$attr['model']]['active_trail'] && !empty($attr['activeTrailClass'])) {

				$classes[] = $attr['activeTrailClass'];
			}



			$class = implode(' ', $classes);

			// Output the menu item.

			$return .= "<{$attr['tag']} class='$class'>";
			if ($primary === true && !empty($attr['parentWrapTag'])) {
				$return .= $this->Html->tag($attr['parentWrapTag'], null, ['class' => $attr['parentWrapClass']]);
			}

			$addImage = '';
			if (isset($item['MenuImage'][0])) {
				$addImage = $this->Image->resize($item['MenuImage'][0], array('width' => 100, 'height' => 100, 'crop' => true));
			}

			$icon = '';
			if (!empty($attr['fa'])) {
				$icon = '<i class="fa fa-fw fa-' . $item['Menu']['icon'] . '"></i> ';
			}

			if (
				!empty($item[$attr['model']]['url'])
				&& ($attr['disableParentLinks'] === false || $hasChildren !== true)
			) {
				$return .= $this->Html->link(
					$addImage .	$icon . $this->I18n->translate($item, $attr['model'] . '.name'),
					$item[$attr['model']]['url'],
					array(
						'escape' => false,
						'target' => (isset($item[$attr['model']]['new_window']) && $item[$attr['model']]['new_window'] ? '_blank' : null)
					)
				);
			} else {
				$return .= $this->Html->tag(
					'span',
					$addImage .	$icon . $this->I18n->translate($item, $attr['model'] . '.name'),
					$item[$attr['model']]['url']
				);
			}

			if ($primary === true && !empty($attr['parentWrapTag'])) {
				$return .= "</{$attr['parentWrapTag']}>";
			}

			if ($hasChildren) {

				$return .= $children;
			}

			$return .= "</{$attr['tag']}>";
		}

		$return .= "</{$attr['wrapTag']}>";

		if ($primary === false && !empty($attr['childWrapTag'])) {
			$return .= "</{$attr['childWrapTag']}>";
		}

		if ($allHidden) {
			$return = '';
		}

		return $return;
	}

/**
 * Check for active state of a menu item. Used by _formatMenu() and can be
 * called from within a view when wanting to play with the unprocessed
 * menu array.
 * @param  array $item 	menu item
 * @param  string $menuModel the menu model as defined by settings or as overwritten by user
 * @return boolean      true if active
 */
	public function checkActive($item, $menuModel) {
		if (isset($item[$menuModel]['active']) && $item[$menuModel]['active']) {

			return true;
		} else {

			return $this->_checkActiveByRoute($item, $menuModel);
		}
	}

/**
 * Check the menu item against the current route.
 * @param  array $item	menu item
 * @param  string $menuModel the menu model as defined by settings or as overwritten by user
 * @return boolean		true if there is a match
 */
	protected function _checkActiveByRoute($item, $menuModel) {
		$isActive = false;

		if (($pos = strrpos($this->here, '/page:')) !== false) {
			$this->here = substr($this->here, 0, $pos);
		}
		// Check if the menu item's URL matches the current page.
		if (!$isActive && $item[$menuModel]['url']) {

			$isActive = Router::normalize($this->here) === Router::normalize($item[$menuModel]['url']);
		}

		// Check if the menu item's pattern matches the current page.
		if (!$isActive && $item[$menuModel]['pattern']) {

			$isActive = preg_match($item[$menuModel]['pattern'], Router::normalize($this->here));
		}

		return $isActive;
	}

	public function breadcrumbs($data, $attr = array()) {
		// Set the depth of menu to return.
		$attr = array_merge($this->settings, (array)$attr);
		$depth = $attr['depth'];
		$breadCrumbData = Cache::read('MenuData');
		$this->buildMenu($data, $depth, $attr);

		// Return the marked up menu.

		$siteTitle = Configure::read('SiteSettings.site_title');
		$return = array(
			$siteTitle => array(
				'Route' => '/'
			)
		);

		$crumbs = $this->_formatBreadcrumbs($data, $attr, true);
		$crumbs = array_reverse($crumbs);
		return array_merge($return, $crumbs);
	}

	protected function _formatBreadcrumbs($data, $attr, $primary = false) {
		$return = array();

		foreach ($data as $item) {

			$hasChildren = isset($item['children']) && !empty($item['children']);

			$hasChildrenImage = false;
			if (isset($item['children'][0]['MenuImage'])) {
				$hasChildrenImage = true;
			}
			$children = $hasChildren ? $this->_formatBreadcrumbs($item['children'], $attr, false) : null;
			if ($children) {

				$return = array_merge($children, $return);
			}

			// Work out the classes to apply to the current menu item.

			$classes = array();

			if ($item[$attr['model']]['active'] && !empty($attr['activeClass'])) {

				$classes[] = $attr['activeClass'];
			}

			if ($item[$attr['model']]['active_trail'] && !empty($attr['activeTrailClass'])) {

				$classes[] = $attr['activeTrailClass'];
			}
			if ($classes && !$item[$attr['model']]['is_breadcrumb_hidden']) {
				$return[$item[$attr['model']]['name']] = array('Route' => $item[$attr['model']]['url']);
			}
		}
		return $return;
	}

}
