<?php
/**
 * Menu model for the Navigation plugin.
 *
 * @author  Andy Carter
 * @package Navigation
 */
App::uses('NavigationAppModel', 'Navigation.Model');

class Menu extends NavigationAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzTranslate.Translatable',
		'Tree'
	);

	public $order = array('lft' => 'ASC');

	public $belongsTo = array(
		'UserGroup' => array(
			'className' => 'EvCore.UserGroup'
		)
	);

/**
 * We want to keep a record of which menu items are currently for use when
 * building our menus.
 *
 * @var int
 */
	private $__menuActiveId = null;

	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'cannot be blank'
			),
			'unique' => array(
				'rule' => array('checkCombinationUnique', array('parent_id')),
				'message' => 'A menu item with this name already exists for this parent'
			)
		),
		'url' => array(
			'url' => array(
				'rule' => 'validateUrls',
				'message' => 'Invalid URL',
				'allowEmpty' => true
			)
		)
	);

/**
 * Custom validation method for checking a URL (internal or external).
 *
 * @param array $data
 * @return bool
 */
	public function validateUrls(array $data) {
		$check = array_pop($data);

		$validates = preg_match('|^\/[\w\-_\/]+|', $check) === 1;
		if ($validates === false) {
			$validates = Validation::url($check);
		}

		return $validates;
	}

	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$this->removeBehaviour('CustomCopyable');

		return;
	}

/**
 * Returns a menu with active menu items flagged as 'active'.
 * @param  integer $id       ID of menu to return
 * @param  integer $activeId optional active menu item ID
 * @return array             array of menu items
 */
	public function getMenu($id, $activeId = null) {
		// Set the ID of the active menu item ready for searching the
		// menu with.
		$activeId = $activeId === null ? $this->__menuActiveId : $activeId;

		// Get the menu record so that we can work out its children
		// using the left and right columns.
		$menu = $this->findById($id);

		// Get the menu's children.
		$data = $this->find('threaded', array(
			'conditions' => array(
				'lft >' => $menu[$this->alias]['lft'],
				'rght <' => $menu[$this->alias]['rght'],
				'is_active' => true
			),
			'order' => array(
				'lft' => 'ASC'
			)
		));

		// Search for the active menu item in the menu data and mark it
		// as active. Then return the menu data.
		return $this->_findActive($data, $activeId);
	}

	protected function _findActive(&$data, $activeId) {
		foreach ($data as &$item) {

			$item[$this->alias]['active'] = ($item[$this->alias]['id'] == $activeId);

			if (isset($item['children'])) {

				$this->_findActive($item['children'], $activeId);

			}

		}

		return $data;
	}

/**
 * Set the active menu item
 * @param integer $id ID of menu item to mark as active
 */
	public function setActive($id) {
		$this->__menuActiveId = $id;

		return true;
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as &$result) {

			if (isset($result[$this->alias]['model']) && !empty($result[$this->alias]['model'])) {

				$result[$this->alias]['url'] = array(
					'controller' => Inflector::tableize($result[$this->alias]['model']),
					'action' => $result[$this->alias]['action']
				);

				// check if there's a plugin set, if not set to false to prevent routing issues when "in a plugin"
				if (isset($result[$this->alias]['plugin']) && !empty($result[$this->alias]['plugin'])) {

					$result[$this->alias]['url']['plugin'] = $result[$this->alias]['plugin'];
				} else {

					$result[$this->alias]['url']['plugin'] = false;
				}

				// add model id if one is set
				if (isset($result[$this->alias]['model_id']) && !empty($result[$this->alias]['model_id'])) {

					$result[$this->alias]['url'][] = $result[$this->alias]['model_id'];
				}
			}
		}

		return $results;
	}

	public function readForEdit($id, $query = array()) {
		$data = parent::readForEdit($id, $query);

		if (!empty($data[$this->alias]['model']) && !empty($data[$this->alias]['id'])) {

			unset($data[$this->alias]['url']);

		}

		return $data;
	}

/**
 * Returns the child menu items of a menu in tree form
 *
 * @param int|array $parentId parent menu ID or an array of IDs
 * @param int $id ID of a menu item to exclude
 * @return array
 */
	public function findTreeBranches($parentId, $id = null) {
		$conditions = array();

		if (is_numeric($parentId) && $parentId != 0) {

			$menu = $this->findById($parentId);

			$conditions = array(
				'lft >=' => $menu[$this->alias]['lft'],
				'rght <=' => $menu[$this->alias]['rght'],
				'is_active' => true
			);

		} elseif (is_array($parentId)) {

			foreach ($parentId as $thisParentId) {

				$menu = $this->findById($thisParentId);

				$conditions['OR'][] = array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght'],
					'is_active' => true
				);

			}

		}

		if (!empty($id)) {

			$menu = $this->findById($id);

			$conditions['NOT'] = array(
				'AND' => array(
					'lft >=' => $menu[$this->alias]['lft'],
					'rght <=' => $menu[$this->alias]['rght']
				)
			);

		}

		return $this->generateTreeList(
			$conditions,
			null,
			null,
			'--'
		);
	}

}
