<?php
/**
 * Menu controller for the Navigation plugin.
 *
 * @author  Andy Carter
 * @package Navigation
 */
App::uses('NavigationAppController', 'Navigation.Controller');

class MenusController extends NavigationAppController {

	public $components = array(
		'BuzzTranslate.Translation' => array(
			'fields' => array(
				'name'
			)
		)
	);

	public function __construct($request = null, $response = null) {
		parent::__construct($request, $response);

		$this->adminActions[] = 'admin_move_up';
		$this->adminActions[] = 'admin_move_down';
	}

/**
 * Prepare data for populating admin dropdowns.
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		parent::_adminPopulateLookups();

		$Menu = $this->{$this->modelClass};

		$conditions = null;
		if ($this->Auth->user('UserGroup.id') > 1) {
			// Remove admin menu from normal admin users.
			$adminMenu = $Menu->findById(Configure::read('Navigation.admin_menu'));
			$conditions = [
				'NOT' => [
					'AND' => [
						$Menu->alias . '.lft >=' . $adminMenu[$Menu->alias]['lft'],
						$Menu->alias . '.rght <=' . $adminMenu[$Menu->alias]['rght']
					]
				]
			];
		}
		$this->set(
			'parents',
			$this->Menu->generateTreeList(
				$conditions,
				null,
				null,
				'--'
			)
		);

		$this->set(
			'userGroups',
			$this->Menu->UserGroup->find(
				'list',
				array('order' => array('UserGroup.id' => 'ASC'))
			)
		);

		return;
	}

/**
 * Prepare the admin form fields for the menu item form
 * @return array
 */
	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};

		$fields = parent::_adminFormFields();

		$fields[$Model->alias . '.url']['label'] = 'URL';

		$fields[$Model->alias . '.new_window']['label'] = 'Open Link in New Window';

		unset($fields[$Model->alias . '.model']);
		unset($fields[$Model->alias . '.action']);
		unset($fields[$Model->alias . '.model_id']);
		unset($fields[$Model->alias . '.plugin']);
		unset($fields[$Model->alias . '.controller']);

		return $fields;
	}

/**
 * Menus admin listings page.
 * @return void
 */
	public function admin_index() {
		$Menu = $this->{$this->modelClass};

		$this->set('toolbar', $this->_adminIndexToolbar());

		// Get the menu items.
		$conditions = null;
		if ($this->Auth->user('UserGroup.id') > 1) {
			// Remove admin menu from normal admin users.
			$adminMenu = $Menu->findById(Configure::read('Navigation.admin_menu'));
			$conditions = [
				'NOT' => [
					'AND' => [
						$Menu->alias . '.lft >=' . $adminMenu[$Menu->alias]['lft'],
						$Menu->alias . '.rght <=' . $adminMenu[$Menu->alias]['rght']
					]
				]
			];
		}
		$data = $Menu->generateTreeList(
			$conditions,
			null,
			null,
			'--'
		);
		$this->set('data', $data);
		$this->set('title_for_layout', InflectorExt::pluralize($Menu->displayName));
		$this->view = 'Navigation.Menus/admin_index';

		return;
	}

/**
 * Menu item edit page.
 *
 * @param int $id menu item ID
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);

		$Menu = $this->{$this->modelClass};

		if ($this->Auth->user('UserGroup.id') > 1) {
			// Prevent normal admin users from viewing the edit form (not the
			// most secure but should suffice).
			$adminMenu = $Menu->findById(Configure::read('Navigation.admin_menu'));
			if (
				!empty($this->request->data)
				&& $this->request->data[$Menu->alias]['lft'] >= $adminMenu[$Menu->alias]['lft']
				&& $this->request->data[$Menu->alias]['rght'] <= $adminMenu[$Menu->alias]['rght']
			) {
				throw new UnauthorizedException();
			}
		}

		// If we're dealing with a menu item associated to another model
		// then we want to remove the URL field from the form. We need
		// to do this directly as the view variables have already been
		// set.
		if (!empty($this->request->data['Menu']['model'])) {
			unset($this->viewVars['fields']['Menu.url']);
		}

		// Hide fields from normal admin users. Evoluted only!
		if ($this->Auth->user('UserGroup.id') > 1) {
			unset(
				$this->viewVars['fields']['Menu.pattern'],
				$this->viewVars['fields']['Menu.class'],
				$this->viewVars['fields']['Menu.user_group_id'],
				$this->viewVars['fields']['Menu.is_menu_hidden'],
				$this->viewVars['fields']['Menu.icon']
			);
		}

		// Can't add to self.
		unset($this->viewVars['parents'][$id]);
		$this->Translation->adminTranslationFormFields();

		return;
	}

	public function admin_move_up($id, $delta = 1) {
		if ($delta > 0) {

			if ($this->Menu->moveUp($id, abs($delta))) {

				$this->Session->setFlash(array(
					'title' => 'Item moved',
					'description' => ''
				), 'flash_success');

			} else {

				$this->Session->setFlash(array(
					'title' => 'Move failed',
					'description' => 'Failed to move menu item!'
				), 'flash_fail');

			}

		}

		$this->redirect(array('action' => 'index'));

		return;
	}

	public function admin_move_down($id, $delta = 1) {
		if ($delta > 0) {

			if ($this->Menu->moveDown($id, abs($delta))) {

				$this->Session->setFlash(array(
					'title' => 'Item moved',
					'description' => ''
				), 'flash_success');

			} else {

				$this->Session->setFlash(array(
					'title' => 'Move failed',
					'description' => 'Failed to move menu item!'
				), 'flash_fail');

			}

		}

		$this->redirect(array('action' => 'index'));

		return;
	}

	public function admin_reorder() {
		parent::admin_reorder();
		$this->view = 'Navigation.admin_reorder';
		return;
	}

}
