<?php
App::uses('BuzzMotoAppController', 'BuzzMoto.Controller');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class MotoPaymentsController extends BuzzMotoAppController implements PurchasableInterface {

/**
 * Components
 *
 * @var array
 */
	public $components = [
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

	public function beforeFilter() {
		parent::beforeFilter();
		$this->Auth->allow(['pay', 'payment_callback', 'confirmation']);
	}

/**
 * Make a MOTO payment
 * @param string $ref Transaction ref supplied by user/client
 * @param float $amount Amount to be paid
 * @return void
 */
	public function pay($ref, $amount) {
		if (empty($amount)) {
			throw new ForbiddenException('Cannot take a MOTO payment of zero');
		} elseif (!empty($this->request->data)) {
			if (! $this->_pay()) {
				$this->Session->setFlash(
					__d('buzz_moto', 'The payment details are incomplete/incorrect'),
					'flash_fail'
				);
			}
		} else {
			// Every time someone arrives at this page we want to log it.
			$this->MotoPayment->createPaymentRequest($ref, $amount);
		}
		$this->_populateLookups();
		$this->layout = 'moto';

		return;
	}

	protected function _pay() {
		$MotoPayment = $this->{$this->modelClass};
		if (Configure::read('BuzzPurchase.onsite') === true) {
			$payment = $this->request->data['Payment'];

			// Split name into first and second names for payment gateways.
			$fullName = trim($payment['card_name']);
			$splitName = explode(' ', $fullName);
			$payment['last_name'] = array_pop($splitName);
			$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
		} else {
			throw new ForbiddenException('MOTO payments are not permitted on this site');
		}

		// Get the latest basket.
		$id = $MotoPayment->getBasketId();
		$basket = $MotoPayment->findById($id);

		if (empty($basket)) {
			return false;
		}

		$transaction = $this->Transactions->takePayment(
			// Return URLs
			[
				'return' => Router::url(['action' => 'payment_callback'], true),
				'cancel' => Router::url(['action' => 'pay', $basket['MotoPayment']['ref'], $basket['MotoPayment']['amount']], true)
			],
			// Calling record
			[
				'model' => 'BuzzMoto.MotoPayment',
				'model_id' => $basket['MotoPayment']['id']
			],
			// Amount to be paid
			$basket['MotoPayment']['amount'],
			// Items
			[],
			// Extras
			[
				'language' => Configure::read('Config.language'),
				'card' => $payment,
				'address' => null,
				'user' => null,
				'description' => __d(
					'buzz_moto',
					'%s Moto Payment %d',
					[Configure::read('SiteSetting.site_title'), $basket['MotoPayment']['ref']]
				)
			],
			0,
			Configure::read('BuzzPurchase.payment_gateway')
		);

		if (!empty($transaction['result'])) {
			// Payment taken successfully, complete purchase.
			return $this->_completePayment($id);
		} else {
			$this->Session->setFlash(
				__d('buzz_moto', 'There was a problem processing your payment, please check your details and try again'),
				'flash_fail'
			);
		}
	}

	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();
		if ($result === true) {
			return $this->_completePayment($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_moto', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'pay'
			]);
		}
	}

	protected function _completePayment($basketId) {
		$MotoPayment = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		$this->Session->write('MotoPayment.hash', $MotoPayment->hashBasketId());
		$MotoPayment->completePayment(true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	public function confirmation($id) {
		$MotoPayment = $this->{$this->modelClass};
		if ($MotoPayment->hashBasketId($id) === $this->Session->read('MotoPayment.hash')) {
			$this->Session->delete('MotoPayment.hash');
		} else {
			throw new ForbiddenException();
		}
		$this->layout = 'moto';

		return;
	}

/**
 * Populate lookups.
 *
 * @return void
 */
	protected function _populateLookups() {
		$MotoPayment = $this->{$this->modelClass};

		$countries = $MotoPayment->CustomerAddress->Country->translatedList();
		$usStates = $MotoPayment->CustomerAddress->UsState->find('list');

		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());

		$this->set(compact('countries', 'usStates'));
		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);

		if ((int)$this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('MotoPayment', $id));
		}

		$this->view = 'admin_form';

		return;
	}

/**
 * Used to populate form drop down selects
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$MotoPayment = $this->{$this->modelClass};
		$this->set('motoPaymentStates', $MotoPayment->MotoPaymentState->find('list'));
		return;
	}

/**
 * Admin index paginate
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();
		$paginate = [
			'conditions' => $conditions,
			'contain' => 'MotoPaymentState'
		];

		return $paginate;
	}

/**
 * Admin index columns
 *
 * @return array
 */
	protected function _adminIndexColumns() {
		$MotoPayment = $this->{$this->modelClass};
		$columns = parent::_adminIndexColumns();
		$columns[$MotoPayment->alias . '.amount']['type'] = 'currency';
		$newColumns = array(
			'MotoPaymentState.name' => array(
				'label' => __d('buzz_moto', 'Status'),
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'MotoPayment.amount', $newColumns);
	}

/**
 * Admin columns whitelist
 *
 * @return array
 */
	protected function _adminIndexColumnsWhitelist() {
		$MotoPayment = $this->{$this->modelClass};
		$whitelist = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $MotoPayment->alias . '.amount';

		return $whitelist;
	}

/**
 * Filters
 *
 * @return array
 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		$newFilters = array(
			'MotoPayment.moto_payment_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => MotoPaymentState::COMPLETE,
				'compare' => array('MotoPayment.moto_payment_state_id' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'MotoPayment.ref', $newFilters);
	}

}
