<?php
App::uses('AppModel', 'Model');
App::uses('BookingState', 'BuzzBookings.Model');

class LateBooking extends BuzzLateBookingsAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

/**
 * Default order
 *
 * @var array|string
 */
	public $order = 'LateBooking.id DESC';

	public $belongsTo = [
		'BookingState' => [
			'className' => 'BuzzBookings.BookingState'
		],
		'CustomerAddress' => [
			'className' => 'BuzzCustomers.CustomerAddress'
		],
		'LateBookingPackage' => [
			'className' => 'BuzzLateBookings.LateBookingPackage'
		]
	];

/**
 * Get the current basket
 *
 * @param int $bookingId
 * @return array
 */
	public function getBasket($lateBookingId = null) {
		$lateBookingId = $lateBookingId ?: $this->getBasketId();
		if (empty($lateBookingId)) {
			return [];
		}
		$params = [
			'contain' => [
				'CustomerAddress' => ['Country'],
				'LateBookingPackage'
			],
			'conditions' => [
				'LateBooking.booking_state_id' => BookingState::UNPAID
			]
		];
		return $this->readForView($lateBookingId, $params);
	}

/**
 * Get a completed booking to display to the customer.
 *
 * @param int $id Purchase ID
 * @return array
 */
	public function getPurchase($id) {
		$params = array(
			'conditions' => array(
				'LateBooking.booking_state_id' => array(
					BookingState::COMPLETE,
					BookingState::API_FAILED
				)
			)
		);
		return $this->readForView($id, $params);
	}

/**
 * Read for edit
 *
 * @param int $id Booking ID
 * @param array $params
 * @return array
 */
	public function readForEdit($id, $params = []) {
		$defaults = array(
			'contain' => [
				'CustomerAddress' => ['Country'],
				'LateBookingPackage'
			]
		);
		$params = Hash::merge($defaults, $params);
		$results = parent::readForEdit($id, $params);

		// Set the participants against the booking items.
		if (!empty($results['BookingItem'])) {
			foreach ($results['BookingItem'] as &$item) {
				if (!empty($item['BookingItemVoucher'])) {
					$item['participants'] = $item['BookingItemVoucher']['participants'];
				} else {
					$item['participants'] = $item['ActivityPackage']['participants'];
				}
			}
		}

		return $results;
	}

/**
 * Update the current basket total.
 *
 * @return bool
 */
	protected function _updateBasketTotal() {
		$basketId = $this->getBasketId();
		$basket = $this->getBasket();
		$total = $basket[$this->alias]['subtotal'];
		if ($basket[$this->alias]['is_insured'] === true) {
			$total += $basket[$this->alias]['insurance_cost'];
		}

		// Save the update total.
		$data = array(
			'id' => $basketId,
			'total_cost' => $total
		);

		return $this->save($data) !== false;
	}

/**
 * Get the available late booking deals.
 *
 * @param int $siteId Site ID
 * @param string $fromDate From date
 * @param string $toDate To date which the method will modify according to the booking
 * @return array
 */
	public function getAvailableDeals($siteId, $fromDate, $toDate) {
		// Get the times and dates available for late bookings.
		$data = ClassRegistry::init('BuzzLateBookings.LateBookingApi')->getLastMinuteDatesBySite(
			(int)$siteId,
			$fromDate,
			$toDate
		);

		if (empty($data)) {
			return [];
		}

		$lateBookingPackages = $this->LateBookingPackage->find('all', [
			'contain' => [
				'Activity',
				'Image'
			],
			'conditions' => [
				'LateBookingPackage.is_active' => true
			]
		]);
		$packagesMap = [];
		foreach ($lateBookingPackages as $key => $package) {
			$packageApiRef = $package['LateBookingPackage']['package_api_reference'];
			$activityApiRef = $package['Activity']['api_reference'];
			$packagesMap[$activityApiRef][$packageApiRef] = $key;
		}

		$deals = [];
		foreach ($data as $row) {
			$date = $row['date'];
			foreach ($row['times'] as $time) {
				$activityApiRef = $time['activity_api_id'];
				$packageApiRef = $time['package_api_id'];
				if (isset($packagesMap[$activityApiRef][$packageApiRef])) {
					$deals[$date][] = $time + $lateBookingPackages[$packagesMap[$activityApiRef][$packageApiRef]];
				}
			}
		}

		return $deals;
	}

/**
 * Returns an array of unavailable dates from the specified date range.
 *
 * @param array $dates
 * @param string $fromDate
 * @param string $toDate
 * @return array
 */
	public function calculateUnavailableDates(array $dates, $fromDate, $toDate) {
		$availableDates = [];
		$unavailableDates = [];

		$dates = array_keys($dates);
		foreach ($dates as $date) {
			$availableDates[$date] = true;
		}

		$fromDate = strtotime($fromDate);
		$toDate = strtotime($toDate);
		for ($date = $fromDate; $date <= $toDate; $date += 86400) {
			$dateKey = date('Y-m-d', $date);
			if (isset($availableDates[$dateKey]) === false) {
				$unavailableDates[$dateKey] = true;
			}
		}

		return $unavailableDates;
	}

/**
 * Add a temporary booking
 *
 * @param int   $siteId id of the site to add the temporary booking for
 * @param array $deal array of deal data
 * @param int   $quantity number of participants
 *
 * @return bool
 * @throws Exception
 */
	public function addTemporaryBooking($siteId, array $deal, $quantity = 1) {
		$datetime = $deal['date'] . ' ' . $deal['from_time'];
		$checkInTime = strtotime($deal['from_time']) - (int)$deal['check_in'] * 60;
		$id = $this->getBasketId();
		$this->save([
			'id' => $id,
			'late_booking_package_id' => $deal['LateBookingPackage']['id'],
			'check_in_time' => date('H:i', $checkInTime),
			'booking_date' => $datetime,
			'quantity' => $quantity,
			'subtotal' => $quantity * $deal['LateBookingPackage']['price'],
			'total_cost' => $quantity * $deal['LateBookingPackage']['price'],
			'booking_state_id' => BookingState::UNPAID,
			'api_site_id' => $siteId
		]);
		$this->setBasketId($this->id);
		$booking = $this->getBasket();
		$result = ClassRegistry::init('BuzzBookings.BookingApi')->createTemporaryBookingWithSale(
			$this->name,
			$this->id,
			$siteId,
			$deal['package_api_id'],
			$deal['experience_id'],
			$quantity,
			$datetime,
			$booking['LateBooking']['sales_ref'],
			$deal['LateBookingPackage']['price'],
			$deal['LateBookingPackage']['api_discount_code']
		);

		if ($result !== false) {
			return (bool)$this->saveField('sales_ref', $result['sales_ref']);
		} else {
			return false;
		}
	}

/**
 * Returns and saves the insurance cost for the current booking.
 *
 * @param array $basket basket data
 * @return float
 */
	public function getInsuranceCost($basket) {
		if (Configure::read('BuzzBookings.insurable') === false) {
			return false;
		}

		// Get the rate from the API.
		$rate = ClassRegistry::init('BuzzBookings.BookingApi')->getCancellationPercentage();
		if ($rate !== false) {
			$rate /= 100;
		} else {
			$rate = 0.1;
		}

		$cost = number_format((float)$basket[$this->alias]['subtotal'] * $rate, 2, '.', '');

		// Store the insurance cost.
		$this->id = $basket['LateBooking']['id'];
		$this->saveField('insurance_cost', $cost);

		return $cost;
	}

/**
 * Updates the insurance status of the basket.
 *
 * @param bool $isInsured
 * @return bool
 */
	public function updateBookingInsurance($isInsured) {
		$bookingId = $this->getBasketId();
		$result = $this->save([
			'id' => $bookingId,
			'is_insured' => (bool)$isInsured
		]);
		if ($result !== false) {
			$this->_updateBasketTotal();
		}
		return (bool)$result;
	}

/**
 * Add the customer billing details to the booking
 *
 * @param array $customer Customer data
 * @param bool  $newsletterSignup Flag for newsletter signup
 * @return bool
 */
	public function addBillingDetails(array $customer, $newsletterSignup = false) {
		$data = array(
			'LateBooking' => array(
				'id' => $this->getBasketId(),
				'newsletter_opt_in' => $newsletterSignup
			),
			'CustomerAddress' => $customer
		);
		return $this->saveAssociated($data) !== false;
	}

/**
 * Complete the purchase
 *
 * @param bool $paid True if payment successfully made
 * @return bool
 */
	public function completePurchase($paid) {
		$id = $this->getBasketId();

		$basket = $this->getBasket($id);

		$response = true;

		if ((int)$basket[$this->alias]['booking_state_id'] === BookingState::UNPAID) {
			if ($paid === true) {


				if ($basket[$this->alias]['is_insured']) {
					// Add insurance to the booking via the API.
					$this->_addItemToBooking(
						$basket[$this->alias]['api_site_id'],
						$basket[$this->alias]['sales_ref'],
						'#INSURANCE',
						1,
						$basket[$this->alias]['insurance_cost']
					);
				}

				// Complete the booking.
				$result = $this->_completeBooking(
					$basket[$this->alias]['sales_ref'],
					$basket['CustomerAddress']['first_name'],
					$basket['CustomerAddress']['last_name'],
					$basket['CustomerAddress']['email'],
					$basket['CustomerAddress']['telephone'],
					CustomerAddress::generateAddressArray($basket['CustomerAddress']),
					$basket[$this->alias]['total_cost'],
					$this->getPaymentDetails($id, 'BuzzLateBookings.LateBooking'),
					null
				);

				$data = array(
					'id' => $id,
					'booking_state_id' => $result === true ? BookingState::COMPLETE : BookingState::API_FAILED,
					'completed_date' => gmdate('Y-m-d H:i:s')
				);

				if ($result === false) {
					$data['api_log'] = json_encode(ClassRegistry::init('BuzzBookings.BookingApi')->getApiResponses());
				}

				$response = $this->save($data);
			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'booking_state_id' => BookingState::PAYMENT_FAILED
				);
				$response = $this->save($data);
			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.LateBooking.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}

		if (! empty($basket['LateBooking']['newsletter_opt_in']) && CakePlugin::loaded('BuzzSubscribe')) {
			ClassRegistry::init('BuzzSubscribe.Subscriber')->addSubscriber(
				$basket['CustomerAddress']['email'],
				$basket['CustomerAddress']['first_name'],
				$basket['CustomerAddress']['last_name'],
				'Late Deal Booking'
			);
		}

		return $response;
	}

/**
 * Add an item to the booking via the API.
 *
 * @param int $salesRef
 * @param string $apiRef
 * @param int $quantity
 * @param float $price
 * @return array|bool
 */
	protected function _addItemToBooking($siteId, $salesRef, $apiRef, $quantity, $price) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->addItemToSalesOrder(
			$this->name,
			$this->getBasketId(),
			$siteId,
			$salesRef,
			$apiRef,
			$quantity,
			$price
		);
	}

/**
 * Complete a booking via the API.
 *
 * @param int $salesRef,
 * @param string $firstName,
 * @param string $lastName,
 * @param string $email,
 * @param string $telephone,
 * @param array $address,
 * @param float $totalPaid,
 * @param array $paymentDetails,
 * @param string $notes
 * @return bool
 */
	protected function _completeBooking($salesRef, $firstName, $lastName, $email, $telephone, array $address, $totalPaid, array $paymentDetails, $notes = null) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->completeSale(
			$this->name,
			$this->getBasketId(),
			$salesRef,
			$firstName,
			$lastName,
			$email,
			$telephone,
			$address,
			$totalPaid,
			$paymentDetails,
			$notes
		);
	}

}
