<?php
App::uses('BuzzLateBookingsAppController', 'BuzzLateBookings.Controller');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');
App::uses('TimerTrait', 'BuzzBookings.Lib/Trait');

class LateBookingsController extends BuzzLateBookingsAppController implements PurchasableInterface
{

	use TimerTrait;

	public $timeoutRedirect = ['action' => 'index'];

	/**
	 * Helpers
	 */
	public $helpers = [
		'Address.Address'
	];

	public function beforeFilter()
	{
		parent::beforeFilter();

		$this->Auth->allow([
			'ajax_date',
			'ajax_find_deals',
			'basket',
			'book',
			'checkout',
			'confirmation',
			'index',
			'insurance',
			'payment_callback',
			'timeout'
		]);
	}

	/**
	 * Components
	 *
	 * @var array
	 */
	public $components = [
		'BuzzPurchase.EcommerceTracking',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

	public function index() {
		if (Configure::check('BuzzLateBookings.late_booking_page_id') === true) {
			$this->assignPage(Configure::read('BuzzLateBookings.late_booking_page_id'));
		}
		$activities = $this->LateBooking->LateBookingPackage->getActivities();
		$participants = array_combine(range(1, 12), range(1, 12));

		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->getAll();
		$siteList = [];

		$translateUtil = new TranslateUtil();

		foreach ($sites as $site) {
			$siteList[$site['Site']['api_site_id']] = $translateUtil->translate($site, 'Site.name');
		}

		$sites = $siteList;

		$this->Meta->canonical(['action' => 'index']);
		$this->set(compact('activities', 'participants', 'sites'));
	}

/**
 * Find Deals AJAX call
 *
 * @return void
 */
	public function ajax_find_deals() {
		$this->_postFindDeals();
		$this->layout = 'ajax';
	}

/**
 * Find deals using post data
 *
 * @return void
 */
	protected function _postFindDeals() {
		ini_set('memory_limit', '256M');
		$fromDate = date('Y-m-d');
		$toDate = date('Y-m-t', strtotime('+1 month'));

		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->getAll();
		$siteId = ! empty($this->request->data['LateBooking']['api_site_id']) ? $this->request->data['LateBooking']['api_site_id'] : null;

		if (! in_array($siteId, Hash::extract($sites, '{n}.Site.api_site_id'))) {
			$defaultSite = $this->Site->getDefaultSite($sites);
			$siteId = $defaultSite['Site']['api_site_id'];
		}

		// Get all the available deals until the end of next month and apply the filters.
		$deals = $this->_filterDeals(
			$this->LateBooking->getAvailableDeals($siteId, $fromDate, $toDate),
			(int)$this->request->data['LateBooking']['activities_id'],
			(int)$this->request->data['LateBooking']['participants']
		);
		// Store the filtered available deals in the session.
		$this->Session->write('BuzzLateBookings.deals', $deals);
		// Determine the unavailable dates.
		$unavailableDates = $this->LateBooking->calculateUnavailableDates($deals, $fromDate, $toDate);
		$this->set(compact('fromDate', 'toDate', 'unavailableDates'));
		// Set the default date, this wants to be the first day with available deals.
		$date = $fromDate;
		while (! empty($unavailableDates[$date]) && $date < $toDate) {
			$date = date('Y-m-d', strtotime($date . ' + 1 day'));
		}
		$this->set(compact('siteId'));
		$this->set('defaultDate', $date);
	}

	/**
	 * Filter the deals
	 *
	 * @param array $deals        Available deals to filter
	 * @param int   $activityId   Activity ID
	 * @param int   $participants Number of participants
	 *
	 * @return array Filtered deals
	 */
	protected function _filterDeals($deals, $activityId, $participants)
	{
		$deals = $this->_filterDealsByActivity($deals, $activityId);
		$deals = $this->_filterDealsByParticipants($deals, $participants);

		return $deals;
	}

	/**
	 * Filter the deals by activity
	 *
	 * @param array $deals      Available deals to filter
	 * @param int   $activityId Activity ID
	 *
	 * @return array Filtered deals
	 */
	protected function _filterDealsByActivity($deals, $activityId)
	{
		if (! empty($activityId)) {
			$this->loadModel('BuzzBookings.Activity');
			$activity = $this->Activity->findById($activityId);
			if (! empty($activity) && $activity['Activity']['api_reference'] !== null) {
				$apiRef = (int) $activity['Activity']['api_reference'];
				foreach ($deals as $date => $packages) {
					foreach ($packages as $key => $package) {
						if ($package['activity_api_id'] !== $apiRef) {
							unset($deals[$date][$key]);
						}
					}
				}
			}
		}

		// Remove any empty array values from the filtered results.
		return array_filter($deals);
	}

	/**
	 * Filter the deals by participants
	 *
	 * @param array $deals        Available deals to filter
	 * @param int   $participants Number of participants
	 *
	 * @return array Filtered deals
	 */
	protected function _filterDealsByParticipants($deals, $participants)
	{
		if (! empty($participants)) {
			foreach ($deals as $date => &$packages) {
				foreach ($packages as $key => &$package) {
					if ((int) $package['LateBookingPackage']['participants'] !== $participants) {
						$quantity = $participants / (int) $package['LateBookingPackage']['participants'];
						if (! is_int($quantity) || $package['quantity'] < $quantity) {
							unset($deals[$date][$key]);
						} else {
							// Set the required quantity so that we don't need to calculate this
							// again.
							$package['quantity_required'] = $quantity;
						}
					}
				}
			}
		}

		// Remove any empty array values from the filtered results.
		return array_filter($deals);
	}

	/**
	 * Book a late deal
	 *
	 * @return void
	 */
	public function book()
	{
		$LateBooking = $this->{$this->modelClass};
		if (! $this->request->is('post')) {
			return $this->redirect(['action' => 'index']);
		}

		$deals  = $this->Session->read('BuzzLateBookings.deals');
		$deal   = Hash::get($deals, $this->request->data['LateBooking']['key']);
		$siteId = $this->request->data['LateBooking']['api_site_id'];

		// If the basket exists we want to delete the temporary bookings before creating new ones.
		// Otherwise the diary completes more bookings than are paid for.
		$basketId = $LateBooking->getBasketId();
		if (! empty($basketId)) {
			$result = ClassRegistry::init('BuzzBookings.BookingApi')->deleteTemporaryBookings(
				$LateBooking->name,
				$basketId
			);
			if ($result === false) {
				throw new InternalErrorException();
			}
		}

		$result = $this->LateBooking->addTemporaryBooking(
			$siteId,
			$deal,
			$this->request->data['LateBooking']['quantity']
		);

		if ($result === true) {
			// Get rid of the available deals from the session (these will need regenerating if
			// the customer returns to this step).
			$this->Session->delete('BuzzLateBookings.deals');
			// Start the booking timer.
			$this->_startTimer();

			// Redirect to the next step.
			return $this->redirect(['action' => 'insurance']);
		}
		// Handle fail case (this will occur when the deal is no longer available).
		$this->Session->setFlash(__d('buzz_late_bookings', 'This deal is no longer available'), 'flash_fail');

		return $this->redirect(['action' => 'index']);
	}

	public function ajax_date($siteId, $date)
	{
		ini_set('memory_limit', '256M');
		$deals = $this->Session->read('BuzzLateBookings.deals.' . $date);
		$this->set(compact('date', 'deals', 'siteId'));
		$this->layout = 'ajax';
		$this->view   = 'BuzzLateBookings.ajax_date';

		return;
	}

	public function insurance()
	{
		$LateBooking = $this->{$this->modelClass};
		// Get the basket.
		$basket = $LateBooking->getBasket();

		$this->_checkTimeRemaining();

		if (! empty($this->request->data)) {
			$result = $LateBooking->updateBookingInsurance($this->request->data['LateBooking']['is_insured']);
			if ($result === true) {
				return $this->redirect(['action' => 'basket']);
			} else {
				$this->Session->setFlash(__d('buzz_late_bookings', 'Please select one of the options below', 'flash_fail'));
			}
		}

		// Get the insurance cost for the basket.
		$insuranceCost = (float) $LateBooking->getInsuranceCost($basket);
		if (empty($insuranceCost)) {
			return $this->redirect(['action' => 'basket']);
		}

		$site = null;
		if (
			CakePlugin::loaded('BuzzSites') &&
			! empty($basket['LateBooking']['api_site_id'])
		) {
			$this->loadModel('BuzzSites.Site');
			$this->Site->setActiveSite($basket['LateBooking']['api_site_id'], true);
			$site = $this->Site->getActiveSite();
		}

		$this->set(compact('basket', 'insuranceCost', 'site'));
		$this->view = 'BuzzLateBookings.insurance';

		return;
	}

	public function basket()
	{
		$LateBooking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $LateBooking->getBasket();
		if (empty($basket['LateBooking']['booking_date'])) {
			$this->Session->setFlash(
				__d('buzz_late_bookings', 'Your session has expired'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'index']);
		}
		if (! empty($this->request->data)) {
			return $this->redirect(['action' => 'checkout']);
		}

		$previousStep = ['action' => 'index'];
		if ((float) $basket['LateBooking']['insurance_cost'] > 0) {
			$previousStep = ['action' => 'insurance'];
		}

		// Determine the step number based on previous steps.
		$step = 3;
		if ((float) $basket['LateBooking']['insurance_cost'] < 0.01) {
			--$step;
		}

		$site = null;
		if (
			CakePlugin::loaded('BuzzSites') &&
			! empty($basket['LateBooking']['api_site_id'])
		) {
			$this->loadModel('BuzzSites.Site');
			$this->Site->setActiveSite($basket['LateBooking']['api_site_id'], true);
			$site = $this->Site->getActiveSite();
		}

		$this->set(compact('basket', 'previousStep', 'step', 'site'));

		return;
	}

	/**
	 * Checkout step
	 *
	 * @return void
	 */
	public function checkout()
	{
		$LateBooking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $LateBooking->getBasket();
		if (empty($basket['LateBooking']['booking_date'])) {
			$this->Session->setFlash(
				__d('buzz_late_bookings', 'Your session has expired'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'index']);
		}

		$this->loadModel('BuzzSites.Site');
		$this->Site->setActiveSite($basket['LateBooking']['api_site_id'], true);
		$site = $this->Site->getActiveSite();
		$this->set(compact('site'));

		if (! empty($this->request->data)) {
			$result = $LateBooking->addBillingDetails(
				$this->request->data['CustomerAddress'],
				! empty($this->request->data['LateBooking']['newsletter_opt_in'])
			);
			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();
			if ($result === true && $paymentValidates === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_late_bookings', 'Please correct the errors below'),
					'flash_fail'
				);
			}
		} else {
			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['CustomerAddress']['country_id']  = CustomerAddress::getDefaultCountry();
			$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();
		}

		// Get the basket.
		$basket = $LateBooking->getBasket();

		// Get gift voucher purchase conditions
		$this->loadModel('BuzzConditions.Condition');
		// Prefix conditions with a default arrival time condition.
		$conditions = [
			-1 => [
				'Condition' => [
					'id'               => 0,
					'rendered_content' => __d(
						'buzz_bookings',
						'I understand that I need to arrive by %s in order to take part in my first activity',
						$basket['LateBooking']['check_in_time']
					)
				]
			]
		];
		$conditions += $this->Condition->getConditions('Booking');

		$this->set(compact('basket', 'conditions'));
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();

		return;
	}

	/**
	 * Processes the payment and triggers the completion of the payment on success
	 *
	 * @return void
	 */
	protected function _pay()
	{
		$LateBooking = $this->{$this->modelClass};

		// Get the latest basket.
		$basket = $LateBooking->getBasket();

		if ($basket['LateBooking']['total_cost'] > 0) {
			if (Configure::read('BuzzPurchase.onsite') === true) {
				$payment = $this->request->data['Payment'];

				// Split name into first and second names for payment gateways.
				$fullName              = trim($payment['card_name']);
				$splitName             = explode(' ', $fullName);
				$payment['last_name']  = array_pop($splitName);
				$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
			} else {
				$payment = [];
			}

			$transaction = $this->Transactions->takePayment(
			// Return URLs
				[
					'return' => Router::url(['action' => 'payment_callback', $basket['LateBooking']['id']], true),
					'cancel' => Router::url(['action' => 'checkout'], true)
				],
				// Calling record
				[
					'model'    => 'BuzzLateBookings.LateBooking',
					'model_id' => $basket['LateBooking']['id']
				],
				// Amount to be paid
				$basket['LateBooking']['total_cost'],
				// Items
				[],
				// Extras
				[
					'language'    => Configure::read('Config.language'),
					'card'        => $payment,
					'address'     => $basket['CustomerAddress'],
					'user'        => ['email' => $basket['CustomerAddress']['email']],
					'description' => __d(
						'buzz_bookings',
						'%s Booking %d',
						[Configure::read('SiteSetting.site_title'), $basket['LateBooking']['id']]
					),
					'api_site_id' => $basket['LateBooking']['api_site_id']
				],
				0,
				Configure::read('Transactions.gateway')
			);

			if (! empty($transaction['result'])) {
				// Payment taken successfully, complete purchase.
				return $this->_completePurchase($basket['LateBooking']['id']);
			} else {
				$this->Session->setFlash(
					__d(
						'buzz_bookings',
						'There was a problem processing your payment, please check your details and try again'
					),
					'flash_fail'
				);
			}
		} else {
			// Nothing to pay, complete purchase.
			return $this->_completePurchase($basket['LateBooking']['id']);
		}

		return;
	}

	/**
	 * Generic payment gateway callback.
	 *
	 * @param int $basketId Basket ID
	 *
	 * @return void
	 */
	public function payment_callback($basketId)
	{
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completePurchase($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_bookings', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);

			return $this->redirect([
				'action' => 'checkout'
			]);
		}
	}

	/**
	 * Complete the purchase (after payment received)
	 *
	 * @param int $basketId Basket ID
	 *
	 * @return void Redirects to confirmation page
	 */
	protected function _completePurchase($basketId)
	{
		$LateBooking = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$LateBooking->setBasketId($basketId);
		$LateBooking->completePurchase(true);

		// The hash *must* be set after completing the purchase as
		// completePurchase() will delete the session.
		$this->Session->write('LateBooking.hash', $LateBooking->hashBasketId($basketId));

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	/**
	 * Confirmation page
	 *
	 * @param int $id Basket ID
	 *
	 * @return void
	 */
	public function confirmation($id)
	{
		$LateBooking = $this->{$this->modelClass};

		if ($LateBooking->hashBasketId($id) === $this->Session->read('LateBooking.hash')) {
			$this->Session->delete('LateBooking.hash');

			$basket = $LateBooking->getPurchase($id);

			if (Configure::check('BuzzBookings.confirmation_page_id') === true) {
				$this->assignPage(Configure::read('BuzzBookings.confirmation_page_id'));
			}

			if (! empty($basket['LateBookingPackage']['id'])) {
				$this->EcommerceTracking->addItem(
					$basket['LateBookingPackage']['name'],
					'B' . $basket['LateBookingPackage']['id'],
					$basket['LateBooking']['subtotal'],
					1,
					'LateBooking'
				);
			}
			if ($basket['LateBooking']['is_insured'] === true) {
				$this->EcommerceTracking->addItem(
					'Insurance',
					'INSURANCE',
					$basket['LateBooking']['insurance_cost'],
					1,
					'LateBooking'
				);
			}
			$tax               = 0;
			$ecommerceTracking = $this->EcommerceTracking->transaction(
				$basket['LateBooking']['sales_ref'],
				$basket['LateBooking']['total_cost'],
				CakeNumber::defaultCurrency(),
				$tax
			);

			$this->loadModel('BuzzSites.Site');
			$this->Site->setActiveSite($basket['LateBooking']['api_site_id'], true);
			$site = $this->Site->getActiveSite();

			$this->set(compact('ecommerceTracking', 'basket', 'site'));
			$this->view = 'BuzzLateBookings.confirmation';
		} else {
			throw new ForbiddenException();
		}

		return;
	}

	/**
	 * Populate lookups for address fields.
	 *
	 * @return void
	 */
	protected function _populateLookups()
	{
		$Booking = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$usStates  = $Booking->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));

		return;
	}

	/**
	 * Admin index paginate
	 *
	 * @return array
	 */
	protected function _adminIndexPaginate()
	{
		$conditions = $this->_processFilter();

		$paginate = [
			'conditions' => $conditions,
			'contain'    => [
				'CustomerAddress',
				'BookingState'
			]
		];

		$paginate['joins'] = [
			[
				'table' => 'sites',
				'alias' => 'Site',
				'type' => 'LEFT',
				'conditions' => 'LateBooking.api_site_id = Site.api_site_id'
			]
		];

		$paginate['fields'] = [
			'LateBooking.*',
			'CONCAT(`CustomerAddress`.`first_name`, \' \', `CustomerAddress`.`last_name`) as full_name',
			'BookingState.*',
			'Site.*'
		];

		return $paginate;
	}

	/**
	 * Admin index columns
	 *
	 * @return array
	 */
	protected function _adminIndexColumns()
	{
		$LateBooking = $this->{$this->modelClass};

		$columns = parent::_adminIndexColumns();

		// Remove created/modified columns.
		unset($columns['LateBooking.created']);
		unset($columns['LateBooking.modified']);

		$columns[$LateBooking->alias . '.total_cost']['type'] = 'currency';

		$newColumns = [
			'0.full_name' => [
				'label' => __d('buzz_late_bookings', 'Customer'),
				'type'  => 'string'
			],
			'Site.name' => [
				'label' => __d('buzz_bookings', 'Location'),
				'type' => 'string'
			],
			'BookingState.name'         => [
				'label' => __d('buzz_late_bookings', 'Status'),
				'type'  => 'string'
			]
		];

		return ArrayUtil::addAfter($columns, 'LateBooking.sales_ref', $newColumns);
	}

	/**
	 * Admin columns whitelist
	 *
	 * @return array
	 */
	protected function _adminIndexColumnsWhitelist()
	{
		$LateBooking = $this->{$this->modelClass};

		$whitelist   = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $LateBooking->alias . '.sales_ref';
		$whitelist[] = $LateBooking->alias . '.total_cost';
		$whitelist[] = $LateBooking->alias . '.completed_date';

		return $whitelist;
	}

	/**
	 * Filters
	 *
	 * @return array
	 */
	protected function _adminFilterFields()
	{
		$filters = parent::_adminFilterFields();

		unset($filters['LateBooking.name']);
		unset($filters['LateBooking.created']);
		unset($filters['LateBooking.modified']);

		$newFilters = [
			'LateBooking.sales_ref'        => [
				'label'   => __d('buzz_bookings', 'Sales Ref'),
				'type'    => 'string',
				'compare' => ['LateBooking.sales_ref' => "%s"]
			],
			'CustomerAddress.full_name'    => [
				'label'   => __d('buzz_bookings', 'Customer'),
				'type'    => 'string',
				'compare' => ['CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%"]
			],
			'LateBooking.booking_state_id' => [
				'label'   => 'Status',
				'type'    => 'select',
				'default' => BookingState::COMPLETE,
				'compare' => ['LateBooking.booking_state_id' => '%s']
			],
			'LateBooking.completed_date'   => [
				'label'   => __d('buzz_bookings', 'Completed Date'),
				'type'    => 'date',
				'compare' => ['LateBooking.completed_date' => '%s']
			]
		];

		return ArrayUtil::addAfter($filters, 'LateBooking.sales_ref', $newFilters);
	}

	/**
	 * Used to populate form drop down selects
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups()
	{
		$this->set('bookingStates', $this->LateBooking->BookingState->find('list'));

		return;
	}

	public function admin_edit($id = null)
	{
		parent::admin_edit($id);
		// Make sure we have the purchase data from the database available in the view. Otherwise,
		// on form submit not all the data is available in the request data.
		if ($this->request->is(['post', 'put'])) {
			$booking = $this->LateBooking->readForEdit($id);
		} else {
			$booking = $this->request->data;
		}
		$this->set(compact('booking'));

		if ((int) $this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('LateBooking', $id));
		}

		// We're overriding the scaffolding template as we want to customise
		// the tabs.
		$this->view = 'BuzzLateBookings.admin_form';

		return;
	}

	protected function _adminIndexToolbar($id = null)
	{
		return [];
	}

	protected function _adminFormToolbar($id = null)
	{
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);

		return $actions;
	}

}
