<?php

App::uses('BuzzFacilitiesAppModel', 'BuzzFacilities.Model');

class Facility extends BuzzFacilitiesAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzAdverts.AdvertHost',
		'BuzzTranslate.Translatable',
		'EvCore.Tabbed' => array(
			'max_tabs' => 3
		),
		'MetaData.Meta',
	);

/**
 * @var array Belongs to associations
 */
	public $belongsTo = array(
		'Theme' => array(
			'className' => 'BuzzTheme.Theme'
		)
	);

/**
 * @var array Has many associations
 */
	public $hasMany = array(
		'FacilityPackage' => array(
			'className' => 'BuzzFacilities.FacilityPackage'
		)
	);

/**
 * Image slots
 *
 * @var array
 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		],
		'listing' => [
			'slots' => 1,
			'fields' => []
		],
	];

/**
 * Define the number of associated files/documents.
 *
 * @var int|array
 */
	public $documentSlots = -1;

/**
 * @var array Validation rules
 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'validate.maxLength150'
			)
		),
		'video_url' => array(
			'required' => array(
				'rule' => array('url', true),
				'message' => 'validate.video_url',
				'allowEmpty' => true
			)
		),
		'sequence' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			),
			'max' => array(
				'rule' => array('naturalNumber', true),
				'message' => 'validate.integer'
			)
		),
		'theme_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			)
		)
	);

/**
 * Constructor
 */
	public function __construct($id = false, $table = null, $ds = null) {
		// If we're using EvTinyMceImage we want to add extra image slots.
		if (CakePlugin::loaded('EvTinyMceImage') === true) {
			$this->imageSlots['other'] = [
				'slots' => -1,
				'fields' => false
			];
		}
		parent::__construct($id, $table, $ds);

		if (Configure::check('BuzzFacilities.routes.facilities') === true) {
			$this->addBehaviour('Routable.Routable', Configure::read('BuzzFacilities.routes.facilities'));
		}

		// If BuzzGallery loaded associate facilities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}
	}

/**
 * Get all visible listed facilities
 *
 * @return array
 */
	public function getListed() {
		$params = [
			'contain' => [
				'ListingImage'
			],
			'conditions' => [
				'Facility.listed' => true,
				'Facility.is_active' => true,
				'Facility.is_removed' => false
			],
			'order' => [
				'Facility.sequence' => 'ASC',
				'Facility.created' => 'ASC'
			]
		];
		return $this->find('all', $params);
	}

/**
 * Read for edit
 *
 * @param int $id
 * @param array $query
 * @return array
 */
	public function readForEdit($id, $query = []) {
		$query['contain']['Advert'] = array(
			'AdvertRelation',
			'AdvertTemplate',
			'Image',
			'order' => array('AdvertRelation.sequence' => 'ASC')
		);
		return parent::readForEdit($id, $query);
	}

/**
 * Read for view
 *
 * @param int $id Facility ID
 * @param array $query Additional query parameters
 * @return array
 */
	public function readForView($id, $query = []) {
		$query['contain']['FacilityPackage'] = array(
			'conditions' => array(
				'FacilityPackage.is_active' => true,
				'FacilityPackage.is_removed' => false
			),
			'order' => array(
				'FacilityPackage.sequence' => 'ASC',
				'FacilityPackage.created' => 'ASC'
			)
		);
		$data = parent::readForView($id, $query);

		if (empty($data)) {
			return [];
		}

		// Remove hidden adverts, these shouldn't show.
		if (!empty($data['Advert'])) {
			foreach ($data['Advert'] as $key => $advert) {
				if ($advert['is_active'] === false) {
					unset($data['Advert'][$key]);
				}
			}
		}

		if (Configure::check('BuzzAdverts.facilities_id') === true) {

			// Remove self from facilities for advert.
			foreach ($data['Advert'] as &$advert) {
				if ((int)$advert['id'] === Configure::read('BuzzAdverts.facilities_id')) {
					foreach ($advert['vars']['facilities'] as $key => $facility) {
						if ($facility['Facility']['id'] === $id) {
							unset($advert['vars']['facilities'][$key]);
						}
					}
				}
			}

		}

		return $data;
	}

}
