<?php

App::uses('BuzzFacilitiesAppController', 'BuzzFacilities.Controller');

class FacilitiesController extends BuzzFacilitiesAppController {

	public $components = array(
		'BuzzAdverts.Adverts'
	);

	public function __construct($request = null, $response = null) {
		$this->adminActions[] = 'admin_adverts';
		// If we're using EvTinyMceImage we want to load the plugin's component.
		if (CakePlugin::loaded('EvTinyMceImage') === true) {
			$this->components['EvTinyMceImage.TinyMCE'] = [
				'model_alias' => 'Facility'
			];
		}
		$this->components['BuzzTranslate.Translation'] = array(
			'fields' => array(
				'name',
				'content' => array(
					'type' => 'html',
					'label' => __d('buzz_facilities', 'Content')
				),
				'secondary_content' => array(
					'type' => 'html',
					'label' => __d('buzz_facilities', 'Secondary Content')
				)
			),
			'meta' => true
		);

		parent::__construct($request, $response);
	}

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view'
		));
	}

	/**
	 * Facilities landing page
	 *
	 * @return void
	 */
	public function index() {
		$facilities = $this->Facility->getListed();

		if (count($facilities) === 1 && !empty($facilities[0]['Facility']['id'])) {
			return $this->redirect([
				'controller' => 'facilities',
				'action' => 'view',
				$facilities[0]['Facility']['id']
			]);
		}

		if (Configure::check('BuzzFacilities.facilities_page_id') === true) {
			$data = $this->assignPage(Configure::read('BuzzFacilities.facilities_page_id'));
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($this->viewVars['data']['Page']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->Page->Gallery->readForView($this->viewVars['data']['Page']['gallery_id'])
			);
		}

		$this->Meta->canonical(['action' => 'index', 'plugin' => 'buzz_facilities']);
		$this->set(compact('facilities'));
		$this->view = 'BuzzFacilities./Facilities/index';

		return;
	}

	/**
	 * Facility page
	 *
	 * @param int $id Facility ID
	 * @return void
	 */
	public function view($id) {
		$data = $this->Facility->readForView($id);

		if (!$data) {
			throw new NotFoundException();
		}

		if ($data['Facility']['is_active'] === false) {
			$this->Session->setFlash(__d('buzz_facilities', 'Preview: This page has not been published yet'), 'flash_info');
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($data['Facility']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->Facility->Gallery->readForView($data['Facility']['gallery_id'])
			);
		}

		$facilities = $this->Facility->getListed();

		$this->Meta->set($data);
		$this->Meta->canonical(['action' => 'view', $id]);
		$this->set(compact('data', 'facilities'));
		$this->view = 'BuzzFacilities./Facilities/view';

		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->Adverts->setupAdvertAdminFields();
		$this->Translation->adminTranslationFormFields();
		return;
	}

	/**
	 * Admin form fields
	 *
	 * @return array
	 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$fields['Facility.content']['type'] = 'html';
		$fields['Facility.secondary_content']['type'] = 'html';

		// Localise labels.
		$fields['Facility.name']['label'] = __d('buzz_facilities', 'Name');
		$fields['Facility.content']['label'] = __d('buzz_facilities', 'Content');
		$fields['Facility.secondary_content']['label'] = __d('buzz_facilities', 'Secondary Content');
		$fields['Facility.theme_id']['label'] = __d('buzz_facilities', 'Theme');
		if (!empty($fields['Facility.gallery_id'])) {
			$fields['Facility.gallery_id']['label'] = __d('buzz_facilities', 'Gallery');
		}
		$fields['Facility.video_url']['label'] = __d('buzz_facilities', 'Video URL');

		return $fields;
	}

	/**
	 * Populate admin lookups
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$Page = $this->{$this->modelClass};

		$this->set('themes', $Page->Theme->find('list'));

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->set('galleries', $Page->Gallery->find('list'));
		}

		return;
	}

	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		if (!empty($id)) {
			$toolbar['Adverts'] = [
				'url' => ['action' => 'adverts', $id],
				'icon' => 'bars'
			];
			$toolbar['Preview'] = [
				'url' => ['action' => 'view', $id, 'admin' => false],
				'icon' => 'search',
				'target' => '_blank'
			];
		}

		return $toolbar;
	}

}
