<?php
App::uses('BuzzExtendVoucherAppModel', 'BuzzExtendVoucher.Model');
App::uses('VoucherExtensionState', 'BuzzExtendVoucher.Model');

class VoucherExtension extends BuzzExtendVoucherAppModel {

/**
 * Default order
 * @var array
 */
	public $order = ['VoucherExtension.created' => 'DESC'];

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = [
		'BuzzPurchase.Payable',
	];

/**
 * Belongs to relations
 *
 * @var array
 */
	public $belongsTo = [
		'CustomerAddress' => [
			'className' => 'BuzzCustomers.CustomerAddress'
		],
		'VoucherExtensionState' => [
			'className' => 'BuzzExtendVoucher.VoucherExtensionState',
		],
	];

/**
 * Read for edit
 * @param int $id ID
 * @param array $query Query parameters
 * @return array
 */
	public function readForEdit($id, $query = []) {
		$query['contain'] = [
			'CustomerAddress' => [
				'Country',
				'UsState',
			],
			'VoucherExtensionState',
		];

		return parent::readForEdit($id, $query);
	}

/**
 * Get a completed purchase to display to the customer.
 *
 * @param int $id Purchase ID
 * @return array
 */
	public function getPurchase($id) {
		$params = [
			'contain' => [
				'CustomerAddress',
			],
			'conditions' => [
				'VoucherExtension.voucher_extension_state_id' => [
					VoucherExtensionState::COMPLETE,
					VoucherExtensionState::API_FAILED
				]
			]
		];
		return $this->readForView($id, $params);
	}

/**
 * Check that expiry date is within the grace period
 *
 * @param string $expiryDate Expiry date
 * @return bool True if in grace period, false if not
 */
	public function checkGracePeriod($expiryDate) {
		$gracePeriod = Configure::read('SiteSetting.voucher_extend_grace_period');
		if (! empty($gracePeriod)) {
			$date = new DateTime($expiryDate);
			$date->add(new DateInterval('P' . (int)$gracePeriod . 'D'));

			return (new DateTime)->format('Y-m-d') <= $date->format('Y-m-d');
		}

		return true;
	}

/**
 * Check voucher and return original cost of purchased voucher
 *
 * @param string $voucherCode Voucher code
 * @param string $expiryDate  Expiry date
 * @return float|bool The cost of the voucher or false if not valid
 */
	public function checkVoucher($voucherCode, $expiryDate) {
		return ClassRegistry::init('BuzzExtendVoucher.VoucherExtensionApi')->checkVoucherCost(
			$voucherCode,
			$expiryDate
		);
	}

/**
 * Get the basket
 *
 * @return array|bool Basket data or false if not found
 */
	public function getBasket() {
		$basketId = $this->getBasketId();

		return empty($basketId) ? false : $this->find('first', [
			'contain' => [
				'CustomerAddress' => [
					'Country',
					'UsState',
				],
			],
			'conditions' => [
				'VoucherExtension.id' => $basketId,
				'VoucherExtension.is_removed' => false,
				'voucher_extension_state_id' => VoucherExtensionState::UNPAID,
			]
		]);
	}

/**
 * Add to basket
 * @param string $voucherCode Voucher code
 * @param string $expiryDate  Original expiry date
 * @param float  $voucherCost Original voucher cost
 * @return bool True if successful, false if fails
 */
	public function addToBasket($voucherCode, $expiryDate, $voucherCost) {
		$redemptionPeriod = Configure::read('SiteSetting.voucher_extend_redemption_period');
		$newExpiryDate = (new DateTime($expiryDate))
			->add(new DateInterval('P' . (int)$redemptionPeriod . 'M'))
			->format('Y-m-d');
		$basket = [
			'id' => $this->getBasketId(),
			'voucher_code' => $voucherCode,
			'original_expiry_date' => $expiryDate,
			'new_expiry_date' => $newExpiryDate,
			'total_cost' => $this->calculateExtensionCost($voucherCost),
			'voucher_extension_state_id' => VoucherExtensionState::UNPAID,
		];
		if ($this->save($basket)) {
			$this->setBasketId($this->id);

			return true;
		}

		return false;
	}

/**
 * Calculate the cost of the extension
 *
 * @param float $voucherCost Original voucher cost
 * @return float Cost of extension
 */
	public function calculateExtensionCost($voucherCost) {
		$chargeType = Configure::read('SiteSetting.voucher_extend_charge_type');
		$chargeAmount = Configure::read('SiteSetting.voucher_extend_charge_amount');
		switch ($chargeType) {
			case 'Percentage':
				$newCost = $voucherCost * $chargeAmount / 100;
				break;
			case 'Fixed Amount':
				$newCost = $chargeAmount;
				break;
			case 'Cost':
			default:
				$newCost = $voucherCost;
		}

		return $newCost;
	}

/**
 * Add the customer billing details to the purchase
 *
 * @param array $customer         Customer data
 * @param bool  $newsletterSignup Flag for newsletter signup
 * @return bool
 */
	public function addBillingDetails(array $customer, $newsletterSignup = false) {
		$data = array(
			'VoucherExtension' => array(
				'id' => $this->getBasketId(),
				'newsletter_opt_in' => $newsletterSignup
			),
			'CustomerAddress' => $customer
		);

		return $this->saveAssociated($data) !== false;
	}

	public function completePurchase($paid) {
		$id = $this->getBasketId();
		$basket = $this->getBasket($id);
		$response = true;
		if ((int)$basket['VoucherExtension']['voucher_extension_state_id'] === VoucherExtensionState::UNPAID) {
			if ($paid === true) {
				$result = ClassRegistry::init('BuzzExtendVoucher.VoucherExtensionApi')->extendVoucher(
					$this->name,
					$id,
					$basket['VoucherExtension']['voucher_code'],
					$basket['VoucherExtension']['original_expiry_date'],
					$basket['VoucherExtension']['new_expiry_date'],
					$basket['VoucherExtension']['total_cost'],
					$basket['CustomerAddress']['first_name'],
					$basket['CustomerAddress']['last_name'],
					$basket['CustomerAddress']['email'],
					$basket['CustomerAddress']['telephone'],
					$basket['CustomerAddress'],
					$this->getPaymentDetails($id, 'BuzzExtendVoucher.VoucherExtension')
				);
				$data = [
					'id' => $id,
					'sales_ref' => $result,
					'voucher_extension_state_id' => $result ? VoucherExtensionState::COMPLETE : VoucherExtensionState::API_FAILED,
					'completed_date' => gmdate('Y-m-d H:i:s')
				];
			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'voucher_extension_state_id' => VoucherExtensionState::PAYMENT_FAILED,
				);
			}
			$response = $this->save($data);
			// Clear the basket session.
			$this->clearBasketId();
			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.VoucherExtension.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}

		if (! empty($basket['VoucherExtension']['newsletter_opt_in']) && CakePlugin::loaded('BuzzSubscribe')) {
			ClassRegistry::init('BuzzSubscribe.Subscriber')->addSubscriber(
				$basket['CustomerAddress']['email'],
				$basket['CustomerAddress']['first_name'],
				$basket['CustomerAddress']['last_name'],
				'Voucher Extension'
			);
		}

		return $response;
	}

}
