<?php
App::uses('BuzzExtendVoucherAppController', 'BuzzExtendVoucher.Controller');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class VoucherExtensionsController extends BuzzExtendVoucherAppController implements PurchasableInterface {

/**
 * List of admin actions the controller supports
 *
 * @var array
 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_email_confirmation'
	);

/**
 * Components
 *
 * @var array
 */
	public $components = [
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

/**
 * Before filter
 *
 * @return void
 */
	public function beforeFilter() {
		parent::beforeFilter();
		$this->Auth->allow([
			'index',
			'basket',
			'checkout',
			'payment_callback',
			'confirmation',
		]);

		return;
	}

/**
 * Helpers
 */
	public $helpers = array(
		'Address.Address',
		'BuzzPurchase.Purchase'
	);

/**
 * Extend voucher page
 *
 * @return void
 */
	public function index() {
		if ($this->request->is('post')) {
			if ($this->VoucherExtension->checkGracePeriod($this->request->data['VoucherExtension']['expiry_date'])) {
				$cost = $this->VoucherExtension->checkVoucher(
					$this->request->data['VoucherExtension']['voucher_code'],
					$this->request->data['VoucherExtension']['expiry_date']
				);
				if ($cost !== false) {
					$this->VoucherExtension->addToBasket(
						$this->request->data['VoucherExtension']['voucher_code'],
						$this->request->data['VoucherExtension']['expiry_date'],
						$cost
					);

					return $this->redirect(['action' => 'basket']);
				} else {
					$this->Session->setFlash(
						__d('buzz_extend_voucher', 'Please try again'),
						'flash_fail'
					);
				}
			} else {
				$this->Session->setFlash(
					__d('buzz_extend_voucher', 'Sorry, your voucher cannot be extended'),
					'flash_fail'
				);
			}
		}
		$this->assignPage('Extend Your Gift Voucher Step 1');

		return;
	}

/**
 * Basket step
 *
 * @return void
 */
	public function basket() {
		$VoucherExtension = $this->{$this->modelClass};
		$basket = $VoucherExtension->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_extend_voucher', 'Your session has expired'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'index']);
		} elseif ($this->request->is('post')) {
			return $this->redirect(['action' => 'checkout']);
		} else {
			$this->set(compact('basket'));
		}

		return;
	}

/**
 * Checkout and pay step
 *
 * @return void
 */
	public function checkout() {
		$VoucherExtension = $this->{$this->modelClass};
		$basket = $VoucherExtension->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_extend_voucher', 'Your session has expired'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'index']);
		} elseif ($this->request->is('post')) {
			$result = $VoucherExtension->addBillingDetails($this->request->data['CustomerAddress']);
			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();
			if ($result === true && $paymentValidates === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_extend_voucher', 'Please correct the errors below'),
					'flash_fail'
				);
			}
		}
		$this->loadModel('BuzzConditions.Condition');
		$conditions = $this->Condition->getConditions('VoucherExtension');
		$this->set(compact('basket', 'conditions'));
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();

		return;
	}

/**
 * Populate lookups for address fields.
 *
 * @return void
 */
	protected function _populateLookups() {
		$VoucherExtension = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$usStates = $VoucherExtension->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));

		return;
	}

/**
 * Processes the payment and triggers the completion of the payment on success
 *
 * @return void
 */
	protected function _pay() {
		$VoucherExtension = $this->{$this->modelClass};
		$basketId = $VoucherExtension->getBasketId();
		$basket = $VoucherExtension->getBasket();

		if (Configure::read('BuzzPurchase.onsite') === true) {
			$payment = $this->request->data['Payment'];

			// Split name into first and second names for payment gateways.
			$fullName = trim($payment['card_name']);
			$splitName = explode(' ', $fullName);
			$payment['last_name'] = array_pop($splitName);
			$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
		} else {
			$payment = [];
		}

		$transaction = $this->Transactions->takePayment(
			// Return URLs
			[
				'return' => Router::url(['action' => 'payment_callback', $basketId], true),
				'cancel' => Router::url(['action' => 'checkout'], true)
			],
			// Calling record
			[
				'model' => 'BuzzBookings.Booking',
				'model_id' => $basket['VoucherExtension']['id']
			],
			// Amount to be paid
			$basket['VoucherExtension']['total_cost'],
			// Items
			[],
			// Extras
			[
				'language' => Configure::read('Config.language'),
				'card' => $payment,
				'address' => $basket['CustomerAddress'],
				'user' => array('email' => $basket['CustomerAddress']['email']),
				'description' => __d(
					'buzz_extend_voucher',
					'%s Voucher Extension %d',
					[Configure::read('SiteSetting.site_title'), $basket['VoucherExtension']['id']]
				)
			],
			0,
			Configure::read('Transactions.gateway')
		);

		if (!empty($transaction['result'])) {
			// Payment taken successfully, complete purchase.
			return $this->_completePurchase($basketId);
		} else {
			$this->Session->setFlash(
				__d(
					'buzz_extend_voucher',
					'There was a problem processing your payment, please check your details and try again'
				),
				'flash_fail'
			);
		}

		return;
	}

/**
 * Complete the purchase (after payment received)
 *
 * @param int $basketId Basket ID
 * @return void Redirects to confirmation page
 */
	protected function _completePurchase($basketId) {
		$VoucherExtension = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$VoucherExtension->setBasketId($basketId);
		$VoucherExtension->completePurchase(true);

		// The hash *must* be set after completing the purchase as
		// completePurchase() will delete the session.
		$this->Session->write('VoucherExtension.hash', $VoucherExtension->hashBasketId($basketId));

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

/**
 * Confirmation page
 *
 * @param int $id Basket ID
 * @return void
 */
	public function confirmation($id) {
		$VoucherExtension = $this->{$this->modelClass};
		if (
			$VoucherExtension->hashBasketId($id) === $this->Session->read('VoucherExtension.hash')
			|| Configure::read('app.enviroment') === 'DEVELOPMENT'
		) {
			$this->Session->delete('VoucherExtension.hash');
			$basket = $VoucherExtension->getPurchase($id);
			$this->set(compact('basket'));
			$this->view = 'BuzzExtendVoucher.confirmation';
		} else {
			throw new ForbiddenException();
		}
		$this->assignPage('Extend Your Gift Voucher Confirmation');

		return;
	}

/**
 * Generic payment gateway callback.
 *
 * @param int $basketId Basket ID
 * @return void
 */
	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completePurchase($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_extend_voucher', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'checkout']);
		}
	}

/**
 * Admin index paginate
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();
		$paginate = [
			'conditions' => $conditions,
			'contain' => [
				'CustomerAddress',
				'VoucherExtensionState'
			]
		];

		return $paginate;
	}

/**
 * Admin index columns
 *
 * @return array
 */
	protected function _adminIndexColumns() {
		$VoucherExtension = $this->{$this->modelClass};

		$columns = parent::_adminIndexColumns();

		// Remove created/modified columns.
		unset($columns['VoucherExtension.created']);
		unset($columns['VoucherExtension.modified']);

		$columns[$VoucherExtension->alias . '.total_cost']['type'] = 'currency';

		$newColumns = array(
			'CustomerAddress.full_name' => array(
				'label' => 'Customer',
				'type' => 'string'
			),
			'VoucherExtensionState.name' => array(
				'label' => 'Status',
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'VoucherExtension.sales_ref', $newColumns);
	}

/**
 * Admin columns whitelist
 *
 * @return array
 */
	protected function _adminIndexColumnsWhitelist() {
		$VoucherExtension = $this->{$this->modelClass};

		$whitelist = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $VoucherExtension->alias . '.sales_ref';
		$whitelist[] = $VoucherExtension->alias . '.total_cost';
		$whitelist[] = $VoucherExtension->alias . '.completed_date';

		return $whitelist;
	}

/**
 * Filters
 *
 * @return array
 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['VoucherExtension.name']);
		unset($filters['VoucherExtension.created']);
		unset($filters['VoucherExtension.modified']);

		$newFilters = array(
			'VoucherExtension.sales_ref' => array(
				'label' => 'Sales Ref',
				'type' => 'string',
				'compare' => array('VoucherExtension.sales_ref' => "%s")
			),
			'CustomerAddress.full_name' => array(
				'label' => 'Customer',
				'type' => 'string',
				'compare' => array('CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%")
			),
			'VoucherExtension.voucher_extension_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => VoucherExtensionState::COMPLETE,
				'compare' => array('VoucherExtension.voucher_extension_state_id' => '%s')
			),
			'VoucherExtension.completed_date' => array(
				'label' => 'Completed Date',
				'type' => 'date',
				'compare' => array('VoucherExtension.completed_date' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'VoucherExtension.sales_ref', $newFilters);
	}

/**
 * Used to populate form drop down selects
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$this->set('voucherExtensionStates', $this->VoucherExtension->VoucherExtensionState->find('list'));

		return;
	}

/**
 * Admin index toolbar
 * @param int $id ID
 * @return array
 */
	protected function _adminIndexToolbar($id = null) {
		return [];
	}

/**
 * Admin form toolbar
 * @param int $id ID
 * @return array
 */
	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);

		$actions['Resend Email'] = array(
			'url' => array('action' => 'email_confirmation', $id),
			'icon' => 'envelope'
		);

		return $actions;
	}

/**
 * Admin edit
 * @param int $id ID
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$basket = $this->VoucherExtension->readForEdit($id);
		$this->set(compact('basket'));

		if ((int)$this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('VoucherExtension', $id));
		}

		// We're overriding the scaffolding template as we want to customise
		// the tabs.
		$this->view = 'BuzzExtendVoucher.admin_form';

		return;
	}

/**
 * Re-sends the confirmation email to the customer and redirects to the admin edit form.
 *
 * @param int $id Purchase ID
 */
	public function admin_email_confirmation($id) {
		$Booking = $this->{$this->modelClass}->alias;

		$Event = new CakeEvent('Model.VoucherExtension.completed', $this, ['id' => $id]);
		$this->getEventManager()->dispatch($Event);

		$this->Session->setFlash(
			array(
				'title' => InflectorExt::humanize($this->$Booking->displayName) . ' confirmation sent',
				'description' => 'The confirmation email has been resent to the customer!'
			),
			'flash_success'
		);

		return $this->redirect(['action' => 'edit', $id]);
	}
}
