<?php

App::uses('BuzzEmailsAppModel', 'BuzzEmails.Model');
App::uses('TranslateUtil', 'BuzzTranslate.Lib');

class Email extends BuzzEmailsAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzTranslate.Translatable'
	);

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'EmailGroup' => array(
			'className' => 'BuzzEmails.EmailGroup'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'subject' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 100),
				'message' => 'No more than 100 characters long'
			)
		),
		'content' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'subject' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 100),
				'message' => 'No more than 100 characters long'
			)
		),
		'from_name' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'from_email' => array(
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address',
				'allowEmpty' => true
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			)
		)
	);

	/**
	 * After validate, check that tokens have been set in content.
	 *
	 * This must come after beforeValidate and before the beforeSave callback to
	 * ensure all validation rules are checked to display to the user.
	 *
	 * @return void
	 */
	public function afterValidate() {
		parent::afterValidate();
		if (!empty($this->data[$this->alias]['content'])) {
			$this->validateRequiredTokens($this->data[$this->alias]['content'], 'content');
		}
		return;
	}

	/**
	 * Check that all the required tokens exist in the content
	 *
	 * @param string $check
	 * @param string $field
	 * @return bool
	 */
	public function validateRequiredTokens($check, $field) {
		if (!empty($this->id)) {

			$missingTokens = [];

			if (array_key_exists('required_tokens', $this->data)) {

			} else {
				$requiredTokens = json_decode($this->field('required_tokens'));
				if (!empty($requiredTokens) && is_array($requiredTokens)) {
					foreach ($requiredTokens as $token) {
						if (strpos($check, '{' . $token . '}') === false) {
							$missingTokens[] = '{' . $token . '}';
						}
					}
				}

			}

			// Flag the missing tokens as a validation error.
			if (!empty($missingTokens)) {
				$this->invalidate(
					$field,
					__d(
						'buzz_emails',
						'The following tokens are missing: %s',
						[implode(', ', $missingTokens)]
					)
				);
				return false;
			}

		}

		return true;
	}

	/**
	 * Read for edit
	 *
	 * @param int $id
	 * @param array $params
	 * @return array
	 */
	public function readForEdit($id, $params = []) {
		$data = parent::readForEdit($id, $params);

		if (!empty($data[$this->alias]['required_tokens'])) {
			$requiredTokens = json_decode($data[$this->alias]['required_tokens']);
			$requiredTokens = array_map(
				function ($val) {
					return '{' . $val . '}';
				},
				$requiredTokens
			);
			$data[$this->alias]['required_tokens'] = implode(', ', $requiredTokens);
		}

		if (!empty($data[$this->alias]['optional_tokens'])) {
			$optionalTokens = json_decode($data[$this->alias]['optional_tokens']);
			$optionalTokens = array_map(
				function ($val) {
					return '{' . $val . '}';
				},
				$optionalTokens
			);
			$data[$this->alias]['optional_tokens'] = implode(', ', $optionalTokens);
		}

		return $data;
	}

	/**
	 * Generate email content replacing tokens with passed data.
	 *
	 * @param int|string $id Email record ID or system name
	 * @param array $data Data to replace tokens
	 * @return array|bool
	 */
	public function generateEmailData($id, array $data = []) {
		$result = is_int($id) ? $this->findById($id) : $this->findBySystemName($id);
		if (empty($result)) {
			return false;
		}

		// Translate the email content before we replace the tokens.
		$result['Email']['subject'] = TranslateUtil::translate($result, 'Email.subject');
		$result['Email']['content'] = TranslateUtil::translate($result, 'Email.content');

		// Replace required tokens.
		$result['Email']['subject'] = $this->_replaceTokens(
			$result['Email']['subject'],
			$result['Email']['required_tokens'],
			$data
		);

		// Replace optional tokens.
		$result['Email']['subject'] = $this->_replaceTokens(
			$result['Email']['subject'],
			$result['Email']['optional_tokens'],
			$data
		);

		// Remove paragraph tags from block-style tokens caused by WYSIWYG editor.
		$pattern = '#<p>(\\s*){==(\w+)==}(\\s*)</p>#isU';
		$result['Email']['content'] = preg_replace($pattern, '{$2}', $result['Email']['content']);

		// Replace required tokens.
		$result['Email']['content'] = $this->_replaceTokens(
			$result['Email']['content'],
			$result['Email']['required_tokens'],
			$data
		);
		unset($result['Email']['required_tokens']);

		// Replace optional tokens.
		$result['Email']['content'] = $this->_replaceTokens(
			$result['Email']['content'],
			$result['Email']['optional_tokens'],
			$data
		);
		unset($result['Email']['optional_tokens']);

		// Configure where email will be sent from.
		$fromName = $result['Email']['from_name'] ?: Configure::read('SiteSetting.site_title');
		$fromEmail = $result['Email']['from_email'] ?: Configure::read('SiteSetting.admin_email');
		$result['Email']['from'] = [$fromEmail => $fromName];
		unset($result['Email']['from_name']);
		unset($result['Email']['from_email']);

		$result['Email']['to'] = null;
		if ($result['Email']['override_to'] === true) {
			$toName = $result['Email']['to_name'] ?: Configure::read('SiteSetting.site_title');
			$toEmail = $result['Email']['to_email'] ?: Configure::read('SiteSetting.admin_email');
			$result['Email']['to'] = [$toEmail => $toName];
		}
		unset($result['Email']['to_name']);
		unset($result['Email']['to_email']);

		return $result;
	}

	/**
	 * Convert an HTML string to plain text.
	 */
	public function textContent($html) {
		// Convert <br> to line breaks.
		$text = preg_replace('/<br(\s+)?\/?>/i', "\n", $html);
		// Convert </p> to line breaks.
		$text = preg_replace('/<\/p>/i', "\n\n", $html);
		// Remove HTML markup.
		$text = trim(strip_tags($text));

		return $text;
	}

	/**
	 * Replaces tokens in content.
	 *
	 * Tokens are in the format of {key}. Use {==key==} to represent a token that
	 * needs to remove wrapping markup (for example a basket representation).
	 *
	 * @param string $content Content containing tokens
	 * @param string $tokensJson JSON encoded tokens
	 * @param array $data Data to replace tokens
	 * @return string
	 */
	protected function _replaceTokens($content, $tokensJson, array $data = []) {
		$tokens = json_decode($tokensJson);

		if (!empty($tokens) && is_array($tokens)) {

			$tokens = array_map(
				function ($val) {
					return str_replace('==', '', $val);
				},
				$tokens
			);

			// Replace each token with the passed data.
			foreach ($tokens as $token) {
				if (array_key_exists($token, $data) === true) {
					$content = str_replace('{' . $token . '}', $data[$token], $content);
				}
			}

		}

		return $content;
	}

	/**
	 * Convinience method for adding an email to the queue.
	 *
	 * @param string $subject
	 * @param string $content
	 * @param array $to
	 * @param array $from
	 * @param string $template
	 * @param array $cc Carbon copy emails
	 * @param array $bcc Blind carbon copy emails
	 * @param array $replyTo Reply-to email address
	 * @return bool
	 */
	public function queueEmail($subject, $content, array $to, array $from, $template = null, $cc = null, $bcc = null, $replyTo = null) {
		return (bool)ClassRegistry::init('Queue.QueuedTask')->createJob(
			'BuzzEmail',
			[
				'to' => $to,
				'cc' => $cc,
				'bcc' => $bcc,
				'from' => $from,
				'replyTo' => $replyTo,
				'subject' => $subject,
				'content' => $content,
				'template' => $template,
				'domain' => Configure::read('app.siteDomain')
			]
		);
	}

}
