<?php

App::uses('BuzzSourceAppModel', 'BuzzSource.Model');

class DiaryApi extends BuzzSourceAppModel {

	public function __construct($id = false, $table = null, $ds = null) {
		// Override the default host if we're using BuzzSites.
		if (CakePlugin::loaded('BuzzSites')) {
			$Site = ClassRegistry::init('BuzzSites.Site');
			ConnectionManager::create('diaryApi', $Site->getActiveSiteConfig());
			$this->useDbConfig = 'diaryApi';
		} else {
			die('Unable to find BuzzSites plugin. This is a requirement for all sites.');
		}

		$this->dataSource = ConnectionManager::getDataSource('diaryApi');

		parent::__construct($id, $table, $ds);
	}

/**
 * Create a user account.
 *
 * @param int $siteId Location/Site ID
 * @param string $name
 * @param string $email
 * @param string $telephone
 * @param array $address Address generated by CustomerAddress::generateAddressArray()
 * @return array|bool
 */
	public function createUserAccount($siteId, $name, $email, $telephone, $address) {
		$result = $this->find('all', array(
			'method' => 'CreateUserAccount',
			'conditions' => array(
				'siteId' => $siteId,
				'name' => $name,
				'email' => $email,
				'contacttel' => $telephone,
				'addressline1' => $address['address_line_1'],
				'addressline2' => $address['address_line_3'],
				'country' => $address['address_line_4'],
				'postalcode' => $address['postcode']
			)
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processCreateUserAccount($result['DiaryApi']['d']) : false;
	}

	protected function _processCreateUserAccount($data) {
		return $data;
	}

/**
 * Update a user account.
 *
 * @param int $userId API Diary User ID
 * @param string $name
 * @param string $email
 * @param string $telephone
 * @param array $address Address generated by CustomerAddress::generateAddressArray()
 * @return array|bool
 */
	public function updateUserAccount($userId, $name, $email, $telephone, $address) {
		$result = $this->find('all', array(
			'method' => 'UpdateUserAccount',
			'conditions' => array(
				'userId' => $userId,
				'name' => $name,
				'email' => $email,
				'contacttel' => $telephone,
				'addressline1' => $address['address_line_1'],
				'addressline2' => $address['address_line_3'],
				'country' => $address['address_line_4'],
				'postalcode' => $address['postcode']
			)
		));

		return !empty($result['DiaryApi']['d']) && strtolower($result['DiaryApi']['d']) === 'ok';
	}

/**
 * Get a user account.
 *
 * @param int $userId API Diary User ID
 * @param int $siteId Location/Site ID
 * @return array|bool
 */
	public function getUserAccount($userId, $siteId) {
		$result = $this->find('all', array(
			'method' => 'GetUserAccount',
			'conditions' => array(
				'userId' => $userId,
				'siteId' => $siteId
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccount($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccount($data) {
		return array(
			'amount_due' => $data['AmountDue'],
			'balance' => $data['Balance'],
			'total_booked' => $data['TotalTimeBooked'],
			'total_cancelled' => $data['TotalTimeCancelled'],
			'overdraft' => $data['OverdraftLimit'],
			'total_available' => $data['Balance'] + $data['OverdraftLimit']
		);
	}

/**
 * Get a user account's prices.
 *
 * @param int $userId API Diary User ID
 * @param int $siteId Location/Site ID
 * @return array|bool
 */
	public function getUserAccountPrices($userId, $siteId) {
		$result = $this->find('all', array(
			'method' => 'GetUserAccountPrices',
			'conditions' => array(
				'accountId' => $userId,
				'siteId' => $siteId
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccountPrices($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account prices data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccountPrices($data) {
		$result = [];

		if (isset($data['LastMinutePrice'])) {
			$result = array(
				'last_minute' => $data['LastMinutePrice'],
				'off_peak' => $data['OffPeakPrice'],
				'peak' => $data['PeakPrice']
			);
		}

		return $result;
	}

/**
 * Get a user account's credit history.
 *
 * @param int $userId API Diary User ID
 * @param int $limit Maximum number of results to return
 * @param int $page Page number
 * @return array|bool
 */
	public function getUserAccountCreditHistory($userId, $limit, $page = 0) {
		$result = $this->find('all', array(
			'method' => 'GetUserAccountCreditHistory',
			'conditions' => array(
				'userId' => $userId,
				'limit' => $limit,
				'page' => $page
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccountCreditHistory($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account credit history data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccountCreditHistory($data) {
		$result = [];

		foreach ($data as $row) {
			// Need to extract JSON from the notes field that belongs to us
			// (Client append their own data in here too).
			$extras = preg_match('|^{.*?}|i', urldecode($row['Notes']), $json);
			$extras = !empty($json[0]) ? json_decode($json[0], true) : null;
			$result[] = array(
				'date' => $row['SalesDate'],
				'reference' => $row['Reference'],
				'contact_name' => $row['ContactName'],
				'amount' => $row['Purchased'],
				'balance' => $row['Balance'],
				'extras' => $extras
			);
		}

		return $result;
	}

/**
 * Add credit to the user's account.
 *
 * @param int $userId API Diary User ID
 * @param float $amount
 * @param string $firstName
 * @param string $lastName
 * @param string $email
 * @param string $telephone
 * @param array $address Address generated by CustomerAddress::generateAddressArray()
 * @param array $payment Payment details for the API
 * @param string $payeeReference
 * @return bool|string
 */
	public function addUserAccountCredit($userId, $amount, $firstName, $lastName, $email, $telephone, $address, $payment, $payeeReference = null) {
		$payment = array_merge(
			[
				'card_number' => '',
				'vendorTxCode' => '',
				'vpstxId' => '',
				'status' => 'OK',
				'txauthno' => ''
			],
			$payment
		);
		$result = $this->find('all', array(
			'method' => 'AddUserAccountCredit',
			'conditions' => array(
				'userId' => $userId,
				'amount' => $amount,
				'firstname' => $firstName,
				'lastname' => $lastName,
				'email' => $email,
				'telephone' => $telephone,
				'addressline1' => $address['address_line_1'],
				'town' => $address['address_line_3'],
				'postcode' => $address['postcode'],
				'cardnumber' => $payment['card_number'],
				'vendortxcode' => $payment['vendorTxCode'],
				'vpstxId' => $payment['vpstxId'],
				'status' => $payment['status'],
				'txauthno' => $payment['txauthno'],
				'payeereference' => $payeeReference
			)
		));

		return !empty($result['DiaryApi']['d']) && $result['DiaryApi']['d'] !== 'ERROR' ? $result['DiaryApi']['d'] : false;
	}

/**
 * Get a booking.
 *
 * @param int $bookingId API booking ID
 * @return array|bool
 */
	public function getUserAccountBooking($bookingId) {
		$result = $this->find('all', array(
			'method' => 'GetUserAccountBooking',
			'conditions' => array(
				'bookingId' => $bookingId
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccountBooking($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account booking data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccountBooking($data) {
		return array(
			'id' => $data['BookingId'],
			'date' => $data['BookingDate'],
			'time' => $data['BookingTime'],
			'duration_in_secs' => $data['DurationInSeconds'],
			'duration' => $data['Duration'],
			'value' => !empty($data['Value']) ? $data['Value'] : 0,
			'notes' => !empty($data['Notes']) ? $data['Notes'] : null,
			'status' => $data['Status']
		);
	}

/**
 * Get a user account's bookings.
 *
 * @param int $userId API Diary User ID
 * @param string $fromDate
 * @param string $sort either 'desc' or 'asc'
 * @param int $limit Maximum number of results to return
 * @param int $page Page number
 * @return array|bool
 */
	public function getUserAccountBookings($userId, $fromDate, $sort, $limit, $page = 0) {

		$result = $this->find('all', array(
			'method' => 'GetUserAccountBookings',
			'conditions' => array(
				'userId' => $userId,
				'fromdate' => $fromDate,
				'datesort' => $sort === 'desc' ? 'd' : 'a',
				'limit' => $limit,
				'page' => $page
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccountBookings($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account bookings data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccountBookings($data) {
		$result = [];

		foreach ($data as $row) {
			$result[$row['BookingId']] = array(
				'id' => $row['BookingId'],
				'date' => $row['BookingDate'],
				'time' => $row['BookingTime'],
				'duration_in_secs' => $row['DurationInSeconds'],
				'duration' => $row['Duration'],
				'value' => !empty($row['Value']) ? $row['Value'] : 0,
				'notes' => !empty($row['Notes']) ? $row['Notes'] : null,
				'status' => $row['Status'],
				'siteId' => $row['SiteId']
			);
		}

		return $result;
	}

/**
 * Get a user account's cancellations.
 *
 * @param int $userId API Diary User ID
 * @param int $limit Maximum number of results to return
 * @param int $page Page number
 * @return array|bool
 */
	public function getUserAccountCancellations($userId, $limit, $page = 0) {
		$result = $this->find('all', array(
			'method' => 'GetUserAccountCancellations',
			'conditions' => array(
				'userId' => $userId,
				'limit' => $limit,
				'page' => $page
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processUserAccountCancellations($result['DiaryApi']['d']) : false;
	}

/**
 * Process user account cancellation data from the API
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processUserAccountCancellations($data) {
		$result = [];

		foreach ($data as $row) {
			$result[] = array(
				'id' => $row['BookingId'],
				'date' => $row['BookingDate'],
				'time' => $row['BookingTime'],
				'cancelled' => $row['CancelledOn'],
				'duration_in_secs' => $row['DurationInSeconds'],
				'duration' => $row['Duration'],
				'cancellation_ref' => $row['CancellationReference']
			);
		}

		return $result;
	}

/**
 * Get the available days for bookings in the diary between two dates.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param string $fromDate
 * @param string $toDate
 * @return array|bool
 */
	public function getAvailableDiaryDays($siteId, $userId, $fromDate, $toDate) {
		$result = $this->find('all', array(
			'method' => 'GetAvailableDiaryDays',
			'conditions' => array(
				'siteId' => $siteId,
				'userId' => $userId,
				'fromdate' => $fromDate,
				'todate' => $toDate
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processDiaryDays($result['DiaryApi']['d']) : false;
	}

/**
 * Process the available days data from the API.
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processDiaryDays($data) {
		$result = [];

		foreach ($data as $row) {
			$result[] = array(
				'date' => $row['Date'],
				'is_booked' => $row['IsBookedByUser']
			);
		}

		return $result;
	}

/**
 * Get the available diary slots for a specific date.
 *
 * @param int $siteId API site/location ID
 * @param string $date
 * @return array
 */
	public function getAvailableDiarySlots($siteId, $date) {
		$result = $this->find('all', array(
			'method' => 'GetAvailableDiarySlots',
			'conditions' => array(
				'siteId' => $siteId,
				'date' => $date
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) ? $this->_processDiarySlots($result['DiaryApi']['d']) : false;
	}

/**
 * Process the diary slots data from the API.
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processDiarySlots($data) {
		$templates = [];
		$bookings = [];

		if (!empty($data['Templates'])) {
			foreach ($data['Templates'] as $row) {
				// Get the times from the date strings returned by the API.
				$fromTime = preg_replace('|^\d{2}/\d{2}/\d{4}\s|', '', $row['FromTime']);
				$toTime = preg_replace('|^\d{2}/\d{2}/\d{4}\s|', '', $row['ToTime']);

				$templates[$fromTime] = array(
					'from' => $fromTime,
					'to' => $toTime,
					'peak' => (bool)$row['IsPeak'],
					'is_last_minute' => (bool)$row['IsLastMinute'],
					'seconds_available' => $row['SecondsAvailable'],
					'bookable' => (bool)$row['IsBookable'],
					'description' => !empty($row['TemplateDescription']) ? $row['TemplateDescription'] : null,
					'percentage_used' => !empty($row['PercentageUsed']) ? $row['PercentageUsed'] : 0
				);
			}
		}

		if (!empty($data['Bookings'])) {
			foreach ($data['Bookings'] as $row) {
				// Get the booking time.
				$time = preg_replace('|^\d{2}/\d{2}/\d{4}\s|', '', $row['BookingDate']);

				$bookings[$time][] = array(
					'account_id' => $row['AccountId'],
					'date' => $row['BookingDate'],
					'time' => $time,
					'description' => $row['Description'],
					'duration_in_secs' => $row['DurationInSeconds'],
					'notes' => $row['Notes'],
					'reference' => $row['Reference'],
					'rotation' => $row['Rotation'],
					'status' => $row['Status']
				);
			}
		}

		return ['Template' => $templates, 'Booking' => $bookings];
	}

/**
 * Create a diary booking.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param int $minutes
 * @param string $bookingDate
 * @param bool $allowRotations
 * @param decimal $totalCost
 * @param string $notes
 * @param string $sessionId
 * @param string $ipAddress
 * @param array $params Options
 * @return array|bool
 */
	public function createDiaryBooking($siteId, $userId, $minutes, $bookingDate, $allowRotations, $totalCost, $notes, $sessionId, $ipAddress, $params = []) {
		$result = $this->find('all', array(
			'method' => 'CreateDiaryBooking',
			'conditions' => array(
				'siteId' => $siteId,
				'accountId' => $userId,
				'minutes' => $minutes,
				'bookingdate' => date('Y-m-d\TH:i', $bookingDate),
				'allowRotations' => $allowRotations,
				'amount' => $totalCost,
				'notes' => $notes,
				'sessionId' => $sessionId,
				'ipaddress' => $ipAddress,
				'ispeak' => ! empty($params['is_peak']) ? 'true' : 'false'
			)
		));

		return !empty($result['DiaryApi']['d']['BookingId']) ? $this->_processCreateDiaryBooking($result['DiaryApi']['d']) : false;
	}

/**
 * Process the booking creation response.
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processCreateDiaryBooking($data) {
		$result = array(
			'ref' => $data['BookingId'],
			'date' => $data['BookingDate'],
			'duration' => $data['Duration'],
			'status' => $data['Status']
		);

		return $result;
	}

/**
 * Cancel a diary booking.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param int $bookingId
 * @return array|bool
 */
	public function cancelDiaryBooking($siteId, $userId, $bookingId) {
		$result = $this->find('all', array(
			'method' => 'CancelDiaryBooking',
			'conditions' => array(
				'siteId' => $siteId,
				'userId' => $userId,
				'bookingId' => $bookingId
			)
		));

		return !empty($result['DiaryApi']['d']['Status']) ? $this->_processCancelDiaryBooking($result['DiaryApi']['d']) : false;
	}

/**
 * Process the booking cancellation response.
 *
 * @param array $data Response from API
 * @return array
 */
	protected function _processCancelDiaryBooking($data) {
		$result = array(
			'ref' => !empty($data['CancellationRef']) ? $data['CancellationRef'] : null,
			'status' => $data['Status']
		);

		return $result;
	}

/**
 * Get a user's diary notification status for a specific date.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param string $date
 * @return bool
 */
	public function getDiaryEmailNotificationStatus($siteId, $userId, $date) {
		$result = $this->find('all', array(
			'method' => 'GetUserDiaryEmailNotificationStatus',
			'conditions' => array(
				'siteId' => $siteId,
				'userId' => $userId,
				'notificationdate' => $date
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']) && (string)$result['DiaryApi']['d'] !== 'Not Subscribed';
	}

/**
 * Request a diary notification.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param string $date
 * @return bool
 */
	public function requestDiaryEmailNotificationStatus($siteId, $userId, $date) {
		$result = $this->find('all', array(
			'method' => 'RequestDiaryEmailNotifications',
			'conditions' => array(
				'siteId' => $siteId,
				'userId' => $userId,
				'notificationdate' => $date
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']);
	}

/**
 * Cancel a diary notification.
 *
 * @param int $siteId API site/location ID
 * @param int $userId API Diary User ID
 * @param string $date
 * @return bool
 */
	public function cancelDiaryEmailNotificationStatus($siteId, $userId, $date) {
		$result = $this->find('all', array(
			'method' => 'CancelRequestDiaryEmailNotifications',
			'conditions' => array(
				'siteId' => $siteId,
				'userId' => $userId,
				'notificationdate' => $date
			),
			'debug' => 2
		));

		return !empty($result['DiaryApi']['d']);
	}

}
