<?php

App::uses('BuzzDiaryAccountsAppModel', 'BuzzDiaryAccounts.Model');

class AccountPaymentRequest extends BuzzDiaryAccountsAppModel {

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Account' => array(
			'className' => 'BuzzDiaryAccounts.Account'
		),
		'AccountCredit' => array(
			'className' => 'BuzzDiaryAccounts.AccountCredit'
		)
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 60),
				'message' => 'No more than 60 characters'
			)
		),
		'email_address' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Valid email addresses only'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters'
			)
		),
		'reference' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters',
				'allowEmpty' => true
			),
		),
		'amount' => array(
			'required' => array(
				'rule' => '/^\d+(\.\d{2})?$/',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 8),
				'message' => 'No more than 8 characters'
			)
		)
	);

/**
 * Called during validation operations, before validation.
 *
 * @param array $options Options passed from Model::save().
 * @return bool True if validate operation should continue, false to abort
 */
	public function beforeValidate($options = []) {
		if (! empty($this->data[$this->alias]['email_address'])) {
			$this->data[$this->alias]['email_address'] = trim($this->data[$this->alias]['email_address']);
		}

		return parent::beforeValidate($options);
	}

	public function afterSave($created, $options = []) {
		if ($created === true) {

			// Get the payment request and details of the requester
			$data = $this->find(
				'first', array(
					'contain' => array(
						'Account'
					),
					'conditions' => array(
						'AccountPaymentRequest.id' => $this->id
					)
				)
			);

			$event = new CakeEvent('Model.AccountPaymentRequest.paymentRequestSent', $this, array(
					'data' => $data
				)
			);
			$this->getEventManager()->dispatch($event);

		}
	}

/**
 * Complete a payment request
 *
 * @param int $id Payment request ID
 * @param int $accountCreditId Account credit ID
 * @param bool $paid True if payment successfully made
 * @return bool
 */
	public function completePaymentRequest($id, $accountCreditId, $paid) {
		if ($paid === true) {
			// Attach the account credit to the payment request so that we know
			// its been paid.
			$this->id = $id;
			$this->saveField('account_credit_id', $accountCreditId);
		}

		// Complete the credit.
		$payeeReference = $this->find(
			'first',
			array(
				'fields' => array(
					'AccountPaymentRequest.name',
					'AccountPaymentRequest.email_address',
					'AccountPaymentRequest.reference'
				),
				'conditions' => array(
					'AccountPaymentRequest.id' => $id
				)
			)
		);
		$result = $this->AccountCredit->completeCredit(
			$paid,
			'creditAdded',
			$payeeReference['AccountPaymentRequest']
		);

		return $result;
	}

/**
 * Get basket
 *
 * @param int $id Basket ID
 * @return array
 */
	public function getBasket($id) {
		return $this->AccountCredit->find(
			'first',
			[
				'contain' => ['CustomerAddress.Country'],
				'conditions' => ['AccountCredit.id' => $id],
			]
		);
	}
}
