<?php

App::uses('BuzzDiaryAccountsAppModel', 'BuzzDiaryAccounts.Model');
App::uses('AccountCreditState', 'BuzzDiaryAccounts.Model');

class AccountCredit extends BuzzDiaryAccountsAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = [
		'BuzzPurchase.Payable'
	];

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = [
		'Account' => [
			'className' => 'BuzzDiaryAccounts.Account'
		],
		'AccountCreditState' => [
			'className' => 'BuzzDiaryAccounts.AccountCreditState'
		],
		'CustomerAddress' => [
			'className' => 'BuzzCustomers.CustomerAddress'
		]
	];

	/**
	 * Has one associations
	 *
	 * @var array
	 */
	public $hasOne = [
		'AccountPaymentRequest' => [
			'className' => 'BuzzDiaryAccounts.AccountPaymentRequest'
		]
	];

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'amount' => array(
			'required' => array(
				'rule' => '/^\d+(\.\d{2})?$/',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('comparison', '<', 1000000),
				'message' => 'Value must be less than 1000000.00'
			)
		)
	);

	/**
	 * Read for view
	 *
	 * @param int $id
	 * @param array $params
	 * @return array
	 */
	public function readForView($id, $params = []) {
		$params['contain'] = [
			'Account',
			'AccountPaymentRequest'
		];
		return parent::readForView($id, $params);
	}

	/**
	 * Get the current basket ID from the session
	 *
	 * @return int
	 */
	public function getBasketId() {
		if (empty($this->_basketId)) {
			$this->_basketId = CakeSession::read('AccountCredit.id');
		}
		return $this->_basketId;
	}

	/**
	 * Clear the basket ID from the session
	 *
	 * @return bool
	 */
	public function clearBasketId() {
		$this->_basketId = null;
		CakeSession::delete('AccountCredit.id');
		return true;
	}

	/**
	 * Returns a hashed basket ID
	 *
	 * @param int $id (Optional) basket ID
	 * @return string
	 */
	public function hashBasketId($id = null) {
		$id = $id ?: $this->getBasketId();
		return md5($id);
	}

	/**
	 * Creates a new (unpaid) credit record.
	 *
	 * @param int $accountId
	 * @param array $data
	 * @return bool
	 */
	public function createCredit($accountId, array $data) {
		$data['AccountCredit']['account_id'] = $accountId;
		$data['AccountCredit']['account_credit_state_id'] = AccountCreditState::UNPAID;
		$result = $this->saveAssociated($data) !== false;

		// Set the 'basket' ID.
		if ($result === true) {
			CakeSession::write('AccountCredit.id', $this->id);
		}

		return $result;
	}

	/**
	 * @param bool $paid True if payment successfully made
	 * @return bool
	 */
	public function completeCredit($paid, $event = 'completed', $payeeReference = null) {
		$id = $this->getBasketId();

		$data = $this->find('first', array(
			'contain' => array(
				'Account',
				'CustomerAddress' => array(
					'Country'
				)
			),
			'conditions' => array(
				'AccountCredit.id' => $id
			)
		));

		$response = true;

		if ((int)$data['AccountCredit']['account_credit_state_id'] === AccountCreditState::UNPAID) {
			if ($paid === true) {
				// Create the vouchers
				$result = $this->addCredit(
					$data,
					$this->getPaymentDetails($id, 'BuzzDiaryAccounts.AccountCredit'),
					$payeeReference
				);

				$state = $result !== false ? AccountCreditState::COMPLETE : AccountCreditState::API_FAILED;
				$data = array(
					'id' => $id,
					'sales_ref' => $result,
					'account_credit_state_id' => $state
				);
				$response = $this->save($data);

			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'account_credit_state_id' => AccountCreditState::PAYMENT_FAILED
				);
				$response = $this->save($data);

			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.AccountCredit.' . $event, $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);

		}
		return $response;
	}

	/**
	 * Add credit via the API.
	 *
	 * @param array $data
	 * @param array $payment
	 * @param array $payeeReference
	 * @return string|bool Sales ref or false on API fail
	 */
	public function addCredit(array $data, array $payment, $payeeReference = null) {
		// JSON encode the payee reference data ready for passing to the API.
		if (!empty($payeeReference)) {
			$payeeReference = json_encode($payeeReference);
		}

		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->addUserAccountCredit(
			$data['Account']['api_reference'],
			$data['AccountCredit']['amount'],
			$data['CustomerAddress']['first_name'],
			$data['CustomerAddress']['last_name'],
			$data['CustomerAddress']['email'],
			$data['CustomerAddress']['telephone'],
			CustomerAddress::generateAddressArray($data['CustomerAddress']),
			$payment,
			$payeeReference
		);

		return $response;
	}

}
