<?php

App::uses('BuzzDiaryAccountsAppModel', 'BuzzDiaryAccounts.Model');
App::uses('CustomerAddress', 'BuzzCustomers.Model');

class Account extends BuzzDiaryAccountsAppModel {

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		)
	);

	/**
	 * Virtual fields
	 *
	 * @var array
	 */
	public $virtualFields = array(
		'full_name' => 'CONCAT(Account.first_name, " ", Account.last_name)'
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'first_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'last_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Invalid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'A user with that email address already exists'
			)
		),
		'api_reference' => array(
			'unique' => array(
				'rule' => 'isUnique',
				'allowEmpty' => true
			)
		),
		'password' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 50),
				'message' => 'No more than 50 characters'
			)
		),
		'confirm_password' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'matches' => array(
				'rule' => array('checkMatches', 'password'),
				'message' => 'Passwords do not match'
			)
		)
	);

	/**
	 * Setup validation for password change.
	 *
	 * @return void
	 */
	public function validatePassword() {
		$validate['password'] = $this->validate['password'];
		$validate['confirm_password'] = $this->validate['confirm_password'];

		$this->validate = $validate;

		return;
	}

	/**
	 * Before save
	 *
	 * @param array $options
	 * @return bool
	 */
	public function beforeSave($options = []) {
		if (isset($this->data['Account']['password'])) {
			$this->data['Account']['password'] = AuthComponent::password($this->data['Account']['password']);
		}

		return parent::beforeSave($options);
	}

	/**
	 * After save
	 *
	 * @param bool $created
	 * @param array $options
	 * @return void
	 */
	public function afterSave($created, $options = []) {
		if ($created === true) {
			// If we're creating a new account that doesn't have an API
			// reference then we want to attempt to create one via the API.
			if (empty($this->data['Account']['api_reference'])) {
				$account = $this->find('first', array(
					'contain' => array(
						'CustomerAddress' => array(
							'Country',
							'UsState'
						)
					),
					'conditions' => array(
						'Account.id' => $this->id
					)
				));
				$this->_createAccount($account);

				$event = new CakeEvent('Model.Account.registered', $this, ['data' => $account]);
				$this->getEventManager()->dispatch($event);

				if (! empty($account['Account']['newsletter_opt_in']) && CakePlugin::loaded('BuzzSubscribe')) {
					ClassRegistry::init('BuzzSubscribe.Subscriber')->addSubscriber(
						$account['Account']['email'],
						$account['Account']['first_name'],
						$account['Account']['last_name'],
						'Pro Flyer Registration'
					);
				}
			}

		} else {
			// Update the account via the API.
			$account = $this->find('first', array(
				'contain' => array(
					'CustomerAddress' => array(
						'Country',
						'UsState'
					)
				),
				'conditions' => array(
					'Account.id' => $this->id
				)
			));
			$this->_updateAccount($account);

			// Update the auth session for the user (need to check the account
			// has been activated first).
			$authAccount = CakeSession::read('Auth.Account');
			if (
				!empty($authAccount['is_active'])
				&& (int)$authAccount['id'] === (int)$this->id
			) {
				$authAccount = $account['Account'];
				CakeSession::write('Auth.Account', $authAccount);
			}

			if (!empty($this->data['Account']['is_active'])) {
				$account = $this->find('first', array(
					'contain' => array(
						'CustomerAddress' => array(
							'Country',
							'UsState'
						)
					),
					'conditions' => array(
						'Account.id' => $this->id
					)
				));
				$event = new CakeEvent('Model.Account.activated', $this, ['data' => $account]);
				$this->getEventManager()->dispatch($event);
			}

		}

		return parent::afterSave($created, $options);
	}

	public function readForEdit($id, $params = []) {
		$params['contain']['CustomerAddress'] = [
			'Country',
			'UsState'
		];
		return parent::readForEdit($id, $params);
	}

	/**
	 * Activates an account
	 *
	 * @param int $accountId
	 * @param string $activationToken
	 * @return array|bool false if failed to activate otherwise account details
	 */
	public function activateAccount($accountId, $activationToken) {
		$data = $this->find(
			'first',
			array(
				'contain' => array(
					'CustomerAddress' => ['Country']
				),
				'conditions' => array(
					'Account.id' => $accountId,
					'Account.activation_token' => $activationToken,
					'Account.api_reference <>' => null
				)
			)
		);

		if (!empty($data)) {
			$this->id = $accountId;
			if ($this->saveField('is_active', true) !== false) {
				$data['Account']['is_active'] = true;
				return $data;
			}
		}

		return false;
	}

	public function resetPassword($email) {
		$account = $this->find('first', array(
			'conditions' => array(
				'Account.email' => $email,
				'Account.is_active' => true,
				'Account.api_reference <>' => null
			)
		));

		if (!empty($account)) {

			$code = $account['Account']['password_reset_code'];
			if (empty($code)) {
				$code = substr(md5(uniqid()), 0, 10);
				$this->id = $account['Account']['id'];
				$this->saveField('password_reset_code', $code);
			}

			// Email password reset code
			$Event = new CakeEvent('Model.Account.passwordReset', $this, ['data' => $account]);
			$this->getEventManager()->dispatch($Event);

			return true;

		}

		return false;
	}

	/**
	 * Creates a new user account on the client's system.
	 *
	 * @param array &$data
	 * @return void
	 */
	protected function _createAccount(array &$data) {
		$Site = ClassRegistry::init('BuzzSites.Site');
		$site = $Site->getDefaultSite();

		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->createUserAccount(
			$site['Site']['api_site_id'],
			$data['Account']['first_name'] . ' ' . $data['Account']['last_name'],
			$data['Account']['email'],
			$data['CustomerAddress']['telephone'],
			CustomerAddress::generateAddressArray($data['CustomerAddress'])
		);

		if ($response !== false) {
			$account = array(
				'id' => $data['Account']['id'],
				'activation_token' => strrev(md5($data['Account']['email'])),
				'api_reference' => $response
			);
			// Specify the fields to save in addition to the passed array to make sure that
			// the password isn't resaved (which causes it to get re-encrypted).
			$this->save($account, false, ['id', 'activation_token', 'api_reference']);

			$data['Account']['activation_token'] = $account['activation_token'];
			$data['Account']['api_reference'] = $account['api_reference'];
		}

		return;
	}

	/**
	 * Updates an existing user account on the client's system.
	 *
	 * @param array $data
	 * @return void
	 */
	protected function _updateAccount(array $data) {
		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->updateUserAccount(
			$data['Account']['api_reference'],
			$data['Account']['first_name'] . ' ' . $data['Account']['last_name'],
			$data['Account']['email'],
			$data['CustomerAddress']['telephone'],
			CustomerAddress::generateAddressArray($data['CustomerAddress'])
		);

		return;
	}

	/**
	 * Gets the account booking info.
	 *
	 * @param integer $diaryUserId
	 * @return array
	 */
	public function getAccount($diaryUserId, $siteId = null) {
		if (empty($siteId)) {
			$Site = ClassRegistry::init('BuzzSites.Site');
			$site = $Site->getDefaultSite();
			$siteId = $site['Site']['api_site_id'];
		}

		return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccount(
			$diaryUserId,
			$siteId
		);
	}

	/**
	 * Gets the account rates.
	 *
	 * @param integer $apiUserId
	 * @param integer $apiLocationId
	 * @return array
	 */
	public function getAccountRates($diaryUserId, $siteId = null) {
		if (empty($siteId)) {
			$Site = ClassRegistry::init('BuzzSites.Site');
			$site = $Site->getDefaultSite();
			$siteId = $site['Site']['api_site_id'];
		}

		return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccountPrices(
			$diaryUserId,
			$siteId
		);
	}

	/**
	 * Gets a bookingfrom the API.
	 *
	 * @param int $bookingId API booking ID
	 * @return array
	 */
	public function getBooking($bookingId) {
		$data = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccountBooking($bookingId);

		if ($data !== false) {
			if (! empty($data['duration'])) {
				list($hours, $mins) = explode(':', $data['duration']);
				$data['duration'] = [
					'hours' => (int)$hours,
					'mins' => (int)$mins
				];
			} else {
				$duration = $data['duration_in_secs'];
				$data['duration'] = [
					'hours' => floor($duration / 3600),
					'mins' => floor(($duration / 60) % 60)
				];
			}
		}

		return $data;
	}

	/**
	 * Gets the bookings for a user from the API.
	 *
	 * @param int $diaryUserId
	 * @param string $fromDate
	 * @param string $sort Either 'desc' or 'asc'
	 * @param int $limit Number of results to return
	 * @param int $page Page number
	 * @return array
	 */
	public function getBookings($diaryUserId, $fromDate, $sort, $limit, $page = 0) {
		$data = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccountBookings(
			$diaryUserId,
			$fromDate,
			$sort,
			$limit,
			$page
		);

		if ($data !== false) {
			foreach ($data as &$row) {
				list($hours, $mins) = explode(':', $row['duration']);
				$row['duration'] = [
					'hours' => (int)$hours,
					'mins' => (int)$mins
				];
			}
		}

		return $data;
	}

	/**
	 * Gets the cancellations for a user from the API.
	 *
	 * @param int $diaryUserId
	 * @param string $fromDate
	 * @param string $sort Either 'desc' or 'asc'
	 * @param int $limit Number of results to return
	 * @param int $page Page number
	 * @return array
	 */
	public function getCancellations($diaryUserId, $limit, $page = 0) {
		$data = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccountCancellations(
			$diaryUserId,
			$limit,
			$page
		);

		if ($data !== false) {
			foreach ($data as &$row) {
				list($hours, $mins) = explode(':', $row['duration']);
				$row['duration'] = array(
					'hours' => intval($hours),
					'mins' => intval($mins)
				);
			}
		}

		return $data;
	}

	/**
	 * Gets the paginated credit history for a user.
	 *
	 * @param int $diaryUserId
	 * @param int $limit number of results to return
	 * @param int $page page number
	 * @return array
	 */
	public function getCredits($diaryUserId, $limit, $page = 0) {
		return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getUserAccountCreditHistory(
			$diaryUserId,
			$limit,
			$page
		);
	}

	/**
	 * Returns the available days in the diary.
	 *
	 * @param int $diaryUserId
	 * @param string $fromDate
	 * @param string $toDate
	 * @return array
	 */
	public function getAvailableDays($siteId, $diaryUserId, $fromDate, $toDate) {
		$data = [];

		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getAvailableDiaryDays(
			$siteId,
			$diaryUserId,
			$fromDate,
			$toDate
		);

		if (empty($response)) {
			return [];
		}

		$days = [];
		foreach ($response as $date) {
			$days[$date['date']] = $date['is_booked'];
		}

		$fromDate = strtotime($fromDate);
		$toDate = strtotime($toDate);
		for ($date = $fromDate; $date <= $toDate; $date += 86400) {

			$day = date('d/m/Y', $date);

			$data[$day] = array(
				'is_booked' => (isset($days[$day]) ? $days[$day] : false),
				'available' => isset($days[$day])
			);

		}

		return $data;
	}

/**
 * Returns the diary slots for a specific day.
 *
 * @param string $date Date to lookup
 * @param int $diaryUserId Diary user's account ID
 * @param int $siteId Site ID
 * @return array
 */
	public function getAvailableDiarySlots($date, $apiUserId, $siteId) {
		$data = [];

		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getAvailableDiarySlots(
			$siteId,
			date('Y-m-d', strtotime($date))
		);

		$slots = [];

		// Build an array of time slots.

		list($year, $month, $day) = explode('-', $date);
		$startTime = mktime(0, 0, 0, $month, $day, $year);
		$endTime = $startTime + 60 * 60 * 24;

		for ($time = $startTime; $time < $endTime; $time += 150) {
			$slots[$time] = array(
				'template' => array(
					'from' => null,
					'to' => null,
					'available' => 0,
					'peak' => false,
					'is_last_minute' => false,
					'bookable' => false,
					'description' => null,
					'percentage_used' => null
				),
				'booking' => []
			);
		}

		// Add bookings to the time slots array.
		foreach ($response['Booking'] as $bookingTime => $bookings) {

			$duration = 0;

			foreach ($bookings as $booking) {

				if ($booking['status'] !== 'Cancelled') {

					$startTime = strtotime("$year-$month-$day {$booking['time']}");
					$endTime = $startTime + $booking['duration_in_secs'];
					for ($time = $startTime; $time < $endTime; $time += 150) {
						$slots[$time]['booking'][$booking['reference']] = array(
							'ref' => $booking['reference'],
							'rotation' => $booking['rotation'] != 'No',
							'duration' => $booking['duration_in_secs'],
							'mine' => $booking['account_id'] == $apiUserId
						);
					}

					$duration += $booking['duration_in_secs'];
				}

			}

			// Check we have the required template for the bookings. If not it's
			// because we a fully booked template that needs creating for the app.
			if (empty($response['Template'][$bookingTime]) && $duration > 0) {

				$startTime = strtotime("$year-$month-$day $bookingTime");

				$response['Template'][$bookingTime] = array(
					'from' => $bookingTime,
					'to' => date('H:i:s', $startTime + $duration),
					'peak' => null,
					'is_last_minute' => null,
					'seconds_available' => 0,
					'bookable' => true,
					'description' => null,
					'percentage_used' => null
				);

			}

		}

		// The minimum bookable time is the booking increment converted into seconds.
		$minBookableTime = Configure::read('BuzzDiaryAccounts.increments') * 60;

		// Fill the time slots array with the templates.
		foreach ($response['Template'] as $template) {

			$startTime = strtotime("$year-$month-$day {$template['from']}");
			// We need to make sure if the end date is midnight we treat this as tomorrow.
			if ($template['to'] === '00:00:00') {
				$endTime = strtotime("$year-$month-$day {$template['to']} + 1day");
			} else {
				$endTime = strtotime("$year-$month-$day {$template['to']}");
			}
			for ($time = $startTime; $time < $endTime; $time += 150) {
				$slots[$time]['template'] = array(
					'from' => $template['from'],
					'to' => $template['to'],
					'available' => $template['seconds_available'],
					'peak' => $template['peak'],
					'is_last_minute' => $template['is_last_minute'],
					'bookable' => $minBookableTime <= $template['seconds_available'] ? $template['bookable'] : false,
					'description' => null,
					'percentage_used' => null
				);
				if (!empty($template['description'])) {
					$slots[$time]['template']['description'] = $template['description'];
					$slots[$time]['template']['percentage_used'] = $template['percentage_used'];
				}
			}

		}

		// Build an array of diary availability using the time slots array.

		reset($slots);
		$time = key($slots);

		$data[$time] = array(
			'available' => false,
			'available_time' => 0,
			'booked' => false,
			'time' => date('H:i', $time),
			'timestamp' => $time,
			'template' => null,
			'bookings' => []
		);

		$template = null;

		$oneHour = CakeTime::toUnix('+1 hours');

		foreach ($slots as $time => $slot) {

			if ($slot['template']['from'] !== $template) {
				$template = $slot['template']['from'];

				$data[$time] = array(
					'available' => $slot['template']['available'] > 0 && $slot['template']['bookable'],
					'available_time' => $slot['template']['available'],
					'booked' => !empty($slot['booking']),
					'time' => date('H:i', $time),
					'timestamp' => $time,
					'rotation' => $slot['template']['available'] > 0,
					'template' => $template,
					'peak' => $slot['template']['peak'],
					'is_last_minute' => $slot['template']['is_last_minute'],
					'bookings' => !empty($slot['booking']) ? $slot['booking'] : [],
					'description' => $slot['template']['description'],
					'percentage_used' => $slot['template']['percentage_used']
				);

				// If the time slot is in the past mark it as unavailable.
				if ($time < $oneHour) {
					$data[$time]['available'] = false;
				}

			}

		}

		ksort($data);

		// Work out the duration of each time slot.
		$prevSlot = null;
		foreach ($data as $time => &$slot) {

			// Work out how much time the current user has booked.

			$slot['my_time'] = 0;

			foreach ($slot['bookings'] as $booking) {

				if ($booking['mine'] === true) {
					$slot['my_time'] += $booking['duration'];
				}

			}

			// Work out the duration of the time slot.

			if ($prevSlot !== null) {

				$data[$prevSlot]['duration'] = $time - $prevSlot;

			}

			$prevSlot = $time;

		}

		// Work out the duration of the last time slot.
		end($data);
		$time = key($data);
		$data[$time]['duration'] = mktime(0, 0, 0, $month, $day, $year) + 60 * 60 * 24 - $time;

		CakeSession::write('Diary.slots', $data);

		return $data;
	}

	/**
	 * Get a user's diary notification status for a specific date.
	 *
	 * @param int $apiUserId API Diary User ID
	 * @param string $date
	 * @return bool
	 */
	public function getDiaryEmailNotificationStatus($apiUserId, $date, $siteId) {
		return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->getDiaryEmailNotificationStatus(
			$siteId,
			$apiUserId,
			$date
		);
	}

	/**
	 * Enable/disable diary email notifications.
	 *
	 * @param int $diaryUserId API Diary User ID
	 * @param string $date
	 * @param bool $enable
	 * @return bool
	 */
	public function setDiaryEmailNotificationStatus($apiUserId, $date, $siteId, $enable = true) {
		if ($enable === true) {
			return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->requestDiaryEmailNotificationStatus(
				$siteId,
				$apiUserId,
				$date
			);
		} else {
			return ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->cancelDiaryEmailNotificationStatus(
				$siteId,
				$apiUserId,
				$date
			);
		}
	}

/**
 * Creates a diary booking via the API.
 *
 * @param int $diaryUserId
 * @param int $time timestamp of start time/date
 * @param int $duration
 * @param bool $rotation
 * @param float $totalCost
 * @param string $notes
 * @param bool $isPeak Is peak booking flag
 * @return array
 */
	public function createDiaryBooking($siteId, $diaryUserId, $time, $duration, $rotation, $totalCost, $notes, $isPeak = false) {
		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->createDiaryBooking(
			$siteId,
			$diaryUserId,
			$duration,
			$time,
			$rotation ? 'true' : 'false',
			$totalCost,
			$notes,
			null,
			$_SERVER['REMOTE_ADDR'],
			['is_peak' => $isPeak]
		);

		if (!empty($response['status']) && strtolower($response['status']) === 'ok') {
			$event = new CakeEvent('Model.Account.bookingConfirmation', $this, array(
					'data' => $response,
					'account' => $this->findByApiReference($diaryUserId)
				)
			);
			$this->getEventManager()->dispatch($event);
		}

		return $response;
	}

	public function cancelDiaryBooking($siteId, $apiUserId, $bookingRef) {
		$response = ClassRegistry::init('BuzzDiaryAccounts.DiaryApi')->cancelDiaryBooking(
			$siteId,
			$apiUserId,
			$bookingRef
		);

		if ($response !== false && strtolower($response['status']) === 'ok') {
			$event = new CakeEvent('Model.Account.bookingCancelled', $this, array(
					'data' => $response,
					'account' => $this->findByApiReference($apiUserId)
				)
			);
			$this->getEventManager()->dispatch($event);
		}

		return $response;
	}

}
