<?php

App::uses('CakeEventListener', 'Event');
App::uses('CustomEmail', 'Lib');
App::uses('DiaryApi', 'BuzzDiaryAccounts.Model');

class AccountListener implements CakeEventListener {

	public function implementedEvents() {
		return array(
			'Model.Account.registered' => 'sendRegistrationEmail',
			'Model.Account.activated' => 'sendAccountActivatedEmail',
			'Model.Account.bookingConfirmation' => 'sendBookingConfirmationEmail',
			'Model.Account.bookingCancelled' => 'sendCancellationConfirmationEmail',
			'Model.AccountPaymentRequest.paymentRequestSent' => 'sendPaymentRequest',
			'Model.Account.passwordReset' => 'sendPasswordReset'
		);
	}

	/**
	 * Send registration email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendRegistrationEmail(CakeEvent $Event) {
		$data = $Event->data['data'];
		// If the activation token hasn't been set we don't want to send an email as the account
		// cannot be activated.
		if (empty($data['Account']['activation_token'])) {
			return;
		}

		// Send registration email
		$Email = ClassRegistry::init('BuzzEmails.Email');

		$email = $Email->generateEmailData(
			'BuzzDiaryAccounts.registration_email',
			[
				'fullName' => $data['Account']['first_name'] . ' ' . $data['Account']['last_name'],
				'firstName' => $data['Account']['first_name'],
				'lastName' => $data['Account']['last_name'],
				'activationUrl' => Router::url([
					'plugin' => 'buzz_diary_accounts',
					'controller' => 'accounts',
					'action' => 'activate',
					$data['Account']['id'],
					$data['Account']['activation_token']
				], true)
			]
		);

		if ($email !== false) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				array($data['Account']['email'] => $data['Account']['first_name'] . ' ' . $data['Account']['last_name']),
				$email['Email']['from']
			);
		}

		return;
	}

	/**
	 * Send account activation confirmation email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendAccountActivatedEmail(CakeEvent $Event) {
		$data = $Event->data['data'];

		// Send account activation confirmation email
		if (empty($data['Account']['activated_email_sent'])) {
			$Email = ClassRegistry::init('BuzzEmails.Email');

			$email = $Email->generateEmailData(
				'BuzzDiaryAccounts.activated_email',
				[
					'fullName' => $data['Account']['full_name'],
					'firstName' => $data['Account']['first_name'],
					'lastName' => $data['Account']['last_name'],
					'url' => Router::url([
						'plugin' => 'buzz_diary_accounts',
						'controller' => 'accounts',
						'action' => 'index'
					], true)
				]
			);

			if ($email !== false) {
				$Email->queueEmail(
					$email['Email']['subject'],
					$email['Email']['content'],
					array($data['Account']['email'] => $data['Account']['full_name']),
					$email['Email']['from']
				);
			}
		}

		return;
	}


	/**
	 * Send account activation confirmation email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendBookingConfirmationEmail(CakeEvent $Event) {
		$data = $Event->data['data'];
		$account = $Event->data['account'];

		// Send booking confirmation email
		$this->_sendBookingConfirmationEmail($data, $account);

		// If the booking is for today we want to send an email informing the
		// site of the booking.
		if (date('dmY', DiaryApi::dateToTime($data['date'])) === date('dmY')) {
			$this->_sendSameDayEmail($data, $account);
		}

		return;
	}

	/**
	 * Send account activation confirmation email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	protected function _sendBookingConfirmationEmail($data, $account) {
		// Send booking confirmation email
		$Email = ClassRegistry::init('BuzzEmails.Email');

		// Nicely format the booking duration.
		$duration = '';
		preg_match('|([0-9]+):([0-9]+)|', $data['duration'], $matches);
		if (!empty($matches[1]) && (int)$matches[1] > 0) {
			$duration = __d('buzz_diary_accounts', '%s hour(s) %s minutes', [$matches[1], $matches[2]]);
		} else {
			$duration = __d('buzz_diary_accounts', '%s minutes', [$matches[2]]);
		}

		$email = $Email->generateEmailData(
			'BuzzDiaryAccounts.booking_confirmation_email',
			[
				'fullName' => $account['Account']['full_name'],
				'firstName' => $account['Account']['first_name'],
				'lastName' => $account['Account']['last_name'],
				'salesRef' => $data['ref'],
				'duration' => $duration,
				// The date/time is already in the correct timezone so treat as UTC.
				'date' => CakeTime::i18nFormat(DiaryApi::dateToTime($data['date']), '%e %B %Y', false, 'UTC'),
				'time' => CakeTime::i18nFormat(DiaryApi::dateToTime($data['date']), '%H:%M', false, 'UTC')
			]
		);

		if ($email !== false) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				array($account['Account']['email'] => $account['Account']['full_name']),
				$email['Email']['from']
			);
		}

		return;
	}

	/**
	 * Sends a same day booking email.
	 *
	 * @param array $data
	 * @param array $account
	 * @return void
	 */
	protected function _sendSameDayEmail($data, $account) {
		// Send confirmation email
		$Email = ClassRegistry::init('BuzzEmails.Email');
		$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
			'BuzzBookings.same_day_booking',
			[
				'salesRef' => $data['ref'],
				'fullName' => $account['Account']['full_name'],
				'firstName' => $account['Account']['first_name'],
				'lastName' => $account['Account']['last_name'],
				'date' => CakeTime::i18nFormat(DiaryApi::dateToTime($data['date']), '%e %B %Y', false, 'UTC'),
				'time' => CakeTime::i18nFormat(DiaryApi::dateToTime($data['date']), '%H:%M', false, 'UTC')
			]
		);

		$emailAddress = null;
		if (! empty($email['Email']['to'])) {
			$emailAddress = $email['Email']['to'];
		} elseif (! empty($account['Account']['email']) && ! empty($account['Account']['full_name'])) {
			$emailAddress = array($account['Account']['email'] => $account['Account']['full_name']);
		}

		$Email->queueEmail(
			$email['Email']['subject'],
			$email['Email']['content'],
			$emailAddress,
			$email['Email']['from']
		);

		return;
	}

	/**
	 * Send account activation confirmation email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendCancellationConfirmationEmail(CakeEvent $Event) {
		$data = $Event->data['data'];
		$account = $Event->data['account'];

		// Send booking cancellation email
		$Email = ClassRegistry::init('BuzzEmails.Email');

		$email = $Email->generateEmailData(
			'BuzzDiaryAccounts.booking_cancellation_email',
			[
				'fullName' => $account['Account']['full_name'],
				'firstName' => $account['Account']['first_name'],
				'lastName' => $account['Account']['last_name'],
				'ref' => $data['ref']
			]
		);

		if ($email !== false) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				array($account['Account']['email'] => $account['Account']['full_name']),
				$email['Email']['from']
			);
		}

		return;
	}

	/**
	 * Send payment request.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendPaymentRequest(CakeEvent $Event) {
		$data = $Event->data['data'];

		// Send payment request email
		$Email = ClassRegistry::init('BuzzEmails.Email');

		$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
			'BuzzDiaryAccounts.payment_request_email',
			[
				'ref' => $data['AccountPaymentRequest']['reference'],
				'amount' => CakeNumber::currency($data['AccountPaymentRequest']['amount']),
				'name' => $data['AccountPaymentRequest']['name'],
				'description' => nl2br($data['AccountPaymentRequest']['description']),
				'requester' => $data['Account']['full_name'],
				'url' => Router::url([
					'plugin' => 'buzz_diary_accounts',
					'controller' => 'account_payment_requests',
					'action' => 'pay',
					$data['AccountPaymentRequest']['id'],
					$data['AccountPaymentRequest']['email_address']
				], true)
			]
		);

		if ($email !== false) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				array($data['AccountPaymentRequest']['email_address'] => $data['AccountPaymentRequest']['name']),
				$email['Email']['from']
			);
		}

		return;
	}

	/**
	 * Send password reset email.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function sendPasswordReset(CakeEvent $Event) {
		$data = $Event->data['data'];

		// Send password reset email
		$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
			'BuzzDiaryAccounts.password_reset_email',
			[
				'fullName' => $data['Account']['first_name'] . ' ' . $data['Account']['last_name'],
				'firstName' => $data['Account']['first_name'],
				'lastName' => $data['Account']['last_name'],
				'url' => Router::url([
					'plugin' => 'buzz_diary_accounts',
					'controller' => 'accounts',
					'action' => 'password_reset',
					$data['Account']['id'],
					$data['Account']['password_reset_code']
				], true)
			]
		);

		if ($email !== false) {
			// As the user is requesting an email to be sent so that they can
			// log in to their account we will send the email straightaway
			// rather than queue it. Otherwise they may keep requesting the
			// email and then get their inbox flooded by responses later on.
			$Email = new CustomEmail();
			$Email->template('default');
			$Email->to([$data['Account']['email'] => $data['Account']['full_name']]);
			$Email->from($email['Email']['from']);
			$Email->subject($email['Email']['subject']);
			$Email->viewVars([
				'content' => $email['Email']['content'],
				'settings' => Configure::read('SiteSetting'),
				'strings' => Configure::read('String')
			]);
			$Email->helpers(['BuzzTranslate.I18n']);
			$Email->send();
		}

		return;
	}

}
