<?php

App::uses('BuzzDiaryAccountsAppController', 'BuzzDiaryAccounts.Controller');

class AccountsController extends BuzzDiaryAccountsAppController {

/**
 * Helpers
 */
	public $helpers = [
		'BuzzPurchase.Purchase'
	];

	/**
	 * Define actions that will require account authentication.
	 *
	 * @var array
	 */
	public $requiresAccountAuth = [
		'book',
		'bookings',
		'camp',
		'cancel',
		'booking',
		'cancellations',
		'diary',
		'get_account',
		'get_account_rates',
		'get_bookings',
		'get_cancellations',
		'get_diary_day',
		'get_next_bookings',
		'get_next_cancellations',
		'index',
		'past_bookings',
		'remove_camp',
		'toggle_account_rates',
		'update',
		'update_notifications',
		'update_password'
	];

	/**
	 * Allowed admin actions
	 *
	 * @var array
	 */
	public $adminActions = array(
		'admin_index',
		'admin_delete',
		'admin_toggle',
		'admin_edit',
		'admin_login'
	);

	/**
	 * Before filter
	 *
	 * @return void
	 */
	public function beforeFilter() {
		parent::beforeFilter();

		if (Configure::check('BuzzDiaryAccounts.main_page') === true) {
			$this->set('diaryAccountsMainPage', Configure::read('BuzzDiaryAccounts.main_page'));
		} else {
			$this->set('diaryAccountsMainPage', '/');
		}

		return;
	}

	/**
	 * Login action.
	 *
	 * Use the plugin's config file to override this action if you want login on
	 * another page (e.g. from a specific activity page).
	 *
	 * @return void
	 */
	public function login() {
		if (!empty($this->request->data)) {
			if ($this->Auth->login()) {
				return $this->redirect(['action' => 'index']);
			} else {
				$this->Session->setFlash(__d('buzz_diary_accounts', 'The username or password is incorrect'), 'flash_fail');
			}
		}

		$this->view = 'BuzzDiaryAccounts./Accounts/login';

		return;
	}

	/**
	 * Logout action.
	 *
	 * @return void
	 */
	public function logout() {
		$this->Auth->logout();
		return $this->redirect(['action' => 'login']);
	}

	public function register() {
		$Account = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			// New accounts are disabled by default.
			$this->request->data['Account']['is_active'] = false;

			if ($Account->saveAssociated($this->request->data)) {
				if (Configure::check('BuzzDiaryAccounts.registered_page_id') === true) {
					$this->assignPage(
						Configure::read('BuzzDiaryAccounts.registered_page_id')
					);
				}
				$this->view = 'BuzzDiaryAccounts.registered';
				return;
			} else {
				$this->request->data['Account']['confirm_password'] = '';
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();

		}

		$this->set('diaryAccountsLoginUrl', $this->_getLoginAction());
		$this->_populateLookups();
		$this->view = 'BuzzDiaryAccounts.register';

		return;
	}

	/**
	 * Activate an account
	 *
	 * @param int $accountId
	 * @param string $activationToken
	 * @return void
	 */
	public function activate($accountId, $activationToken) {
		$Account = $this->{$this->modelClass};

		// Activate account.
		$data = $Account->activateAccount($accountId, $activationToken);
		if ($data !== false) {
			// Move the country and location data inside $data['Account'] for
			// the auth component.
			$data['Account']['CustomerAddress'] = $data['CustomerAddress'];
			if ($this->Auth->login($data['Account'])) {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Your account has been activated and you are now logged in'),
					'flash_success'
				);
				return $this->redirect(['action' => 'index']);
			}
		}

		$this->Session->setFlash(
			__d('buzz_diary_accounts', 'Sorry, there was a problem activating your account'),
			'flash_fail'
		);

		return $this->redirect(['action' => 'login']);
	}

	/**
	 * Forgotten password
	 *
	 * @return void
	 */
	public function forgotten_password() {
		$Account = $this->{$this->modelClass};

		if (!empty($this->request->data)) {
			if ($Account->resetPassword($this->request->data['Account']['email']) === true) {
				$this->view = 'BuzzDiaryAccounts.password_reset_email';
				return;
			} else {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Account not found'),
					'flash_fail'
				);
			}
		}

		$this->view = 'BuzzDiaryAccounts.forgotten_password';

		return;
	}

	/**
	 * Reset password
	 *
	 * @param int $id Account ID
	 * @param string $code
	 * @return void
	 */
	public function password_reset($id, $code) {
		$Account = $this->{$this->modelClass};

		// We want to retrieve the account using both the account ID and reset
		// code to try and ensure that someone isn't accessing an account they
		// don't have access to (including the account ID makes it a little
		// more secure).
		$account = $Account->find(
			'first',
			array(
				'conditions' => array(
					'Account.id' => $id,
					'Account.password_reset_code' => $code
				)
			)
		);

		if (empty($account)) {
			$this->Session->setFlash(
				__d('buzz_diary_accounts', 'Password reset code not found'),
				'flash_fail'
			);
			$this->redirect(['action' => 'forgotten_password']);
			return;
		}

		if (!empty($this->request->data)) {
			$this->request->data['Account']['id'] = $account['Account']['id'];
			$this->request->data['Account']['password_reset_code'] = '';
			if ($Account->save($this->request->data) !== false) {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Password changed'),
					'flash_success'
				);
				$this->redirect(['action' => 'login']);
				return;
			} else {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Please correct the errors below'),
					'flash_fail'
				);
			}
		}

		$this->view = 'BuzzDiaryAccounts.password_reset';

		return;
	}

	/**
	 * Account overview. Actual page content is loaded in via AJAX.
	 *
	 * @return void
	 */
	public function index() {
		$this->_loadPageContent();
		$this->view = 'BuzzDiaryAccounts./Accounts/index';

		return;
	}

	/**
	 * 'My Account' page for updating details.
	 *
	 * @return void
	 */
	public function update() {
		$Account = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			// Save the data, the API will get updated in the afterSave callback.
			if ($Account->saveAssociated($this->request->data) !== false) {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Your account details have been updated'),
					'flash_success'
				);
				return $this->redirect($this->request->here);
			} else {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			$accountId = $this->Auth->user('id');
			$this->request->data = $Account->readForEdit($accountId);

		}

		$this->_loadPageContent();
		$this->_populateLookups();
		$this->view = 'BuzzDiaryAccounts./Accounts/update';
		return;
	}

	/**
	 * Update password. Redirects to 'update'.
	 *
	 * @return void
	 */
	public function update_password() {
		$Account = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$Account->id = $this->request->data['Account']['id'];
			$currentPassword = $Account->field('password');

			if (AuthComponent::password($this->request->data['Account']['current_password']) === $currentPassword) {
				$Account->validatePassword();
				if ($Account->save($this->request->data) !== false) {
					$this->Session->setFlash(__d('buzz_diary_accounts', 'Your password has been updated'), 'flash_success');
				} else {
					$this->Session->setFlash(__d('buzz_diary_accounts', 'There was a problem with your new password'), 'flash_fail');
				}
			} else {
				$this->Session->setFlash(__d('buzz_diary_accounts', 'Your current password was incorrect'), 'flash_fail');
			}

		}

		return $this->redirect(['action' => 'update']);
	}

	/**
	 * Future bookings.
	 *
	 * @return void
	 */
	public function bookings() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$limit = 10;
		$data = $Account->getBookings($apiUserId, date('Y-m-d'), 'asc', $limit);
		$paged = !(count($data) < $limit);

		// Store the bookings in the session so that we can retrieve the individual booking details.
		$this->_cacheBookings($data);

		$this->_loadPageContent();
		$this->set(compact('data', 'limit', 'paged'));
		$this->set('allowCancellations', true);
		$this->view = 'BuzzDiaryAccounts./Accounts/bookings';

		return;
	}

	/**
	 * Get booking details
	 *
	 * @param int $siteId Site Id
	 * @param int $bookingId Booking ID
	 * @return void
	 */
	public function booking($siteId, $bookingId) {
		$Account = $this->{$this->modelClass};
		// Handle cancellation request
		if (!empty($this->request->data)) {
			$apiUserId = $this->Auth->user('api_reference');
			$data = $Account->cancelDiaryBooking($siteId, $apiUserId, $bookingId);

			$this->_loadPageContent();
			$this->set(compact('data', 'siteId'));
			$this->view = 'BuzzDiaryAccounts.cancellation_complete';

			return;
		}

		$this->loadModel('BuzzSites.Site');
		$site = $this->Site->find('first', ['conditions' => ['api_site_id' => $siteId]]);

		$data = $Account->getBooking($bookingId);
		$this->set(compact('data', 'siteId', 'site'));
		$this->set('accountUser', $this->Session->read('Auth'));
		$this->view = 'BuzzDiaryAccounts./Accounts/booking';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * Past bookings.
	 *
	 * @return void
	 */
	public function past_bookings() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$limit = 10;
		$data = $Account->getBookings($apiUserId, date('Y-m-d'), 'desc', $limit);
		$paged = !(count($data) < $limit);

		$this->_cacheBookings($data);

		$this->_loadPageContent();
		$this->set(compact('data', 'limit', 'paged'));
		$this->set('allowCancellations', false);
		$this->view = 'BuzzDiaryAccounts./Accounts/past_bookings';

		return;
	}

	/**
	 * Cache booking data in the session.
	 *
	 * @param array $data Booking data
	 *
	 * @return void
	 */
	protected function _cacheBookings($data) {
		// Store the bookings in the session so that we can retrieve the individual booking details.
		$bookings = $this->Session->read('BuzzDiaryAccounts.bookings');
		$bookings = (array)$data + (array)$bookings;
		$this->Session->write('BuzzDiaryAccounts.bookings', $bookings);
	}

	/**
	 * Loads the next page of bookings
	 *
	 * @param int $page
	 * @param int $limit
	 * @param bool $past
	 * @return void
	 */
	public function get_next_bookings($page = 0, $limit = 10, $past = false) {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getBookings($apiUserId, date('Y-m-d'), ($past ? 'desc' : 'asc'), $limit, $page);

		if (empty($data)) {
			// No more bookings to display so just die so that there is no response.
			die();
		}
		$this->_cacheBookings($data);

		$paged = !(count($data) < $limit);

		$this->set(compact('data', 'page', 'paged', 'limit'));
		$this->set('allowCancellations', !$past);

		$this->view = 'BuzzDiaryAccounts./Elements/bookings_table';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * Cancelled bookings.
	 *
	 * @return void
	 */
	public function cancellations() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$limit = 10;
		$data = $Account->getCancellations($apiUserId, $limit);
		$paged = !(count($data) < $limit);

		$this->_loadPageContent();
		$this->set(compact('data', 'limit', 'paged'));
		$this->view = 'BuzzDiaryAccounts./Accounts/cancellations';

		return;
	}

	/**
	 * Loads the next page of bookings
	 *
	 * @param int $page
	 * @param int $limit
	 * @return void
	 */
	public function get_next_cancellations($page = 0, $limit = 10) {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getCancellations($apiUserId, $limit, $page);

		if (empty($data)) {
			// No more credits to display so just die so that there is no response.
			die();
		}

		$paged = !(count($data) < $limit);

		$this->set(compact('data', 'page', 'paged', 'limit'));
		$this->view = 'BuzzDiaryAccounts./Elements/cancellations_table';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * Diary page.
	 *
	 * @param string $date date to load the diary on (dd-mm-yyyy)
	 * @return void
	 */
	public function diary($siteId, $date = null) {
		$Account = $this->{$this->modelClass};

		$this->loadModel('BuzzSites.Site');

		$this->Site->setActiveSite($siteId, true);

		$apiUserId = $this->Auth->user('api_reference');

		$expires = strtotime('+6 months');
		$fromDate = date('Y-m-d');
		$toDate = date('Y-m-d', $expires);

		$data = $Account->getAvailableDays($siteId, $apiUserId, $fromDate, $toDate, $siteId);

		if ($date === null) {
			// Find the first bookable day.
			foreach ($data as $key => $value) {
				if ($value['available'] === true) {
					$date = DiaryApi::dateToSqlFormat($key);
					break;
				}
			}
		}

		$this->_loadPageContent();
		$this->set(compact('data', 'date', 'fromDate', 'toDate', 'siteId'));
		$this->view = 'BuzzDiaryAccounts.diary';

		return;
	}

	/**
	 * Diary day page (pulled in via AJAX).
	 *
	 * @param integer $siteId
	 * @param string $date
	 * @return void
	 */
	public function get_diary_day($siteId, $date) {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$this->loadModel('BuzzSites.Site');
		$this->Site->setActiveSite($siteId, true);
		$this->set('site', $this->activeSite);

		$this->loadModel('BuzzSites.Site');
		$site = $this->Site->find('first', ['conditions' => ['api_site_id' => $siteId]]);

		$notifications = $Account->getDiaryEmailNotificationStatus($apiUserId, $date, $siteId);
		$this->request->data['Account']['notifications'] = $notifications;
		$this->request->data['Account']['date'] = $date;
		$this->request->data['Account']['site_id'] = $siteId;

		$data = $Account->getAvailableDiarySlots($date, $apiUserId, $siteId);
		$camps = $this->_getCampBookings($siteId, $data);
		$campBookingIds = array_keys($camps);

		// Made sure that for no availability we have hour slots to avoid the diary showing up as just a blank area.
		foreach ($data as $time => &$item) {
			// We need to flag slots that contain a camp so that we can always show the camp adverts
			// in the diary.
			$item['has_camp'] = (bool)array_intersect(array_keys($item['bookings']), $campBookingIds);

			if ($item['available'] === false && $item['duration'] > 3600) {

				$duration = $item['duration'];
				$data[$time]['duration'] = 3600;

				while ($duration > 3600) {

					$time += 3600;

					$data[$time] = $item;
					$data[$time]['bookings'] = [];
					$data[$time]['my_time'] = 0;
					$data[$time]['time'] = date('H:i', $time);
					$data[$time]['duration'] = $duration > 3600 ? 3600 : $duration;
					$data[$time]['site_id'] = $siteId;

					$duration -= $data[$time]['duration'];

				}

			}

		}

		ksort($data);

		$this->set(compact('data', 'date', 'camps', 'site'));
		$this->view = 'BuzzDiaryAccounts./Elements/get_diary_day';
		$this->layout = 'ajax';

		return;
	}

/**
 * Returns the camps associated with the bookings data indexed by booking ref
 * @param array $data Bookings data
 * @return array
 */
	protected function _getCampBookings($siteId, $data) {
		$this->loadModel('BuzzCamps.Camp');
		$bookings = Hash::extract($data, '{n}.bookings.{n}.ref');
		$camps = $this->Camp->find('all', [
			'contain' => [
				'Image'
			],
			'conditions' => [
				'Camp.booking_ref' => $bookings
			]
		]);

		return Hash::combine($camps, '{n}.Camp.booking_ref', '{n}');
	}

	/**
	 * Update diary email notifications (called via AJAX).
	 *
	 * @return void
	 */
	public function update_notifications($siteId) {
		$Account = $this->{$this->modelClass};

		$data = $this->request->query['data'];

		$Account->setDiaryEmailNotificationStatus(
			$this->Auth->user('api_reference'),
			$data['Account']['date'],
			$siteId,
			!empty($data['Account']['notifications'])
		);

		die();

		return;
	}

/**
 * Make a booking in the diary.
 *
 * @param string|int $siteId Site ID
 * @param string $time Time of booking slot
 * @return void
 */
	public function book($siteId, $time) {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$this->loadModel('BuzzSites.Site');
		$this->Site->setActiveSite($siteId, true);
		$this->set('site', $this->Site->getActiveSite());
		if (! empty($this->request->data)) {
			$this->_postBook($siteId, $apiUserId);
		} else {
			$this->request->data[$Account->alias]['time'] = $time;
		}

		// We need to get the account for the total available balance.
		$account = $Account->getAccount($apiUserId, $siteId);

		// We need to check that the slot exists within the known diary slots.
		$diary = $this->Session->read('Diary.slots');
		if (empty($diary[$time])) {
			throw new NotFoundException();
		} else {
			$data = $diary[$time];
		}

		$rates = $Account->getAccountRates($apiUserId, $siteId);
		if ($diary[$time]['is_last_minute'] === true && !empty($rates['is_last_minute'])) {
			$rate = $rates['last_minute'];
		} else {
			$rate = $diary[$time]['peak'] === true ? $rates['peak'] : $rates['off_peak'];
		}
		$this->request->data[$Account->alias]['hourly_rate'] = $rate;
		// Convert rate per hour to per minute.
		$rate /= 60;

		$available = $data['available_time'] / 60;
		$maxBookingTime = Configure::read('BuzzDiaryAccounts.max_booking_time');
		if ($maxBookingTime === null || $maxBookingTime > $available) {
			$maxBookingTime = $available;
		}

		$durations = [];
		$increment = Configure::read('BuzzDiaryAccounts.increments');
		for ($duration = $increment; $duration <= $maxBookingTime; $duration += $increment) {
			if ($duration * $rate <= $account['total_available']) {
				$durations[$duration] = __d('buzz_diary_accounts', '%d Minutes', $duration);
			} else {
				break;
			}
		}

		if (empty($this->request->data[$Account->alias]['duration'])) {
			// Set the default duration to 30 minutes if available, otherwise
			// the highest available duration.
			end($durations);
			$this->request->data[$Account->alias]['duration'] = isset($durations[30]) ? 30 : key($durations);
		}

		$this->set(compact('data', 'durations', 'rate', 'time'));
		$this->view = 'BuzzDiaryAccounts.book';
		$this->layout = 'ajax';
	}

/**
 * Process a booking
 * @param int $apiUserId API user ID
 * @return void
 */
	protected function _postBook($siteId, $apiUserId) {
		$Account = $this->{$this->modelClass};
		// Process booking
		$data = $Account->createDiaryBooking(
			$siteId,
			$apiUserId,
			$this->request->data[$Account->alias]['time'],
			$this->request->data[$Account->alias]['duration'],
			$this->request->data[$Account->alias]['rotation'],
			number_format(
				$this->request->data[$Account->alias]['duration'] * $this->request->data[$Account->alias]['hourly_rate'] / 60,
				2,
				'.',
				''
			),
			$this->request->data[$Account->alias]['notes'],
			$this->request->data[$Account->alias]['peak']
		);

		if ($data !== false) {
			$this->Session->setFlash(__d('buzz_diary_accounts', 'Thank You! Your booking is complete'), 'flash_success');
		} else {
			$this->Session->setFlash(__d('buzz_diary_accounts', 'There was an issue placing this booking'), 'flash_fail');
		}

		return $this->redirect([
			'action' => 'diary',
			$siteId,
			date('Y-m-d', $this->request->data[$Account->alias]['time'])
		]);
	}

	public function cancel($siteId, $bookingRef) {
		$Account = $this->{$this->modelClass};

		if (!empty($this->request->data)) {
			$apiUserId = $this->Auth->user('api_reference');
			$data = $Account->cancelDiaryBooking($siteId, $apiUserId, $bookingRef);

			$this->_loadPageContent();
			$this->set(compact('data'));
			$this->view = 'BuzzDiaryAccounts.cancellation_complete';

			return;
		}

		$this->set('ref', $bookingRef);
		$this->view = 'BuzzDiaryAccounts.cancel';
		$this->layout = 'ajax';

		return;
	}

/**
 * Attach a camp to a booking
 * @param string $bookingRef Booking Ref
 * @param string $date Booking date
 * @return void
 */
	public function camp($siteId, $bookingRef, $date) {
		$canCreateCamps = $this->Auth->user('can_create_camps');
		if (empty($canCreateCamps)) {
			throw new ForbiddenException('User does not have permission to create camps');
		}
		if ($this->request->is('post')) {
			return $this->_postCamp($siteId, $bookingRef, $date);
		} else {
			return $this->_getCamp($siteId, $bookingRef, $date);
		}
	}

/**
 * Get request camp modal
 * @param string $bookingRef Booking Ref
 * @param string $date Booking date
 * @return void
 */
	protected function _getCamp($siteId, $bookingRef, $date) {
		$this->loadModel('BuzzCamps.Camp');

		$camp = $this->Camp->findByBookingRef($bookingRef);
		if (! empty($camp['Camp']['id'])) {
			$this->request->data['Account']['camp_id'] = $camp['Camp']['id'];
		}

		$camps = $this->Camp->find('list', [
			'contain' => ['AccountProfile'],
			'conditions' => [
				'Camp.date' => $date,
				'Camp.site_id' => $siteId,
				'AccountProfile.account_id' => $this->Auth->user('id'),
				'OR' => [
					'Camp.booking_ref IS NULL',
					'Camp.booking_ref' => $bookingRef
				]
			],
			// We only want the before callbacks enabled to ensure `contain` works, we don't need
			// the `afterFind` of the Camps model here.
			'callbacks' => 'before'
		]);

		$this->set('ref', $bookingRef);
		$this->set(compact('camps', 'date', 'siteId'));
		$this->view = 'BuzzDiaryAccounts.camp';
		$this->layout = 'ajax';

		return;
	}

/**
 * Post request camp modal
 * @param string $bookingRef Booking Ref
 * @param string $date Booking date
 * @return void
 */
	protected function _postCamp($siteId, $bookingRef, $date) {
		$this->loadModel('BuzzCamps.Camp');
		if (! empty($this->request->data['Account']['camp_id'])) {
			$camp = $this->Camp->find('first', [
				'contain' => ['AccountProfile'],
				'conditions' => [
					'Camp.id' => $this->request->data['Account']['camp_id'],
					'AccountProfile.account_id' => $this->Auth->user('id')
				]
			]);
			if (! empty($camp)) {
				$camp['Camp']['booking_ref'] = $bookingRef;
				if ($this->Camp->save($camp)) {
					$this->Session->setFlash(__d('buzz_diary_accounts', 'Your camp has been added to the booking'), 'flash_success');
					return $this->redirect(['action' => 'diary', $siteId, $date]);
				}
			}
		}
		$this->Session->setFlash(__d('buzz_diary_accounts', 'There was a problem adding your camp to the booking'), 'flash_fail');

		return $this->redirect(['action' => 'diary', $siteId, $date]);
	}

/**
 * Dettach a camp from a booking
 * @param string $bookingRef Booking Ref
 * @param string $date Booking date
 * @return void
 */
	public function remove_camp($siteId, $bookingRef, $date) {
		$this->loadModel('BuzzCamps.Camp');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		if (empty($canCreateCamps)) {
			throw new ForbiddenException('User does not have permission to create camps');
		}
		$camp = $this->Camp->find('first', [
			'contain' => ['AccountProfile'],
			'conditions' => [
				'Camp.booking_ref' => $bookingRef,
				'AccountProfile.account_id' => $this->Auth->user('id')
			]
		]);
		if (! empty($camp)) {
			$camp['Camp']['booking_ref'] = null;
			if ($this->Camp->save($camp)) {
				$this->Session->setFlash(__d('buzz_diary_accounts', 'Your camp has been removed from the booking'), 'flash_success');
				return $this->redirect(['action' => 'diary', $siteId, $date]);
			}
		}

		$this->Session->setFlash(__d('buzz_diary_accounts', 'There was a problem removing your camp from the booking'), 'flash_fail');

		return $this->redirect(['action' => 'diary', $siteId, $date]);
	}

/**
 * Populate lookups.
 *
 * @return void
 */
	protected function _populateLookups() {
		$Account = $this->{$this->modelClass};

		$countries = $Account->CustomerAddress->Country->translatedList();
		$usStates = $Account->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));
		return;
	}

	/**
	 * AJAX method for retrieving basic account booking info.
	 *
	 * @return void
	 */
	public function get_account() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');
		$data = $Account->getAccount($apiUserId);

		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts./Elements/get_account';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * AJAX method for retrieving account rates.
	 *
	 * @return void
	 */

	public function get_account_rates() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');
		$data = $Account->getAccountRates($apiUserId);

		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts./Elements/get_account_rates';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * AJAX method for retrieving bookings.
	 *
	 * @param bool $past pass true to retrieve past bookings, otherwise returns future ones
	 * @return void
	 */
	public function get_bookings($past = false) {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getBookings($apiUserId, date('Y-m-d'), ($past ? 'desc' : 'asc'), 3);
		$this->_cacheBookings($data);

		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->find('list', ['fields' => ['api_site_id', 'name']]);

		$this->set(compact('data', 'sites'));
		$this->set('allowCancellations', !$past);
		$this->view = 'BuzzDiaryAccounts./Elements/bookings_table';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * AJAX method for cancellations.
	 *
	 * @return void
	 */
	public function get_cancellations() {
		$Account = $this->{$this->modelClass};

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getCancellations($apiUserId, 3);

		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts./Elements/cancellations_table';
		$this->layout = 'ajax';

		return;
	}

	public function toggle_account_rates() {
		$Account = $this->{$this->modelClass};

		$this->request->data['Account']['id'] = $this->Auth->user('id');
		$Account->save($this->request->data);

		$this->redirect($this->referer());

		return;
	}

	/**
	 * Admin login for masquerading as an account user.
	 *
	 * @param int $id Account record ID
	 * @return void
	 */
	public function admin_login($id) {
		$Account = $this->{$this->modelClass};

		$data = $Account->find('first', array(
			'conditions' => array(
				'Account.id' => $id,
				'Account.is_active' => true,
				'Account.api_reference <>' => null
			)
		));

		if (empty($data)) {
			return $this->redirect(['admin' => true, 'action' => 'edit', $id]);
		}

		$user = $data['Account'];

		AuthComponent::$sessionKey = 'Auth.Account';
		$this->Auth->login($user);

		$this->redirect(['action' => 'index', 'admin' => false]);

		return;
	}

	protected function _adminFormToolbar($id = null) {
		$data = parent::_adminFormToolbar($id);

		unset($data['Add New']);

		if (!empty($id)) {
			$data['Login'] = array(
				'url' => array(
					'action' => 'login',
					$id
				),
				'icon' => 'key',
				'target' => '_blank'
			);
		}

		return $data;
	}

	/**
	 * Defines which actions are available to each row of the admin_index table results
	 *
	 * Override in your controller to customise
	 * Customise the handlers for these actions in /view/Admin/Elements/index_results.ctp
	 *
	 * @return array
	 */
	protected function _adminIndexActions() {
		$actions = array(
			'Delete',
			'Edit'
		);

		return $actions;
	}

	/**
	 * Admin index columns
	 *
	 * @var array
	 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$newColumns = array(
			'Account.full_name' => array(
				'label' => __d('buzz_diary_accounts', 'Name'),
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'Account.id', $newColumns);
	}

	/**
	 * Filters
	 *
	 * @return array
	 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['Account.name']);

		$newFilters = array(
			'Account.full_name' => array(
				'label' => __d('buzz_diary_accounts', 'Name'),
				'type' => 'string',
				'compare' => array('CONCAT(Account.first_name, " ", Account.last_name) LIKE' => "%%%s%%")
			)
		);

		return ArrayUtil::addAfter($filters, 'Account.id', $newFilters);
	}

/**
 * Admin edit
 *
 * @param int $id Account ID
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		// Make sure we have the purchase data from the database available in the view. Otherwise,
		// on form submit not all the data is available in the request data.
		if ($this->request->is(['post', 'put'])) {
			$account = $this->Booking->readForEdit($id);
		} else {
			$account = $this->request->data;
		}
		$this->set(compact('account'));
		$this->view = 'BuzzDiaryAccounts.admin_form';
	}

}
