<?php

App::uses('BuzzDiaryAccountsAppController', 'BuzzDiaryAccounts.Controller');

class AccountPaymentRequestsController extends BuzzDiaryAccountsAppController {

	/**
	 * Define actions that will require account authentication.
	 *
	 * @var array
	 */
	public $requiresAccountAuth = [
		'add',
		'cancel',
		'index'
	];

	/**
	 * Allowed admin actions
	 *
	 * @var array
	 */
	public $adminActions = [];

	/**
	 * Components
	 *
	 * @var array
	 */
	public $components = array(
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	);

/**
 * Helpers
 */
	public $helpers = array(
		'BuzzPurchase.Purchase'
	);

	/**
	 * View all outstanding payments
	 *
	 * @return void
	 */
	public function index() {
		$AccountPaymentRequest = $this->{$this->modelClass};

		$userId = $this->Auth->user('id');

		$data = $AccountPaymentRequest->find(
			'all',
			array(
				'conditions' => array(
					'AccountPaymentRequest.account_id' => $userId,
					'AccountPaymentRequest.account_credit_id' => null,
					'AccountPaymentRequest.is_removed' => false
				),
				'order' => array(
					'AccountPaymentRequest.created' => 'DESC',
					'AccountPaymentRequest.id' => 'DESC'
				)
			)
		);

		$this->_loadPageContent();
		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts.index';

		return;
	}

	/**
	 * Add a payment request
	 *
	 * @return void
	 */
	public function add() {
		$AccountPaymentRequest = $this->{$this->modelClass};

		if ($this->Auth->user('can_request_payments') === false) {
			$this->Session->setFlash(__d('buzz_diary_accounts', 'You do not have permission to request payments'), 'flash_fail');
			return $this->redirect(['controller' => 'account_credits', 'action' => 'index']);
		}

		if (!empty($this->request->data)) {

			$userId = $this->Auth->user('id');

			// Associate the request with the current user.
			$this->request->data['AccountPaymentRequest']['account_id'] = $userId;
			// Make sure that the request doesn't get associated with a payment yet for security.
			$this->request->data['AccountPaymentRequest']['account_credit_id'] = null;

			if ($AccountPaymentRequest->save($this->request->data)) {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'A payment request has been sent to %s', [$this->request->data['AccountPaymentRequest']['email_address']]),
					'flash_success'
				);
				return $this->redirect(['controller' => 'account_credits', 'action' => 'index']);
			} else {
				$this->Session->setFlash(__d('buzz_diary_accounts', 'Please correct the errors below'), 'flash_fail');
			}

		}

		$this->_loadPageContent();
		$this->view = 'BuzzDiaryAccounts.add';

		return;
	}

	/**
	 * Cancel a payment request
	 *
	 * @param int $id
	 * @return void
	 */
	public function cancel($id) {
		$AccountPaymentRequest = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			if ($AccountPaymentRequest->delete($id)) {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Payment request cancelled.'),
					'flash_success'
				);
				return $this->redirect(['controller' => 'account_credits', 'action' => 'index']);
			}

		}

		$data = $AccountPaymentRequest->readForView($id);

		$this->_loadPageContent();
		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts.cancel';

		return;
	}

/**
 * Allow someone to pay a payment request
 *
 * @param int $id Payment request ID
 * @param string $email Email address of payment request (user may use a different address for the payment)
 * @return void
 * @throws ForbiddenException
 */
	public function pay($id, $email) {
		$AccountPaymentRequest = $this->{$this->modelClass};

		// Retrieve the payment request.
		$data = $AccountPaymentRequest->find(
			'first',
			array(
				'contain' => array(
					'Account'
				),
				'conditions' => array(
					'AccountPaymentRequest.id' => $id,
					'AccountPaymentRequest.email_address' => $email
				)
			)
		);

		if (!empty($data['AccountPaymentRequest']['account_credit_id'])) {
			// Already paid
			$this->view = 'BuzzDiaryAccounts.paid';
			return;
		} elseif ($data['AccountPaymentRequest']['is_removed'] === true) {
			// Already paid
			$this->view = 'BuzzDiaryAccounts.cancelled';
			return;
		} elseif (empty($data)) {
			throw new ForbiddenException();
		}

		if (!empty($this->request->data)) {

			$this->request->data['AccountCredit']['amount'] = $data['AccountPaymentRequest']['amount'];
			$result = $AccountPaymentRequest->AccountCredit->createCredit(
				$data['AccountPaymentRequest']['account_id'],
				$this->request->data
			);

			if ($result === true) {
				$this->Session->write('AccountPaymentRequest.id', $id);
				$this->_pay($id, $email);
			} else {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Set the email address to the one used for the request, the payer
			// can change this via the form as we don't need them to match.
			$this->request->data['CustomerAddress']['email'] = $email;

			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
			$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();

		}

		$this->_populateLookups();
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts.pay';
	}

/**
 * Stripe payment
 *
 * @param int $id Payment request ID
 * @param string $email Email address of payment request (user may use a different address for the payment)
 * @return void
 */
	public function stripe_payment($id, $email) {
		if (Configure::read('Transactions.gateway') !== 'stripe_checkout') {
			throw new MethodNotAllowedException('This site is not configured to use this action');
		}

		$request = compact('id', 'email');

		$AccountPaymentRequest = $this->{$this->modelClass};
		$basketId = $AccountPaymentRequest->AccountCredit->getBasketId();
		$basket = $AccountPaymentRequest->getBasket($basketId);

		$stripeTransaction = $this->_takePayment($basketId, $basket, $request);

		if ($stripeTransaction['result'] === false || empty($stripeTransaction['transaction_id'])) {
			$this->Session->setFlash(
				__d('buzz_diary_accounts', 'There was a problem processing your payment, please check your details and try again'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'pay', $id, $email]);
		}

		$this->loadModel('Transactions.Transaction');
		$transaction = $this->Transaction->findById(
			$stripeTransaction['transaction_id']
		);

		$stripeKey = Configure::read('Transactions.stripe.live.publishable_key');
		if (Configure::read('db.config') !== 'live') {
			$stripeKey = Configure::read('Transactions.stripe.dev.publishable_key');
		}

		$this->set(compact('transaction', 'stripeKey', 'id', 'email'));
		// Use the AJAX layout to improve the redirect transition to Stripe.
		$this->layout = 'ajax';
	}

	/**
	 * Populate lookups.
	 *
	 * @return void
	 */
	protected function _populateLookups() {
		$AccountCredit = $this->{$this->modelClass}->AccountCredit;

		$countries = $AccountCredit->CustomerAddress->Country->translatedList();
		$usStates = $AccountCredit->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));
		return;
	}

	/**
	 * Processes the payment and triggers the completion of the payment on success
	 *
	 * @return void
	 */
	protected function _pay($id, $email) {
		$AccountPaymentRequest = $this->{$this->modelClass};

		// Get the latest basket.
		$basketId = $AccountPaymentRequest->AccountCredit->getBasketId();
		$basket = $AccountPaymentRequest->getBasket($basketId);

		if ($basket['AccountCredit']['amount'] <= 0) {
			return $this->_completeCreditTransaction($basketId, $id);
		}

		if (Configure::read('Transactions.gateway') === 'stripe_checkout') {
			return $this->redirect(['action' => 'stripe_payment', $id, $email]);
		}

		$payment = [];

		if (Configure::read('BuzzPurchase.onsite') === true) {
			$payment = $this->request->data['Payment'];

			// Split name into first and second names for payment gateways.
			$fullName = trim($payment['card_name']);
			$splitName = explode(' ', $fullName);
			$payment['last_name'] = array_pop($splitName);
			$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
		}

		$request = compact('id', 'email');
		$transaction = $this->_takePayment($basketId, $basket, $request, $payment);

		if (!empty($transaction['result'])) {
			// Payment taken successfully, complete purchase.
			return $this->_completeCreditTransaction($basketId, $id);
		}

		$this->Session->setFlash(
			__d('buzz_diary_accounts', 'There was a problem processing your payment, please check your details and try again'),
			'flash_fail'
		);
	}

/**
 * Take a payment
 *
 * @param int $basketId Basket ID
 * @param array $basket Basket data
 * @param array $request Credit request data
 * @param array $payment Payment data
 * @return mixed The response from Transactions::takePayment()
 */
	protected function _takePayment($basketId, $basket, $request, $payment = []) {
		return $this->Transactions->takePayment(
			// Return URLs
			[
				'return' => Router::url(['action' => 'payment_callback', $basketId, $request['id']], true),
				'cancel' => Router::url(['action' => 'pay', $request['id'], $request['email']], true)
			],
			// Calling record
			[
				'model' => 'BuzzDiaryAccounts.AccountCredit',
				'model_id' => $basket['AccountCredit']['id']
			],
			// Amount to be paid
			$basket['AccountCredit']['amount'],
			// Items
			[],
			// Extras
			[
				'language' => Configure::read('Config.language'),
				'card' => $payment,
				'address' => $basket['CustomerAddress'],
				'user' => ['email' => $basket['CustomerAddress']['email']],
				'description' => __d(
					'buzz_diary_accounts',
					'%s Account Credit %d',
					[Configure::read('SiteSetting.site_title'), $basketId]
				)
			],
			0,
			Configure::read('Transactions.gateway')
		);
	}

/**
 * Complete the credit transaction (after payment received)
 *
 * @param int $basketId Basket ID
 * @param int $paymentRequestId Payment Request ID
 * @return void Redirects to confirmation page
 */
	protected function _completeCreditTransaction($basketId, $paymentRequestId) {
		$AccountPaymentRequest = $this->{$this->modelClass};
		if (empty($basketId) || empty($paymentRequestId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$AccountPaymentRequest->AccountCredit->setBasketId($basketId);
		$this->Session->write('AccountCredit.hash', $AccountPaymentRequest->AccountCredit->hashBasketId());

		$AccountPaymentRequest->completePaymentRequest($paymentRequestId, $basketId, true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	/**
	 * Credit confirmation page.
	 *
	 * @return void
	 */
	public function confirmation($id) {
		$AccountPaymentRequest = $this->{$this->modelClass};

		if ($AccountPaymentRequest->AccountCredit->hashBasketId($id) === $this->Session->read('AccountCredit.hash')) {

			$this->Session->delete('AccountCredit.hash');
			$this->Session->delete('AccountPaymentRequest.id');

			$credit = $AccountPaymentRequest->AccountCredit->readForView($id);

			if (Configure::check('BuzzDiaryAccounts.payment_request_confirmation_page_id') === true) {
				$this->assignPage(Configure::read('BuzzDiaryAccounts.payment_request_confirmation_page_id'));
			}

			$this->_loadPageContent();
			$this->set(compact('credit'));
			$this->view = 'BuzzDiaryAccounts.confirmation';

		} else {

			throw new ForbiddenException();

		}

		return;
	}

/**
 * Generic payment gateway callback.
 *
 * @param int $basketId Basket ID
 * @param int $paymentRequestId Payment Request ID
 * @return void
 */
	public function payment_callback($basketId, $paymentRequestId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completeCreditTransaction($basketId, $paymentRequestId);
		} else {
			$this->Session->setFlash(
				__d('buzz_diary_accounts', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect('/');
		}
	}

}
