<?php

App::uses('BuzzDiaryAccountsAppController', 'BuzzDiaryAccounts.Controller');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class AccountCreditsController extends BuzzDiaryAccountsAppController implements PurchasableInterface {

	/**
	 * Define actions that will require account authentication.
	 *
	 * @var array
	 */
	public $requiresAccountAuth = [
		'add',
		'confirmation',
		'get_credits',
		'get_next_credits',
		'history',
		'index'
	];

	/**
	 * Allowed admin actions
	 *
	 * @var array
	 */
	public $adminActions = [];

	/**
	 * Components
	 *
	 * @var array
	 */
	public $components = array(
		'BuzzPurchase.EcommerceTracking',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	);

	/**
	 * Account credits overview tab
	 *
	 * @return void
	 */
	public function index() {
		$this->loadModel('BuzzDiaryAccounts.AccountPaymentRequest');

		$userId = $this->Auth->user('id');

		$data = $this->AccountPaymentRequest->find(
			'all',
			array(
				'conditions' => array(
					'AccountPaymentRequest.account_id' => $userId,
					'AccountPaymentRequest.account_credit_id' => null,
					'AccountPaymentRequest.is_removed' => false
				),
				'order' => array(
					'AccountPaymentRequest.created' => 'DESC',
					'AccountPaymentRequest.id' => 'DESC'
				),
				'limit' => 3
			)
		);

		if ($this->Session->check('BuzzDiaryAccounts.tracking')) {
			$this->set('ecommerceTracking', $this->Session->read('BuzzDiaryAccounts.tracking'));
			$this->Session->delete('BuzzDiaryAccounts.tracking');
		}

		$this->_loadPageContent();
		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts.index';

		return;
	}

/**
 * Add credit to account.
 *
 * @return void
 * @throws UnauthorizedException
 */
	public function add() {
		$AccountCredit = $this->{$this->modelClass};

		if (empty($this->Auth->user('can_add_credit'))) {
			throw new UnauthorizedException('You do not have permission to add credit');
		}

		if (!empty($this->request->data)) {

			if ($AccountCredit->createCredit($this->Auth->user('id'), $this->request->data) === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_diary_accounts', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Populate the form with existing account details.
			$accountId = $this->Auth->user('id');
			$this->request->data = $AccountCredit->Account->readForEdit($accountId);
			$this->request->data['CustomerAddress']['id'] = null;
			$this->request->data['CustomerAddress']['email'] = $this->request->data['Account']['email'];
			$this->request->data['CustomerAddress']['first_name'] = $this->request->data['Account']['first_name'];
			$this->request->data['CustomerAddress']['last_name'] = $this->request->data['Account']['last_name'];

			$this->request->data['AccountCredit']['name'] = $this->request->data['Account']['full_name'];

		}

		$this->_loadPageContent();
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();
		$this->view = 'BuzzDiaryAccounts.add';
	}

/**
 * Stripe payment
 *
 * @return void
 * @throws MethodNotAllowedException
 */
	public function stripe_payment() {
		if (Configure::read('Transactions.gateway') !== 'stripe_checkout') {
			throw new MethodNotAllowedException('This site is not configured to use this action');
		}

		$AccountCredit = $this->{$this->modelClass};
		$id = $AccountCredit->getBasketId();
		$credit = $AccountCredit->find(
			'first',
			[
				'contain' => ['CustomerAddress.Country'],
				'conditions' => ['AccountCredit.id' => $id]
			]
		);

		$stripeTransaction = $this->_takePayment($id, $credit);

		if ($stripeTransaction['result'] === false || empty($stripeTransaction['transaction_id'])) {
			$this->Session->setFlash(
				__d('buzz_diary_accounts', 'There was a problem processing your payment, please check your details and try again'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'add']);
		}

		$this->loadModel('Transactions.Transaction');
		$transaction = $this->Transaction->findById(
			$stripeTransaction['transaction_id']
		);

		$stripeKey = Configure::read('Transactions.stripe.live.publishable_key');
		if (Configure::read('db.config') !== 'live') {
			$stripeKey = Configure::read('Transactions.stripe.dev.publishable_key');
		}

		$this->set(compact('transaction', 'stripeKey'));
		// Use the AJAX layout to improve the redirect transition to Stripe.
		$this->layout = 'ajax';
	}

/**
 * Processes the payment and triggers the completion of the payment on success
 *
 * @return void
 */
	protected function _pay() {
		$AccountCredit = $this->{$this->modelClass};

		// Get the latest basket.
		$id = $AccountCredit->getBasketId();
		$basket = $AccountCredit->find(
			'first',
			[
				'contain' => ['CustomerAddress.Country'],
				'conditions' => ['AccountCredit.id' => $id]
			]
		);

		if ($basket['AccountCredit']['amount'] <= 0) {
			return $this->_completeCreditTransaction($id);
		}

		if (Configure::read('Transactions.gateway') === 'stripe_checkout') {
			return $this->redirect(['action' => 'stripe_payment']);
		}

		$payment = [];

		if (Configure::read('BuzzPurchase.onsite') === true) {
			$payment = $this->request->data['Payment'];

			// Split name into first and second names for payment gateways.
			$fullName = trim($payment['card_name']);
			$splitName = explode(' ', $fullName);
			$payment['last_name'] = array_pop($splitName);
			$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
		}

		$transaction = $this->_takePayment($id, $basket, $payment);

		if (! empty($transaction['result'])) {
			// Payment taken successfully, complete purchase.
			return $this->_completeCreditTransaction($id);
		}

		$this->Session->setFlash(
			__d('buzz_diary_accounts', 'There was a problem processing your payment, please check your details and try again'),
			'flash_fail'
		);
	}

/**
 * Take a payment
 *
 * @param int $basketId Basket ID
 * @param array $basket Basket data
 * @param array $payment Payment data
 * @return mixed The response from Transactions::takePayment()
 */
	protected function _takePayment($basketId, $basket, $payment = []) {
		$this->loadModel('BuzzSites.Site');
		$site = $this->Site->getActiveSite();

		return $this->Transactions->takePayment(
			// Return URLs
			[
				'return' => Router::url(['action' => 'payment_callback', $basketId], true),
				'cancel' => Router::url(['action' => 'add'], true)
			],
			// Calling record
			[
				'model' => 'BuzzDiaryAccounts.AccountCredit',
				'model_id' => $basket['AccountCredit']['id']
			],
			// Amount to be paid
			$basket['AccountCredit']['amount'],
			// Items
			[],
			// Extras
			[
				'language' => Configure::read('Config.language'),
				'card' => $payment,
				'address' => $basket['CustomerAddress'],
				'user' => ['email' => $basket['CustomerAddress']['email']],
				'description' => __d(
					'buzz_diary_accounts',
					'%s Account Credit %d',
					[Configure::read('SiteSetting.site_title'), $basketId]
				),
				'site_id' => $site['Site']['api_site_id']
			],
			0,
			Configure::read('Transactions.gateway')
		);
	}

/**
 * Complete the credit transaction (after payment received)
 *
 * @param int $basketId Basket ID
 * @return void Redirects to confirmation page
 */
	protected function _completeCreditTransaction($basketId) {
		$AccountCredit = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$AccountCredit->setBasketId($basketId);
		$this->Session->write('AccountCredit.hash', $AccountCredit->hashBasketId());

		$AccountCredit->completeCredit(true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	/**
	 * Credit confirmation page.
	 *
	 * @return void
	 */
	public function confirmation($id) {
		$AccountCredit = $this->{$this->modelClass};

		if ($AccountCredit->hashBasketId($id) === $this->Session->read('AccountCredit.hash')) {

			$this->Session->delete('AccountCredit.hash');

			$data = $AccountCredit->readForView($id);

			if (empty($data['AccountCredit']['sales_ref'])) {
				throw new InternalErrorException(__d('buzz_diary_accounts', 'Something went wrong'));
			} else {
				$this->EcommerceTracking->addItem(
					'Credit',
					'CREDIT',
					$data['AccountCredit']['amount'],
					1,
					['category' => 'Pro Flyer']
				);
				$this->Session->write(
					'BuzzDiaryAccounts.tracking',
					$this->EcommerceTracking->transaction(
						$data['AccountCredit']['sales_ref'],
						$data['AccountCredit']['amount'],
						CakeNumber::defaultCurrency(),
						0
					)
				);
				$this->Session->setFlash(
					__d(
						'buzz_diary_accounts',
						'Thank You! %s has been added to your account (your sales reference number is: %s)',
						[
							CakeNumber::currency($data['AccountCredit']['amount']),
							$data['AccountCredit']['sales_ref']
						]
					),
					'flash_success'
				);
			}
			return $this->redirect(['action' => 'index']);

		} else {

			throw new ForbiddenException();

		}

		return;
	}

	/**
	 * Populate lookups.
	 *
	 * @return void
	 */
	protected function _populateLookups() {
		$AccountCredit = $this->{$this->modelClass};

		$countries = $AccountCredit->CustomerAddress->Country->translatedList();
		$usStates = $AccountCredit->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));
		return;
	}

/**
 * Generic payment gateway callback.
 *
 * @param int $basketId Basket ID
 * @return void
 */
	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completeCreditTransaction($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_diary_accounts', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'add'
			]);
		}
	}

	/**
	 * Credit history.
	 *
	 * @return void
	 */
	public function history() {
		$Account = $this->{$this->modelClass}->Account;

		$apiUserId = $this->Auth->user('api_reference');

		$limit = 10;
		$data = $Account->getCredits($apiUserId, $limit);
		$paged = !(count($data) < $limit);

		$this->_loadPageContent();
		$this->set(compact('data', 'limit', 'paged'));
		$this->view = 'BuzzDiaryAccounts.history';

		return;
	}

	/**
	 * AJAX method for credit history.
	 *
	 * @return void
	 */
	public function get_credits() {
		$Account = $this->{$this->modelClass}->Account;

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getCredits($apiUserId, 3);

		$this->set(compact('data'));
		$this->view = 'BuzzDiaryAccounts./Elements/credits_table';
		$this->layout = 'ajax';

		return;
	}

/**
 * Loads the next page of bookings
 *
 * @param int $page
 * @param int $limit
 * @return void
 */
	public function get_next_credits($page = 0, $limit = 10) {
		$Account = $this->{$this->modelClass}->Account;

		$apiUserId = $this->Auth->user('api_reference');

		$data = $Account->getCredits($apiUserId, $limit, $page);

		if (empty($data)) {
			// No more credits to display so just die so that there is no response.
			die();
		}

		$paged = !(count($data) < $limit);

		$this->set(compact('data', 'page', 'paged', 'limit'));
		$this->view = 'BuzzDiaryAccounts./Elements/credits_table';
		$this->layout = 'ajax';

		return;
	}

}
