<?php

App::uses('BuzzCustomersAppModel', 'BuzzCustomers.Model');

class CustomerAddress extends BuzzCustomersAppModel {

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Country' => array(
			'className' => 'BuzzCustomers.Country'
		),
		'UsState',
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'first_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'last_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'address_line_1' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'address_line_2' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'city' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'county' => array(
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'country_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'us_state_id' => array(
			'required' => array(
				'rule' => array('validateRequiredByCountry', 226),
				'message' => 'Required'
			)
		),
		'postcode' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'postcode' => array(
				'rule' => 'validatePostcode',
				'message' => 'Not a valid postcode'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 20),
				'message' => 'No more than 20 characters long'
			)
		),
		'telephone' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'telephone' => array(
				'rule' => '/^\+?[\d\ ]+$/',
				'message' => 'Not a valid telephone number'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 20),
				'message' => 'No more than 20 characters long'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			)
		)
	);

	/**
	 * Constructor
	 *
	 * @param bool|int|string|array $id Set this ID for this model on startup,
	 * can also be an array of options, see above.
	 * @param string $table Name of database table to use.
	 * @param string $ds DataSource connection name.
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);
		$this->virtualFields['full_name'] = 'CONCAT(`' . $this->alias . '`.`first_name`, \' \', `' . $this->alias . '`.`last_name`)';
		return;
	}

	/**
	 * Custom validation method for checking a postcode.
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validatePostcode(array $data) {
		$check = array_pop($data);

		if (!empty($this->data[$this->alias]['country_id'])) {

			if ((int)$this->data[$this->alias]['country_id'] === 225) {
				// Check the postcode is valid in the UK.
				return preg_match('|^\D{1,2}\d{1,2}\D? \d\D{2}$|i', $check) === 1;
			}

		}

		return true;
	}

	/**
	 * Custom validation method for checking a field has been provided for a
	 * particular country.
	 *
	 * @param array $data
	 * @param int $countryId
	 * @return bool
	 */
	public function validateRequiredByCountry(array $data, $countryId) {
		$check = array_pop($data);

		if (!empty($this->data[$this->alias]['country_id'])) {

			if ((int)$this->data[$this->alias]['country_id'] === (int)$countryId) {
				// Check required field has been supplied.
				return !empty($check);
			}

		}

		return true;
	}

	public function beforeSave($options = []) {
		if (parent::beforeSave($options) === true) {

			// For non-US addresses we don't want to save a state.
			if (!empty($this->data[$this->alias]['country_id']) && (int)$this->data[$this->alias]['country_id'] !== 226) {
				$this->data[$this->alias]['us_state_id'] = null;
			}

			// Make all postcodes uppercase.
			if (!empty($this->data[$this->alias]['postcode'])) {
				$this->data[$this->alias]['postcode'] = strtoupper($this->data[$this->alias]['postcode']);
			}

			return true;
		}
		return false;
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		// Return all postcodes in uppercase (this is for backwards
		// compatability for postcodes saved before we forced all postcodes
		// to be saved in uppercase in `self::beforeSave()`).
		foreach ($results as &$result) {
			if (!empty($result[$this->alias]['postcode'])) {
				$result[$this->alias]['postcode'] = strtoupper($result[$this->alias]['postcode']);
			}
		}

		return $results;
	}

	/**
	 * Returns the default country ID to use for addresses.
	 *
	 * @return int
	 */
	public static function getDefaultCountry() {
		if (Configure::check('BuzzCustomers.default_country')) {
			return Configure::read('BuzzCustomers.default_country');
		} else {
			// Fallback default if config not setup. Use the UK.
			return 225;
		}
	}

	/**
	 * Returns the default US state ID to use for addresses.
	 *
	 * @return int
	 */
	public static function getDefaultUsState() {
		if (Configure::check('BuzzCustomers.default_state')) {
			return Configure::read('BuzzCustomers.default_state');
		} else {
			return;
		}
	}

	/**
	 * Returns an address array for use with the API.
	 *
	 * @return array
	 */
	public static function generateAddressArray($address) {
		$data = array(
			'address_line_1' => $address['address_line_1'],
			'address_line_2' => $address['address_line_2'],
			'address_line_3' => $address['city'],
			'address_line_4' => $address['Country']['name'],
			'postcode' => $address['postcode']
		);
		// For US addresses append the state to the city.
		if ((int)$address['country_id'] === 226 && !empty($address['UsState']['name'])) {
			$data['address_line_3'] .= ', ' . $address['UsState']['name'];
		}
		return $data;
	}

}
