<?php

App::uses('CakeEmail', 'Network/Email');

/**
 * Business logic for user accounts
 */
class EvUser extends AppModel {

	public $useTable = "users";

	public $displayField = "email";

	public $order = array('email' => 'ASC');

	public $belongsTo = array(
		'UserGroup'
	);

	/**
	 * Ensure the password is encrypted if present before it's saved to the DB
	 *
	 * @param array $options beforeSave options
	 * @return void
	 */
	public function beforeSave($options = array()) {
		if (isset($this->data[$this->alias]['password']) && ! empty($this->data[$this->alias]['password'])) {

			$this->data[$this->alias]['password'] = AuthComponent::password($this->data[$this->alias]['password']);
		} else {

			unset($this->data[$this->alias]['password']);
		}

		return;
	}

	public $validate = array(
		'user_group_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'User group cannot be blank'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Email cannot be left blank'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'You must enter a valid email address'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'A user with that email address already exists'
			)
		),
		'password' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Password cannot be left blank',
				'on' => 'create'
			)
		),
		'confirm_password' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'You must confirm your password',
				'on' => 'create'
			),
			'matches' => array(
				'rule' => array('checkMatches', 'password'),
				'message' => 'The passwords do not match'
			)
		)
	);

	/**
	 * Validate password rules
	 */
	public function validatePassword() {
		return array(
			'password' => array(
				'required' => array(
					'rule' => 'notEmpty',
					'message' => 'Password cannot be left blank'
				)
			),
			'confirm_password' => array(
				'required' => array(
					'rule' => 'notEmpty',
					'message' => 'You must confirm your password'
				),
				'matches' => array(
					'rule' => array('checkMatches', 'password'),
					'message' => 'The passwords do not match'
				)
			)
		);
	}

	public function validateEdit() {
		$rules = $this->validate;

		unset($rules['password']);
		unset($rules['confirm_password']);

		return $rules;
	}

	/**
	 * Validation rules for a login form submission
	 *
	 * @return array
	 */
	public function validateLogin() {
		return array(
			'email' => array(
				'required' => array(
					'rule' => 'notEmpty',
					'message' => 'Email cannot be left blank.'
				),
				'email' => array(
					'rule' => 'email',
					'message' => 'You must enter a valid email address'
				)
			),
			'password' => array(
				'required' => array(
					'rule' => 'notEmpty',
					'message' => 'Password cannot be left blank'
				)
			)
		);
	}

	/**
	 * Returns true if an email already exists
	 *
	 * @param array $check email field to check
	 * @return true if email doesn't exists, otherwise false
	 */
	public function checkIsRegistered($check) {
		App::uses('User', 'Model');

		$value = array_pop($check);

		$user = $this->User->findByEmail($value);

		return empty($user);
	}

	/**
	 * Gets the user and all associated data out that is needed for login
	 *
	 * @param  int $id  The ID of the user
	 * @return array 	User array
	 */
	public function getUserForLogin($id) {
		$fullUser = $this->find('first', array(
				'conditions' => array(
					$this->alias . '.id' => $id
				),
				'contain' => array(
					'UserGroup'
				)
			)
		);

		return $fullUser;
	}

	/**
	 * Sets a password reset code for a user.
	 *
	 * @param integer $userId
	 * @return boolean - returns false if the user cannot be found
	 */
	public function resetPassword($userId) {
		$user = $this->findById($userId);

		if (!empty($user)) {

			$code = $user['User']['password_reset_code'];

			if (empty($code)) {
				$code = substr(md5(uniqid()), 0, 10);
				$this->id = $userId;
				$this->saveField('password_reset_code', $code);
			}

			App::uses('EvUserPasswordResetListener', 'EvCore.Lib/Event');
			$this->getEventManager()->attach(new EvUserPasswordResetListener());
			$event = new CakeEvent('Model.User.passwordReset', $this, array(
					'Model' => $this->alias,
					'User' => $user,
					'code' => $code,
					'url' => Router::url(
						array(
							'controller' => 'users',
							'action' => 'password_reset_callback',
							$code
						),
						true
					)
				)
			);
			$this->getEventManager()->dispatch($event);

			return true;

		}

		return false;
	}

	/**
	 * Returns true if the current user has permission to preview content.
	 *
	 * @param bool
	 */
	public static function canPreviewContent() {
		$userGroup = CakeSession::read('Auth.User.User.user_group_id');
		return !empty($userGroup) && $userGroup < 3;
	}

}
