<?php

App::uses('AppModel', 'Model');

/**
 * Business rules for Pages
 */
class EvPage extends AppModel {

	public $useTable = "pages";

	public $displayField = 'title';

/**
 * Define the number of associated files/documents.
 *
 * @var int|array
 */
	public $documentSlots = -1;

/**
 * @var array Behaviours
 */
	public $actsAs = array(
		'BuzzAdverts.AdvertHost',
		'BuzzTranslate.Translatable',
		'Routable.Routable',
		'MetaData.Meta',
		'Navigation.Navigatable',
		'EvCore.Tabbed'
	);

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Theme' => array(
			'className' => 'BuzzTheme.Theme'
		)
	);

/**
 * @var array Validation rules
 */
	public $validate = array(
		'title' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters'
			)
		),
		'internal_description' => array(
			'max' => array(
				'rule' => array('maxLength', 100),
				'message' => 'No more than 100 characters',
				'allowEmpty' => true
			)
		),
		'theme_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'video_url' => array(
			'required' => array(
				'rule' => array('url', true),
				'message' => 'Invalid URL',
				'allowEmpty' => true
			)
		)
	);

/**
 * Image slots
 *
 * @var array
 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		]
	];

/**
 * Constructor
 */
	public function __construct($id = false, $table = null, $ds = null) {
		// If we're using EvTinyMceImage we want to add extra image slots.
		if (CakePlugin::loaded('EvTinyMceImage') === true) {
			$this->imageSlots['other'] = [
				'slots' => -1,
				'fields' => false
			];
		}
		parent::__construct($id, $table, $ds);

		// If BuzzGallery loaded associate activities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}

		// If BuzzSites loaded associate with the sites table. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzSites is excluded.
		if (CakePlugin::loaded('BuzzSites') === true) {
			$this->belongsTo['Site'] = array(
				'className' => 'BuzzSites.Site'
			);
		}

		if (Configure::check('app.primary_nav') === true) {
			$this->parentMenu = Configure::read('app.primary_nav');
		}

		// Set the multi-experience activity.
		$this->_multiExperience = Configure::read('BuzzBookings.multi_experience');
	}

/**
 * Model pre-save logic
 *
 * @param array $options
 * @return bool
 */
	public function beforeSave($options = array()) {
		// remove trailing empty paragraph tag from body
		if (isset($this->data[$this->alias]['body']) && !empty($this->data[$this->alias]['body'])) {
			$this->data[$this->alias]['body'] = preg_replace("/(<p[^>]*>[\s|&nbsp;]*<\/p>\n?)*$/", '', $this->data[$this->alias]['body']);
		}

		// If this page has a redirect setup for it then we want to remove the
		// Routable behaviour and not save a route. This page content cannot be
		// directly accessed.
		if (!empty($this->data[$this->alias]['redirect'])) {
			unset($this->data['Route']);
			$this->Behaviors->unload('Routable.Routable');
		}

		return parent::beforeSave($options);
	}

/**
 * Read for edit.
 *
 * @param int $id
 * @param array $params
 * @return array
 */
	public function readForEdit($id, $params = []) {
		$params['contain']['Advert'] = array(
			'AdvertRelation',
			'AdvertTemplate',
			'Image',
			'order' => array('AdvertRelation.sequence' => 'ASC')
		);
		return parent::readForEdit($id, $params);
	}

/**
 * Read for view.
 *
 * @param int $id
 * @param array $params
 * @return array
 */
	public function readForView($id, $params = []) {
		$data = parent::readForView($id, $params);

		// Remove hidden adverts, these shouldn't show.
		if (!empty($data['Advert'])) {
			foreach ($data['Advert'] as $key => $advert) {
				if ($advert['is_active'] === false) {
					unset($data['Advert'][$key]);
				}
			}
		}

		if (Configure::check('app.primary_nav') === true) {
			$this->parentMenu = Configure::read('app.primary_nav');
		}

		// Get the child menu (if the current page exists in the menu).
		if (!empty($data)) {
			$menuItem = $this->getMenuItem($data);
			$data['Menu'] = $this->_getPageMenu($menuItem);
		}

		return $data;
	}

/**
 * Returns the child menu of a specific menu item.
 *
 * @param array $menuItem
 * @return array
 */
	protected function _getPageMenu($menuItem) {
		$data = [];
		if (!empty($menuItem)) {

			$Menu = ClassRegistry::init('Navigatable.Menu');

			$conditions = array(
				'Menu.rght <' => $menuItem['rght'],
				'Menu.lft >' => $menuItem['lft'],
				'Menu.is_active' => true,
				'Menu.is_menu_hidden' => false
			);

			$order = 'Menu.lft ASC';

			$menu = $Menu->find('all', compact('conditions', 'order'));
			$data = Hash::extract($menu, '{n}.Menu');

			// If the menu item has no children, we want to get it's siblings.
			if (empty($data) && !empty($menuItem['parent_id'])) {
				$parent = $Menu->findById($menuItem['parent_id']);
				// Make sure we're not at the top of the tree before fetching
				// the siblings.
				if (!empty($parent['Menu']['parent_id'])) {
					$conditions['Menu.rght <'] = $parent['Menu']['rght'];
					$conditions['Menu.lft >'] = $parent['Menu']['lft'];
					$menu = $Menu->find('all', compact('conditions', 'order'));
					$data = Hash::extract($menu, '{n}.Menu');
				}
			}

		}
		return $data;
	}

}
