<?php

App::uses('AppController', 'Controller');

class EvPagesController extends AppController {

	public $components = array(
		'BuzzTranslate.Translation' => array(
			'fields' => array(
				'title',
				'body' => array(
					'type' => 'html'
				)
			),
			'menu' => true,
			'meta' => true
		),
		'BuzzAdverts.Adverts'
	);

	public function __construct($request = null, $response = null) {
		$this->adminActions[] = 'admin_adverts';
		// If we're using EvTinyMceImage we want to load the plugin's component.
		if (CakePlugin::loaded('EvTinyMceImage') === true) {
			$this->components['EvTinyMceImage.TinyMCE'] = [
				'model_alias' => 'Page'
			];
		}
		parent::__construct($request, $response);
	}

	/**
	 * Controller name
	 *
	 * @var string
	 */
	public $name = 'Pages';

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'home',
			'view',
			'locale'
		));
	}

	protected function _adminIndexColumns() {
		return array(
			'Page.id' => array(
				'label' => 'ID',
				'type' => 'integer',
			),
			'Page.title' => array(
				'label' => 'Title',
				'type' => 'string',
			),
			'Page.internal_description' => array(
				'label' => 'Description',
				'type' => 'string',
			),
			'Page.is_active' => array(
				'label' => 'Active',
				'type' => 'boolean'
			),
			'Page.created' => array(
				'label' => 'Created',
				'type' => 'datetime'
			),
			'Page.modified' => array(
				'label' => 'Modified',
				'type' => 'datetime'
			)
		);
	}

	/**
	 * Admin add/edit
	 *
	 * @param int $id
	 * @return void
	 */
	public function admin_edit($id = null) {
		$Page = &$this->{$this->modelClass};

		if ((int)$id === 1) {
			// Modify the admin form for the homepage content.
			$Page->setMaxTabs(0);
			// Remove translated fields for fields we are going to remove
			// after calling parent.
			$this->Translation->settings['fields'] = [];
		} elseif (Configure::check('BuzzContact.page_id') === true && Configure::read('BuzzContact.page_id') == $id) {
			// Reduce the number of tabs for the contact page and make required fields.
			$Page->setMaxTabs(2);
			$Page->Tab->validator()->add('title', 'required', array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			));
		}

		parent::admin_edit($id);

		if (Configure::read('EvCore.Homepage.show_body') !== true && (int)$id === 1) {
			unset($this->viewVars['fields']['Page.body']);
		}

		$this->Translation->adminTranslationFormFields();
		$this->Adverts->setupAdvertAdminFields();

		return;
	}

	/**
	 * Admin form fields
	 *
	 * @return array
	 */
	protected function _adminFormFields() {
		$Page = $this->{$this->modelClass};
		$fields = parent::_adminFormFields();

		$fields['Page.body']['type'] = 'html';

		// Only Evoluted admin users should be able to see the view_template
		// and redirect fields.
		if ($this->Auth->user('UserGroup.id') > 1) {
			unset($fields[$Page->alias . '.view_template']);
			unset($fields[$Page->alias . '.redirect']);
			// Internal description should be display only.
			$fields[$Page->alias . '.internal_description']['displayonly'] = 'displayonly';
		}

		$fields[$Page->alias . '.video_url']['label'] = 'Video URL';

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$galleryFields = array(
				'Page.gallery_id' => array(
					'type' => 'select',
					'label' => __('Gallery')
				)
			);
			$fields = ArrayUtil::addAfter($fields, 'Page.theme_id', $galleryFields);
		}

		if (CakePlugin::loaded('BuzzSites') === true) {
			$siteField = array(
				'Page.site_id' => array(
					'type' => 'select',
					'label' => __('Site'),
					'after' => ' <span class="help-text">This is optional and is only used for certain view templates to determine which site to associate the page with.</span>'
				)
			);
			$fields = ArrayUtil::addAfter($fields, 'Page.theme_id', $siteField);
		}

		return $fields;
	}

	/**
	 * Populate admin lookups
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$Page = $this->{$this->modelClass};

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->set('galleries', $Page->Gallery->find('list'));
		}

		if (CakePlugin::loaded('BuzzSites') === true) {
			$this->set('sites', $Page->Site->find('list'));
		}

		$this->set('themes', $Page->Theme->find('list'));

		return;
	}

	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		if (!empty($id)) {
			$toolbar['Adverts'] = [
				'url' => ['action' => 'adverts', $id],
				'icon' => 'bars'
			];
			$toolbar['Preview'] = [
				'url' => ['action' => 'view', $id, 'admin' => false],
				'icon' => 'search',
				'target' => '_blank'
			];
		}

		return $toolbar;
	}

	/**
	 * Displays a view
	 *
	 * @param mixed What page to display
	 * @return void
	 */
	public function display() {
		$path = func_get_args();

		$count = count($path);
		if (!$count) {
			$this->redirect('/');
		}
		$page = $subpage = $title_for_layout = null;

		if (!empty($path[0])) {
			$page = $path[0];
		}
		if (!empty($path[1])) {
			$subpage = $path[1];
		}
		if (!empty($path[$count - 1])) {
			$title_for_layout = Inflector::humanize($path[$count - 1]);
		}
		$this->set(compact('page', 'subpage', 'title_for_layout'));
		$this->render(implode('/', $path));
	}

	/**
	 * Method required by CakePHP. You can ignore this.
	 */
	public function home() {
	}

	public function view($id) {
		$data = $this->Page->readForView($id);

		if (!$data) {
			throw new NotFoundException();
		} elseif (!empty($data['Page']['redirect'])) {
			return $this->redirect($data['Page']['redirect']);
		}

		if ($data['Page']['is_active'] === false) {
			$this->Session->setFlash(__d('ev_core', 'Preview: This page has not been published yet'), 'flash_info');
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($data['Page']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->Page->Gallery->readForView($data['Page']['gallery_id'])
			);
		}

		// For BuzzSites enabled sites load the site details if a site has been set for this page.
		if (CakePlugin::loaded('BuzzSites') === true && !empty($data['Page']['site_id'])) {
			$this->set(
				'pageSite',
				$this->Page->Site->readForView($data['Page']['site_id'])
			);
		}

		$this->set('data', $data);
		$this->Meta->set($data);
		$this->Meta->canonical(['action' => 'view', $id]);
		$this->view = 'EvCore.view';

		// Check if we're overriding the view template.
		if (!empty($data['Page']['view_template'])) {

			$this->view = $data['Page']['view_template'];

			// Check if we need to grab any extra data for this template.
			$template = Inflector::camelize($data['Page']['view_template']);
			if (method_exists($this, '_get' . $template . 'Extras')) {
				$method = '_get' . $template . 'Extras';
				$this->$method();
			}
		}

		return;
	}

	/**
	 * Set default homepage view variables.
	 *
	 * @return void
	 */
	protected function _getHomepageExtras() {
		if (CakePlugin::loaded('BuzzBookings') === true) {
			$this->loadModel('BuzzBookings.Activity');
			$activities = $this->Activity->translatedList([
				'joins' => [
					[
						'table' => 'activity_packages',
						'alias' => 'ActivityPackage',
						'type' => 'INNER',
						'conditions' => 'ActivityPackage.activity_id = Activity.id'
					]
				],
				'conditions' => [
					'Activity.api_reference IS NOT NULL',
					'Activity.api_reference <>' => '',
					'Activity.is_hidden_activity' => false,
					'ActivityPackage.is_active' => true
				],
				'order' => [
					'Activity.sequence' => 'ASC',
					'Activity.created' => 'ASC'
				]
			]);
			if ($this->Activity->ActivityPackage->hasPrivateHirePackages() === true) {
				$activities += [-1 => __d('ev_core', 'Group / Party Booking')];
			}
			$this->set('activities', $activities);
			// Include listed activities
			$this->set('listActivities', $this->Activity->getListed());
		}

		if (CakePlugin::loaded('BuzzVouchers') === true) {
			$this->loadModel('BuzzVouchers.VoucherActivity');
			$this->set('voucherActivities', $this->VoucherActivity->listListed());
		}

		if (CakePlugin::loaded('BuzzSlides') === true) {
			$this->loadModel('BuzzSlides.Slide');
			$this->set('slides', $this->Slide->getSlides());
		}

		// Include calls to action.
		$this->loadModel('EvCore.HomepageCta');
		$this->set('ctas', $this->HomepageCta->getCtas());

		return;
	}

	/**
	 * Handles the translation of strings for JavaScript.
	 *
	 * @return void
	 */
	public function locale() {
		$this->view = 'EvCore.locale';
		$this->layout = 'ajax';
		return;
	}

}
