<?php

App::uses('BuzzCompetitionsAppModel', 'BuzzCompetitions.Model');
App::uses('CompetitionEntryState', 'BuzzCompetitions.Model');

class CompetitionEntry extends BuzzCompetitionsAppModel {

	public $order = [
		'CompetitionEntry.id' => 'DESC'
	];

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Competition' => array(
			'className' => 'BuzzCompetitions.Competition'
		),
		'CompetitionCategory' => array(
			'className' => 'BuzzCompetitions.CompetitionCategory'
		),
		'CompetitionEntryState' => array(
			'className' => 'BuzzCompetitions.CompetitionEntryState'
		),
		'TeamCountry' => array(
			'className' => 'BuzzCustomers.Country'
		),
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		)
	);

	public $hasMany = array(
		'CompetitionEntryParticipant' => array(
			'className' => 'BuzzCompetitions.CompetitionEntryParticipant'
		)
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'competition_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'competition_category_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'int' => array(
				'rule' => 'naturalNumber'
			)
		),
		'team_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters long'
			)
		),
		'team_country_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'int' => array(
				'rule' => 'naturalNumber'
			)
		),
		'alternate_name' => array(
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters long',
				'allowEmpty' => true
			)
		)
	);

/**
 * Read for edit
 *
 * @param int   $id     Record ID
 * @param array $params Query parameters
 * @return array
 */
	public function readForEdit($id, $params = []) {
		$defaults = array(
			'contain' => array(
				'CustomerAddress' => array(
					'Country',
					'UsState'
				),
				'Competition',
				'CompetitionCategory',
				'CompetitionEntryParticipant',
				'TeamCountry'
			)
		);
		$params = Hash::merge($defaults, $params);

		return parent::readForEdit($id, $params);
	}

/**
 * Get the current basket
 *
 * @param int $competitionEntryId
 * @return array
 */
	public function getBasket($competitionEntryId = null) {
		$competitionEntryId = $competitionEntryId ?: $this->getBasketId();
		if (empty($competitionEntryId)) {
			return [];
		}
		$params = array(
			'conditions' => array(
				'CompetitionEntry.competition_entry_state_id' => CompetitionEntryState::UNPAID
			)
		);
		return $this->readForView($competitionEntryId, $params);
	}

/**
 * Get a completed competition entry to display to the customer.
 *
 * @param int $id Competition entry ID
 * @return array
 */
	public function getPurchase($id) {
		$params = array(
			'conditions' => array(
				'CompetitionEntry.competition_entry_state_id' => array(
					CompetitionEntryState::COMPLETE,
					CompetitionEntryState::API_FAILED
				)
			)
		);
		return $this->readForView($id, $params);
	}

/**
 * Save a competition entry,
 *
 * @param int   $competitionId Competition ID
 * @param array $data          Entry data
 * @return bool
 */
	public function submitEntry($competitionId, array $data) {
		$competition = $this->Competition->find(
			'first',
			array(
				'contain' => array(
					'CompetitionCategory'
				),
				'conditions' => array(
					'Competition.id' => $competitionId
				)
			)
		);

		$participants = Hash::combine(
			$competition,
			'CompetitionCategory.{n}.id',
			'CompetitionCategory.{n}.participants'
		);

		if (
			!empty($data['CompetitionEntry']['competition_category_id'])
			&& !empty($participants[$data['CompetitionEntry']['competition_category_id']])
		) {
			$data['CompetitionEntryParticipant'] = array_slice(
				$data['CompetitionEntryParticipant'],
				0,
				$participants[$data['CompetitionEntry']['competition_category_id']]
			);
		}

		// Make sure that we don't create a new entry if one remains in the
		// basket.
		$data['CompetitionEntry']['id'] = $this->getBasketId();

		$data['CompetitionEntry']['competition_id'] = $competitionId;
		$data['CompetitionEntry']['competition_entry_state_id'] = CompetitionEntryState::UNPAID;
		$data['CompetitionEntry']['total_cost'] = $competition['Competition']['price'];
		$data['CompetitionEntry']['newsletter_opt_in'] = ! empty($data['CompetitionEntry']['newsletter_opt_in']);

		$result = $this->saveAssociated($data, ['deep' => true]) !== false;

		// Set the 'basket' ID.
		if ($result === true) {
			$this->setBasketId($this->id);
		}

		return $result;
	}

/**
 * Complete the competition entry
 *
 * @param bool $paid True if payment successfully made
 * @return bool
 */
	public function completeEntry($paid) {
		$id = $this->getBasketId();

		$basket = $this->getBasket($id);

		$response = true;

		if ((int)$basket['CompetitionEntry']['competition_entry_state_id'] === CompetitionEntryState::UNPAID) {
			if ($paid === true) {

				$BookingApi = ClassRegistry::init('BuzzBookings.BookingApi');

				$result = $BookingApi->addItemToSalesOrder(
					$this->name,
					$id,
					Configure::read('api.site_id'),
					null,
					$basket['Competition']['api_reference'],
					1,
					$basket['CompetitionEntry']['total_cost']
				);

				$salesRef = null;

				if ($result !== false) {
					$salesRef = $result['sales_ref'];

					$result = $BookingApi->completeSale(
						$this->name,
						$id,
						$salesRef,
						$basket['CustomerAddress']['first_name'],
						$basket['CustomerAddress']['last_name'],
						$basket['CustomerAddress']['email'],
						$basket['CustomerAddress']['telephone'],
						CustomerAddress::generateAddressArray($basket['CustomerAddress']),
						$basket['CompetitionEntry']['total_cost'],
						$this->getPaymentDetails($id, 'BuzzCompetitions.CompetitionEntry')
					);
				}

				// Update the competition entry's record in the database.
				$data = array(
					'id' => $id,
					'sales_ref' => $salesRef,
					'competition_entry_state_id' => $result === true ? CompetitionEntryState::COMPLETE : CompetitionEntryState::API_FAILED,
					'completed_date' => gmdate('Y-m-d H:i:s')
				);

				if ($result === false) {
					$data['api_log'] = json_encode(ClassRegistry::init('BuzzBookings.BookingApi')->getApiResponses());
				}

				$response = $this->save($data);

			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'competition_entry_state_id' => CompetitionEntryState::PAYMENT_FAILED
				);
				$response = $this->save($data);
			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.CompetitionEntry.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}

		if (! empty($basket['CompetitionEntry']['newsletter_opt_in']) && CakePlugin::loaded('BuzzSubscribe')) {
			ClassRegistry::init('BuzzSubscribe.Subscriber')->addSubscriber(
				$basket['CustomerAddress']['email'],
				$basket['CustomerAddress']['first_name'],
				$basket['CustomerAddress']['last_name'],
				'Competition Entry'
			);
		}

		return $response;
	}

}
