<?php

App::uses('BuzzCompetitionsAppModel', 'BuzzCompetitions.Model');
App::uses('CakeTime', 'Utility');

class Competition extends BuzzCompetitionsAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzAdverts.AdvertHost',
		'BuzzTranslate.Translatable',
		'EvCore.Tabbed' => array(
			'max_tabs' => 4
		),
		'MetaData.Meta',
	);

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Theme' => array(
			'className' => 'BuzzTheme.Theme'
		)
	);

	/**
	 * HABTM associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'CompetitionCategory' => array(
			'className' => 'BuzzTheme.CompetitionCategory'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more than 250 characters long'
			)
		),
		'api_reference' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long'
			)
		),
		'open_date' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'date' => array(
				'rule' => '/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/',
				'message' => 'Invalid date'
			)
		),
		'closing_date' => array(
			'date' => array(
				'rule' => '/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/',
				'message' => 'Invalid date',
				'allowEmpty' => true
			)
		),
		'price' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'decimal' => array(
				'rule' => '|^\d+(\.\d{2})?$|',
				'message' => 'Decimal numbers only'
			)
		),
		'email' => array(
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			)
		),
		'video_url' => array(
			'required' => array(
				'rule' => array('url', true),
				'message' => 'Invalid URL',
				'allowEmpty' => true
			)
		)
	);

	/**
	 * Image slots
	 *
	 * @var array
	 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		]
	];

	/**
	 * Constructor
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		if (Configure::check('BuzzCompetitions.routes.competitions') === true) {
			$this->addBehaviour('Routable.Routable', Configure::read('BuzzCompetitions.routes.competitions'));
		}

		// If BuzzGallery loaded associate activities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}
	}

	/**
	 * Read for edit
	 *
	 * @param int $id
	 * @param array $params
	 * @return array
	 */
	public function readForEdit($id, $params = []) {
		$params['contain'][] = 'CompetitionCategory';
		$params['contain']['Advert'] = array(
			'AdvertRelation',
			'AdvertTemplate',
			'Image',
			'order' => array('AdvertRelation.sequence' => 'ASC')
		);
		return parent::readForEdit($id, $params);
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as &$result) {
			if (
				(!empty($result[$this->alias]['open_date']) && CakeTime::isFuture($result[$this->alias]['open_date']))
				|| (!empty($result[$this->alias]['closing_date'])	&& CakeTime::isPast($result[$this->alias]['closing_date']))
			) {
				$result[$this->alias]['is_open'] = false;
			} else {
				$result[$this->alias]['is_open'] = true;
			}
		}

		return $results;
	}

	/**
	 * Read for view
	 *
	 * @param int $id
	 * @param array $params
	 * @return array
	 */
	public function readForView($id, $params = []) {
		$data = parent::readForView($id, $params);

		// Calculate the maximum number of participant for the competition.
		$participants = Hash::extract($data, 'CompetitionCategory.{n}.participants');
		$data[$this->alias]['max_participants'] = !empty($participants) ? max($participants) : 0;

		return $data;
	}

}
