<?php

App::uses('BuzzCompetitionsAppController', 'BuzzCompetitions.Controller');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class CompetitionEntriesController extends BuzzCompetitionsAppController implements PurchasableInterface {

	/**
	 * List of admin actions the controller supports
	 *
	 * @var array
	 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_email_confirmation'
	);

	/**
	 * Components
	 *
	 * @var array
	 */
	public $components = [
		'BuzzPurchase.EcommerceTracking',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'enter',
			'payment_callback',
			'confirmation'
		));
	}

	/**
	 * @param int $competitionId
	 * @return void
	 */
	public function enter($competitionId) {
		$CompetitionEntry = $this->{$this->modelClass};

		$data = $CompetitionEntry->Competition->readForView($competitionId);
		if (empty($data) || $data['Competition']['is_open'] === false) {
			throw new NotFoundException();
		}

		// Set active site
		$this->loadModel('BuzzSites.Site');
		$this->Site->setActiveSite($data['Competition']['site_id']);
		$site = $this->Site->getActiveSite();

		if (!empty($this->request->data)) {

			$result = $CompetitionEntry->submitEntry($competitionId, $this->request->data);
			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();
			if ($result === true && $paymentValidates === true) {
				$this->_pay($site);
			} else {
				$this->Session->setFlash(
					__d('buzz_competitions', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			$this->request->data = $CompetitionEntry->getBasket();

			if (empty($this->request->data)) {
				// Set the address field defaults (these are configured by the
				// CustomerAddress plugin to make it easier to override for each site).
				$this->request->data['CompetitionEntry']['team_country_id'] = CustomerAddress::getDefaultCountry();
				$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
				$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();
			}

		}

		// Get competition entry conditions
		$this->loadModel('BuzzConditions.Condition');
		$conditions = $this->Condition->getConditions('CompetitionEntry');

		$this->set(compact('data', 'conditions'));
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->Meta->canonical(['action' => 'enter', $competitionId]);
		$this->_populateLookups();
		$this->view = 'BuzzCompetitions.form';

		return;
	}

	/**
	 * Processes the payment and triggers the completion of the payment on success
	 *
	 * @return void
	 */
	protected function _pay($site) {
		$CompetitionEntry = $this->{$this->modelClass};

		// Get the latest basket.
		$basket = $CompetitionEntry->getBasket();

		if (Configure::check('skipPayment') === false && $basket['CompetitionEntry']['total_cost'] > 0) {

			if (Configure::read('BuzzPurchase.onsite') === true) {
				$payment = $this->request->data['Payment'];

				// Split name into first and second names for payment gateways.
				$fullName = trim($payment['card_name']);
				$splitName = explode(' ', $fullName);
				$payment['last_name'] = array_pop($splitName);
				$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
			} else {
				$payment = [];
			}


			$transaction = $this->Transactions->takePayment(
				// Return URLs
				array(
					'return' => Router::url(['action' => 'payment_callback', $basket['CompetitionEntry']['id']], true),
					'cancel' => Router::url(['action' => 'enter', $basket['CompetitionEntry']['competition_id']], true)
				),
				// Calling record
				array(
					'model' => 'BuzzCompetitions.CompetitionEntry',
					'model_id' => $basket['CompetitionEntry']['id']
				),
				// Amount to be paid
				$basket['CompetitionEntry']['total_cost'],
				// Items
				array(),
				// Extras
				array(
					'language' => Configure::read('Config.language'),
					'card' => $payment,
					'address' => $basket['CustomerAddress'],
					'user' => array('email' => $basket['CustomerAddress']['email']),
					'description' => __d(
						'buzz_competitions',
						'%s Competition Entry %d',
						[Configure::read('SiteSetting.site_title'), $basket['CompetitionEntry']['id']]
					),
					'site_id' => $site['Site']['api_site_id']
				),
				0,
				Configure::read('Transactions.gateway')
			);

			if (!empty($transaction['result'])) {
				// Payment taken successfully, complete purchase.
				return $this->_completeCompetitionEntry($basket['CompetitionEntry']['id']);
			} else {
				$this->Session->setFlash(
					__d('buzz_competitions', 'There was a problem processing your payment, please check your details and try again'),
					'flash_fail'
				);
			}

		} else {
			// Nothing to pay, complete purchase.
			return $this->_completeCompetitionEntry($basket['CompetitionEntry']['id']);
		}


		return;
	}

	/**
	 * Generic payment gateway callback.
	 *
	 * @param int $basketId Basket ID
	 * @return void
	 */
	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completeCompetitionEntry($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_competitions', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect('/');
		}
	}

/**
 * Complete the competition entry (after payment received)
 *
 * @param int $basketId Basket ID
 * @return void Redirects to confirmation page
 */
	protected function _completeCompetitionEntry($basketId) {
		$CompetitionEntry = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}

		$this->Session->write('CompetitionEntry.hash', $CompetitionEntry->hashBasketId());

		$CompetitionEntry->completeEntry(true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

	/**
	 * Confirmation page
	 *
	 * @param int $id Basket ID
	 * @return void
	 */
	public function confirmation($id) {
		$CompetitionEntry = $this->{$this->modelClass};

		if ($CompetitionEntry->hashBasketId($id) === $this->Session->read('CompetitionEntry.hash')) {

			$this->Session->delete('CompetitionEntry.hash');

			$basket = $CompetitionEntry->getPurchase($id);

			// Check that a sale has been generated by the API before
			// proceeding. This should never happen if the API is up and
			// running and the site is correctly configured.
			if (empty($basket['CompetitionEntry']['sales_ref'])) {
				throw new InternalErrorException();
			}

			if (Configure::check('BuzzCompetitions.confirmation_page_id') === true) {
				$this->assignPage(Configure::read('BuzzCompetitions.confirmation_page_id'));
			}

			$this->EcommerceTracking->addItem(
				$basket['Competition']['name'],
				'CO' . $basket['Competition']['id'],
				$basket['CompetitionEntry']['total_cost'],
				1,
				['category' => 'Competition']
			);
			$tax = 0;
			$ecommerceTracking = $this->EcommerceTracking->transaction(
				$basket['CompetitionEntry']['sales_ref'],
				$basket['CompetitionEntry']['total_cost'],
				CakeNumber::defaultCurrency(),
				$tax
			);
			$this->set(compact('ecommerceTracking'));

			$this->set(compact('basket'));
			$this->view = 'BuzzCompetitions.confirmation';

		} else {

			throw new ForbiddenException();

		}

		return;
	}

	/**
	 * Populate lookups for address fields.
	 *
	 * @return void
	 */
	protected function _populateLookups() {
		$CompetitionEntry = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$usStates = $CompetitionEntry->CustomerAddress->UsState->find('list');
		$teamCountries = $countries;

		$this->set(compact('countries', 'teamCountries', 'usStates'));

		return;
	}

	/**
	 * Admin index paginate
	 *
	 * @return array
	 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'Competition',
				'CompetitionCategory',
				'CompetitionEntryState',
				'CustomerAddress'
			)
		);

		return $paginate;
	}

	/**
	 * Admin index columns
	 *
	 * @return array
	 */
	protected function _adminIndexColumns() {
		$CompetitionEntry = $this->{$this->modelClass};

		$columns = parent::_adminIndexColumns();

		// Remove created/modified columns.
		unset($columns['CompetitionEntry.created']);
		unset($columns['CompetitionEntry.modified']);

		$columns[$CompetitionEntry->alias . '.total_cost']['type'] = 'currency';

		$newColumns = array(
			'Competition.name' => array(
				'label' => __d('buzz_competitions', 'Competition'),
				'type' => 'string'
			),
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_competitions', 'Name'),
				'type' => 'string'
			),
			'CompetitionEntry.team_name' => array(
				'label' => __d('buzz_competitions', 'Team Name'),
				'type' => 'string'
			),
			'CompetitionCategory.name' => array(
				'label' => __d('buzz_competitions', 'Category'),
				'type' => 'string'
			),
			'CompetitionEntryState.name' => array(
				'label' => __d('buzz_competitions', 'Status'),
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'CompetitionEntry.sales_ref', $newColumns);
	}

	/**
	 * Admin columns whitelist
	 *
	 * @return array
	 */
	protected function _adminIndexColumnsWhitelist() {
		$CompetitionEntry = $this->{$this->modelClass};

		$whitelist = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $CompetitionEntry->alias . '.sales_ref';
		$whitelist[] = $CompetitionEntry->alias . '.total_cost';
		$whitelist[] = $CompetitionEntry->alias . '.completed_date';

		return $whitelist;
	}

	/**
	 * Filters
	 *
	 * @return array
	 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['CompetitionEntry.name']);
		unset($filters['CompetitionEntry.created']);
		unset($filters['CompetitionEntry.modified']);

		$newFilters = array(
			'CompetitionEntry.sales_ref' => array(
				'label' => __d('buzz_competitions', 'Sales Ref'),
				'type' => 'string',
				'compare' => array('CompetitionEntry.sales_ref' => "%s")
			),
			'CompetitionEntry.competition_id' => array(
				'label' => 'Competition',
				'type' => 'select',
				'compare' => array('CompetitionEntry.competition_id' => '%s')
			),
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_competitions', 'Name'),
				'type' => 'string',
				'compare' => array('CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%")
			),
			'CompetitionEntry.team_name' => array(
				'label' => __d('buzz_competitions', 'Team Name'),
				'type' => 'string',
				'compare' => array('CompetitionEntry.team_name LIKE' => "%%%s%%")
			),
			'CompetitionEntry.competition_category_id' => array(
				'label' => __d('buzz_competitions', 'Category'),
				'type' => 'select',
				'compare' => array('CompetitionEntry.competition_category_id' => "%s")
			),
			'CompetitionEntry.competition_entry_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => CompetitionEntryState::COMPLETE,
				'compare' => array('CompetitionEntry.competition_entry_state_id' => '%s')
			),
			'CompetitionEntry.completed_date' => array(
				'label' => __d('buzz_competitions', 'Completed Date'),
				'type' => 'date',
				'compare' => array('CompetitionEntry.completed_date' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'CompetitionEntry.sales_ref', $newFilters);
	}

	/**
	 * Used to populate form drop down selects
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$this->set('competitions', $this->CompetitionEntry->Competition->find('list'));
		$this->set('competitionCategories', $this->CompetitionEntry->CompetitionCategory->find('list'));
		$this->set('competitionEntryStates', $this->CompetitionEntry->CompetitionEntryState->find('list'));
		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);
		// Make sure we have the purchase data from the database available in the view. Otherwise,
		// on form submit not all the data is available in the request data.
		if ($this->request->is(['post', 'put'])) {
			$competitionEntry = $this->GiftVoucherPurchase->readForEdit($id);
		} else {
			$competitionEntry = $this->request->data;
		}
		$this->set(compact('competitionEntry'));

		$this->set('teamCountries', $this->CompetitionEntry->TeamCountry->find('list'));

		$participants = Hash::extract($this->request->data['CompetitionEntryParticipant'], '{n}.name');

		$this->request->data['CompetitionEntry']['team_captain'] = array_shift($participants);
		$this->request->data['CompetitionEntry']['participants'] = implode(', ', $participants);

		if ((int)$this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('CompetitionEntry', $id));
		}

		// We're overriding the scaffolding template as we want to customise
		// the tabs.
		$this->view = 'BuzzCompetitions.admin_form';

		return;
	}

	protected function _adminIndexToolbar($id = null) {
		return [];
	}

	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);

		$actions['Resend Email'] = array(
			'url' => array('action' => 'email_confirmation', $id),
			'icon' => 'envelope'
		);

		return $actions;
	}

	/**
	 * Re-sends the confirmation email to the customer and redirects to the
	 * admin edit form.
	 *
	 * @param int $id Purchase ID
	 */
	public function admin_email_confirmation($id) {
		$CompetitionEntry = $this->{$this->modelClass}->alias;

		$Event = new CakeEvent('Model.CompetitionEntry.completed', $this, array(
			'id' => $id
		));
		$this->getEventManager()->dispatch($Event);

		$this->Session->setFlash(
			array(
				'title' => __d('buzz_competitions', '%s confirmation sent', [InflectorExt::humanize($this->$CompetitionEntry->displayName)]),
				'description' => __d('buzz_competitions', 'The confirmation email has been resent to the customer!')
			),
			'flash_success'
		);

		return $this->redirect(['action' => 'edit', $id]);
	}

}
