<?php

App::uses('BuzzCampsAppModel', 'BuzzCamps.Model');
App::uses('CampBookingState', 'BuzzCamps.Model');
App::uses('AccountCreditState', 'BuzzDiaryAccounts.Model');

class CampBooking extends BuzzCampsAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

/**
 * Belongs to associations
 */
	public $belongsTo = array(
		'AccountCredit' => array(
			'className' => 'BuzzDiaryAccounts.AccountCredit'
		),
		'Camp' => array(
			'className' => 'BuzzCamps.Camp'
		),
		'CampBookingState' => array(
			'className' => 'BuzzCamps.CampBookingState'
		),
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		)
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'first_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'last_name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'email' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters'
			)
		),
		'telephone' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		)
	);

/**
 * Constructor
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);
		$this->virtualFields['full_name'] = 'CONCAT(`' . $this->alias . '`.`first_name`, \' \', `' . $this->alias . '`.`last_name`)';
		return;
	}

/**
 * Read for view
 *
 * @param int $id
 * @param array $params
 * @return array
 */
	public function readForView($id, $params = []) {
		$defaults = array(
			'contain' => array(
				'Camp' => array(
					'AccountProfile' => 'Account'
				)
			)
		);
		$params = Hash::merge($defaults, $params);
		return parent::readForView($id, $params);
	}

/**
 * Get the current basket ID from the session
 *
 * @return int
 */
	public function getBasketId() {
		if (empty($this->_basketId)) {
			$this->_basketId = CakeSession::read('CampBooking.id');
		}
		return $this->_basketId;
	}

/**
 * Clear the basket ID from the session
 *
 * @return bool
 */
	public function clearBasketId() {
		$this->_basketId = null;
		CakeSession::delete('CampBooking.id');
		return true;
	}

/**
 * Returns a hashed basket ID
 *
 * @param int $id (Optional) basket ID
 * @return string
 */
	public function hashBasketId($id = null) {
		$id = $id ?: $this->getBasketId();
		return md5($id);
	}

/**
 * Create a camp booking and account credit
 *
 * @param int $campId Camp ID
 * @param array $data Details of participant and payment
 * @return bool
 */
	public function createBooking($campId, array $data) {
		$camp = $this->Camp->find('first', array(
			'contain' => array(
				'AccountProfile'
			),
			'conditions' => array(
				'Camp.id' => $campId
			)
		));

		$data['CampBooking']['camp_id'] = $campId;
		$data['CampBooking']['total_paid'] = $camp['Camp']['cost'];
		$data['CampBooking']['camp_booking_state_id'] = CampBookingState::UNPAID;
		$data['AccountCredit']['account_id'] = $camp['AccountProfile']['account_id'];
		$data['AccountCredit']['amount'] = $camp['Camp']['cost'];
		$data['AccountCredit']['account_credit_state_id'] = AccountCreditState::UNPAID;
		$result = $this->saveAssociated($data, ['deep' => true]) !== false;

		// Set the 'basket' ID.
		if ($result === true) {
			CakeSession::write('CampBooking.id', $this->id);
		}

		return $result;
	}

/**
 * Complete the booking
 *
 * @param bool $paid True if payment successfully made
 * @param array $payment
 * @return bool
 */
	public function completeBooking($paid) {
		$id = $this->getBasketId();

		$data = $this->find('first', array(
			'contain' => array(
				'Camp',
				'AccountCredit' => array(
					'Account',
					'CustomerAddress' => array(
						'Country'
					)
				)
			),
			'conditions' => array(
				'CampBooking.id' => $id
			)
		));

		// Restructure the data so that it is in the expected format for
		// AccountCredit::addCredit().
		$data['Account'] = $data['AccountCredit']['Account'];
		unset($data['AccountCredit']['Account']);
		$data['CustomerAddress'] = $data['AccountCredit']['CustomerAddress'];
		unset($data['AccountCredit']['CustomerAddress']);

		$response = true;

		if ((int)$data['CampBooking']['camp_booking_state_id'] === CampBookingState::UNPAID) {
			if ($paid === true) {
				// Create the account credit via the API
				$payeeReference = array(
					'name' => $data['CampBooking']['first_name'] . ' ' . $data['CampBooking']['last_name'],
					'email_address' => $data['CampBooking']['email'],
					'telephone' => $data['CampBooking']['telephone'],
					'reference' => $data['Camp']['name']
				);
				// Map the participant details onto the customer address for
				// sending to the API.
				$data['CustomerAddress']['email'] = $data['CampBooking']['email'];
				$data['CustomerAddress']['telephone'] = $data['CampBooking']['telephone'];
				$result = $this->AccountCredit->addCredit(
					$data,
					$this->getPaymentDetails($id, 'BuzzCamps.CampBooking'),
					$payeeReference
				);

				$state = $result !== false ? CampBookingState::COMPLETE : CampBookingState::API_FAILED;
				$data = array(
					'id' => $id,
					'sales_ref' => $result ?: null,
					'camp_booking_state_id' => $state
				);
				$response = $this->save($data);

			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'camp_booking_state_id' => CampBookingState::PAYMENT_FAILED
				);
				$response = $this->save($data);

			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.CampBooking.created', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);

		}
		return $response;
	}

}
