<?php

App::uses('BuzzCampsAppModel', 'BuzzCamps.Model');
App::uses('CampBookingState', 'BuzzCamps.Model');
App::uses('DiaryApi', 'BuzzDiaryAccounts.Model');

class Camp extends BuzzCampsAppModel {

	public $imageSlots = 1;

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'AccountProfile' => array(
			'className' => 'BuzzCamps.AccountProfile'
		),
		'Site' => array(
			'className' => 'BuzzSites.Site'
		)
	);

	/**
	 * Has many associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'CampBooking' => array(
			'className' => 'BuzzCamps.CampBooking'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'tag_line' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters'
			)
		),
		'date' => array(
			'required' => array(
				'rule' => '/^\d{4}-\d{2}-\d{2}$/',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			),
			'future' => array(
				'rule' => 'validateFutureDate',
				'message' => 'Date must be in the future'
			)
		),
		'time' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'slots_available' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'int' => array(
				'rule' => 'naturalNumber',
				'message' => 'Integers only'
			)
		),
		'site_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'cost' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'currency' => array(
				'rule' => '/^\d+(\.\d{2})?$/',
				'message' => 'Not a valid price',
				'allowEmpty' => true
			),
			'max' => array(
				'rule' => array('comparison', '<', 1000000),
				'message' => 'Value must be less than 1000000.00'
			)
		)
	);

	/**
	 * Custom validation method for checking a date is in the future.
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validateFutureDate(array $data) {
		$check = array_pop($data);
		return date('Y-m-d') <= $check;
	}

	/**
	 * After find callback.
	 *
	 * @param mixed $results The results of the find operation
	 * @param boolean $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 */
	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		// When directly querying the Camp model we want to work out the
		// availability. This shouldn't be done when querying as an association
		// as there's a risk of getting stuck in an infinite loop.
		if ($primary === true) {

			$bookings = $this->CampBooking->find(
				'all',
				array(
					'conditions' => array(
						'CampBooking.camp_id' => Hash::extract($results, '{n}.Camp.id'),
						'CampBooking.is_removed' => false,
						'CampBooking.camp_booking_state_id' => CampBookingState::COMPLETE
					)
				)
			);
			$bookings = Hash::combine($bookings, '{n}.CampBooking.id', '{n}.CampBooking', '{n}.CampBooking.camp_id');

			foreach ($results as &$result) {
				$campId = $result['Camp']['id'];
				$result['Camp']['bookings'] = !empty($bookings[$campId]) ? count($bookings[$campId]) : 0;
				$result['Camp']['remaining_availability'] = $result['Camp']['slots_available'] - $result['Camp']['bookings'];
				$result['Camp']['remaining_availability'] = $result['Camp']['remaining_availability'] < 0 ? 0 : $result['Camp']['remaining_availability'];
			}

		}

		return $results;
	}

	public function readForView($id, $params = []) {
		$params['contain'] = [
			'Image',
			'AccountProfile' => [
				'Image'
			],
		];
		$params['conditions']['Camp.date >='] = gmdate('Y-m-d');
		return parent::readForView($id, $params);
	}

	/**
	 * Returns a camp belonging to a specific account.
	 *
	 * @param int $campId Camp record ID
	 * @param int $accountId Account ID
	 * @return array
	 */
	public function getAccountCamp($campId, $accountId) {
		return $this->find(
			'first',
			[
				'contain' => [
					'Image'
				],
				'conditions' => [
					'Camp.id' => $campId,
					'Camp.account_profile_id' => $accountId
				]
			]
		);
	}

	/**
	 * Returns a camp's bookings belonging to a specific account.
	 *
	 * @param int $campId Camp record ID
	 * @param int $accountId Account ID
	 * @return array
	 */
	public function getAccountCampBooking($bookingId, $accountId) {
		return $this->CampBooking->find('first', [
			'contain' => ['Camp'],
			'conditions' => [
				'CampBooking.id' => $bookingId,
				'CampBooking.is_removed' => false,
				'CampBooking.camp_booking_state_id' => CampBookingState::COMPLETE,
				'Camp.account_profile_id' => $accountId
			]
		]);
	}

	/**
	 * Returns a camp's bookings belonging to a specific account.
	 *
	 * @param int $campId Camp record ID
	 * @param int $accountId Account ID
	 * @return array
	 */
	public function getAccountCampBookings($campId, $accountId) {
		return $this->find(
			'first',
			array(
				'contain' => array(
					'CampBooking' => array(
						'conditions' => array(
							'CampBooking.is_removed' => false,
							'CampBooking.camp_booking_state_id' => CampBookingState::COMPLETE
						)
					)
				),
				'conditions' => array(
					'Camp.id' => $campId,
					'Camp.account_profile_id' => $accountId
				)
			)
		);
	}

	/**
	 * Returns all upcoming camps for a particular account profile grouped by month.
	 *
	 * @param int $accountProfileId ID of account profile (must have a profile in order to have camps)
	 * @param int $limit Maximum number of results to return
	 * @param int $page
	 * @return array
	 */
	public function getUpcomingCamps($accountProfileId, $limit = null, $page = 0) {
		$params = array(
			'conditions' => array(
				'Camp.account_profile_id' => $accountProfileId,
				'Camp.date >=' => gmdate('Y-m-d')
			),
			'order' => array(
				'Camp.date' => 'ASC'
			),
			'limit' => $limit,
			'offset' => $page
		);
		return $this->find('all', $params);
	}

	/**
	 * Returns past camps for a particular account profile.
	 *
	 * @param int $accountProfileId ID of account profile (must have a profile in order to have camps)
	 * @param int $limit Maximum number of results to return
	 * @return array
	 */
	public function getPastCamps($accountProfileId, $limit = null, $page = 0) {
		$params = array(
			'conditions' => array(
				'Camp.account_profile_id' => $accountProfileId,
				'Camp.date <' => gmdate('Y-m-d')
			),
			'order' => array(
				'Camp.date' => 'DESC'
			),
			'limit' => $limit,
			'offset' => $page
		);
		return $this->find('all', $params);
	}

	/**
	 * Returns all bookable upcoming camps (includes any that have sold out).
	 *
	 * @param int $coachId Coach ID
	 * @return array
	 */
	public function getUpcomingBookableCamps($coachId = null) {
		$params = array(
			'fields' => array(
				'*',
				'Camp.*',
				'Site.*',
				'Account.*',
				'AccountProfile.*'
			),
			'contain' => array(
				'Image',
				'AccountProfile' => array(
					'Account',
					'Image'
				)
			),
			'joins' => array(
				array(
					'table' => 'accounts',
					'alias' => 'Account',
					'type' => 'INNER',
					'conditions' => 'Account.id = AccountProfile.account_id'
				),
				array(
					'table' => 'sites',
					'alias' => 'Site',
					'type' => 'INNER',
					'conditions' => 'Camp.site_id = Site.api_site_id'
				)
			),
			'conditions' => array(
				'Camp.is_bookable' => true,
				'Camp.date >=' => gmdate('Y-m-d'),
				'AccountProfile.is_active' => true,
				'Account.is_active' => true
			),
			'order' => array(
				'Camp.date' => 'ASC'
			)
		);

		if ($coachId !== null) {
			$params['conditions']['Camp.account_profile_id'] = $coachId;
		}

		$camps = $this->find('all', $params);

		// Group the camps by month.
		$data = [];
		foreach ($camps as $camp) {
			$month = substr($camp['Camp']['date'], 0, 7);
			$data[$month][] = $camp;
		}

		return $data;
	}

	/**
	 * Returns a bookable camp
	 *
	 * @param int $id Camp ID
	 * @return array
	 */
	public function getBooking($id) {
		$data = $this->find(
			'first',
			array(
				'contain' => array(
					'AccountProfile' => array(
						'Image'
					)
				),
				'conditions' => array(
					'Camp.id' => $id,
					'Camp.is_bookable' => true,
					'Camp.date >=' => gmdate('Y-m-d')
				)
			)
		);
		return $data;
	}

}
