<?php

App::uses('BuzzDiaryAccountsAppController', 'BuzzDiaryAccounts.Controller');
App::uses('TranslateUtil', 'BuzzTranslate.Lib');

class CampsController extends BuzzDiaryAccountsAppController {

	/**
	 * Define actions that will require account authentication.
	 *
	 * @var array
	 */
	public $requiresAccountAuth = [
		'account_add_booking',
		'account_bookings',
		'account_camps',
		'account_edit_camp',
		'account_delete_booking',
		'account_index',
		'account_past_camps',
		'account_profile',
		'get_next_camps'
	];

	/**
	 * Camps listing page (publically viewable).
	 *
	 * @return
	 */
	public function index() {
		$Camp = $this->{$this->modelClass};

		if (Configure::check('BuzzCamps.camps_page_id') === true) {
			$this->assignPage(Configure::read('BuzzCamps.camps_page_id'));
		}

		$camps = $Camp->getUpcomingBookableCamps();

		$this->Meta->canonical(['action' => 'index']);
		$this->set(compact('camps'));
		$this->view = 'BuzzCamps./Camps/index';

		return;
	}

	/**
	 * Camps listing page (publically viewable).
	 *
	 * @return
	 */
	public function coach($id) {
		$Camp = $this->{$this->modelClass};

		$data = $Camp->AccountProfile->findById($id);
		$camps = $Camp->getUpcomingBookableCamps($id);

		$this->Meta->canonical(['action' => 'coach', $id]);
		$this->set(compact('camps', 'data'));
		$this->view = 'BuzzCamps./Camps/coach';

		return;
	}

	/**
	 * Camp details page (publically viewable).
	 *
	 * @return
	 */
	public function view($id) {
		$Camp = $this->{$this->modelClass};

		$data = $Camp->readForView($id);

		$site = null;
		if (! empty($data['Camp']['site_id'])) {
			$this->loadModel('BuzzSites.Site');
			$site = $this->Site->find('first', ['conditions' => ['api_site_id' => $data['Camp']['site_id']]]);
		}

		if (empty($data)) {
			// We want to redirect here to the camps landing page as the camp
			// may have expired or be temporarily unavailable rather than not
			// exist.
			return $this->redirect(['action' => 'index']);
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && Configure::check('BuzzCamps.camps_gallery_id') === true) {
			$this->loadModel('BuzzGallery.Gallery');
			$this->set(
				'gallery',
				$this->Gallery->readForView(Configure::read('BuzzCamps.camps_gallery_id'))
			);
		}

		$this->Meta->canonical(['action' => 'view', $id]);
		$this->set(compact('data', 'site'));
		$this->view = 'BuzzCamps./Camps/view';

		return;
	}

	/**
	 * Camp bookings
	 *
	 * @param int $campId Camp ID
	 * @return void
	 */
	public function account_bookings($campId) {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		if (empty($profile) || $canCreateCamps === false) {
			throw new ForbiddenException();
		}

		$data = $Camp->getAccountCampBookings($campId, $profile['AccountProfile']['id']);

		if (empty($data)) {
			throw new ForbiddenException();
		}

		$this->set(compact('data'));
		$this->view = 'BuzzCamps./Camps/account_bookings';

		return;
	}

	public function account_delete_booking($bookingId) {
		$Camp = $this->{$this->modelClass};
		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);
		if (empty($profile) || $canCreateCamps === false) {
			throw new ForbiddenException();
		}
		$data = $Camp->getAccountCampBooking($bookingId, $profile['AccountProfile']['id']);
		if (empty($data)) {
			throw new ForbiddenException();
		}
		if (!empty($this->request->data)) {
			if ($this->Camp->CampBooking->delete($bookingId)) {
				$this->Session->setFlash(__d('buzz_camps', 'The booking has been deleted'), 'flash_success');
				return $this->redirect(['action' => 'account_bookings', $data['Camp']['id']]);
			} else {
				$this->Session->setFlash(__d('buzz_camps', 'There was a problem deleteing the booking'), 'flash_fail');
			}
		}
		$this->set(compact('data'));
		$this->view = 'BuzzCamps./Camps/account_delete_booking';
		return;
	}

	public function account_add_booking($campId) {
		$Camp = $this->{$this->modelClass};
		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);
		if (empty($profile) || $canCreateCamps === false) {
			throw new ForbiddenException();
		}
		$data = $Camp->getAccountCampBookings($campId, $profile['AccountProfile']['id']);
		if (empty($data)) {
			throw new ForbiddenException();
		}
		if (!empty($this->request->data)) {
			$this->request->data['CampBooking']['camp_id'] = $campId;
			$this->request->data['CampBooking']['camp_booking_state_id'] = CampBookingState::COMPLETE;
			if ($this->Camp->CampBooking->save($this->request->data)) {
				$this->Session->setFlash(__d('buzz_camps', 'The booking has been created'), 'flash_success');
				return $this->redirect(['action' => 'account_bookings', $campId]);
			} else {
				$this->Session->setFlash(__d('buzz_camps', 'There was a problem creating the booking'), 'flash_fail');
			}
		}
		$this->set(compact('data'));
		$this->view = 'BuzzCamps./Camps/account_add_booking';
		return;
	}

	/**
	 * Account's camps overview.
	 *
	 * @return void
	 */
	public function account_index() {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		if ($canCreateCamps === false) {
			throw new ForbiddenException();
		}

		// Get the next 3 upcoming camps.
		$data = [];
		$pastCamps = [];
		if (!empty($profile['AccountProfile']['id'])) {
			$data = $Camp->getUpcomingCamps($profile['AccountProfile']['id'], 3);
			$pastCamps = $Camp->getPastCamps($profile['AccountProfile']['id'], 3);
		}

		if (! empty($data) || ! empty($pastCamps)) {
			$this->loadModel('BuzzSites.Site');
			$sites = $this->Site->find('all');
			$siteList = [];

			foreach ($sites as $site) {
				$siteList[$site['Site']['api_site_id']] = $site;
			}

			$this->set('sites', $siteList);
		}

		$this->set(compact('data', 'pastCamps', 'profile'));
		$this->view = 'BuzzCamps./Camps/account_index';

		return;
	}

	/**
	 * Account's upcoming camps.
	 *
	 * @return void
	 */
	public function account_camps() {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		if ($canCreateCamps === false) {
			throw new ForbiddenException();
		}

		$limit = 10;

		$data = [];
		if (!empty($profile['AccountProfile']['id'])) {
			$data = $Camp->getUpcomingCamps($profile['AccountProfile']['id'], $limit);
		}
		$paged = !(count($data) < $limit);

		if (! empty($data) || ! empty($pastCamps)) {
			$this->loadModel('BuzzSites.Site');
			$sites = $this->Site->find('all');
			$siteList = [];

			foreach ($sites as $site) {
				$siteList[$site['Site']['api_site_id']] = $site;
			}

			$this->set('sites', $siteList);
		}

		$this->set(compact('data', 'limit', 'paged', 'profile'));
		$this->view = 'BuzzCamps.account_camps';

		return;
	}

	/**
	 * Account's past camps.
	 *
	 * @return void
	 */
	public function account_past_camps() {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		if ($canCreateCamps === false) {
			throw new ForbiddenException();
		}

		$limit = 10;

		$data = [];
		if (!empty($profile['AccountProfile']['id'])) {
			$data = $Camp->getPastCamps($profile['AccountProfile']['id'], $limit);
		}
		$paged = !(count($data) < $limit);

		if (! empty($data)) {
			$this->loadModel('BuzzSites.Site');
			$this->set('sites', $this->Site->getListByApiId());
		}

		$this->set(compact('data', 'limit', 'paged', 'profile'));
		$this->view = 'BuzzCamps.account_past_camps';

		return;
	}

	/**
	 * Loads the next page of camps
	 *
	 * @param int $page
	 * @param int $limit
	 * @param bool $past
	 * @return void
	 */
	public function get_next_camps($page = 0, $limit = 10, $past = false) {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		if ($canCreateCamps === false) {
			throw new ForbiddenException();
		}

		if ((bool)$past === true) {
			$data = $Camp->getPastCamps($profile['AccountProfile']['id'], $limit, $page);
		} else {
			$data = $Camp->getUpcomingCamps($profile['AccountProfile']['id'], $limit, $page);
		}

		if (empty($data)) {
			// No more bookings to display so just die so that there is no response.
			die();
		}

		$paged = !(count($data) < $limit);

		$this->set(compact('data', 'page', 'paged', 'limit'));
		$this->set('edit', !$past);

		$this->view = 'BuzzCamps./Elements/camps_table';
		$this->layout = 'ajax';

		return;
	}

	/**
	 * Account profile.
	 *
	 * @return void
	 */
	public function account_profile() {
		$Camp = $this->{$this->modelClass};

		// Get logged in account details.
		$account = $this->Auth->user();

		if ($account['can_create_camps'] === false) {
			throw new ForbiddenException();
		}

		if (!empty($this->request->data)) {

			// Make sure they are editing their own profile.
			$this->request->data['AccountProfile']['account_id'] = $account['id'];

			if ($Camp->AccountProfile->saveAssociated($this->request->data, ['deep' => true])) {
				$this->Session->setFlash(__d('buzz_camps', 'Your profile details have been updated'), 'flash_success');
				return $this->redirect($this->here);
			} else {
				$this->Session->setFlash(__d('buzz_camps', 'There was a problem updating your profile'), 'flash_fail');
			}

		} else {

			// Populate the form data.
			$this->request->data = $Camp->AccountProfile->getAccountProfile($account['id']);
			if (empty($this->request->data['AccountProfile']['display_name'])) {
				$this->request->data['AccountProfile']['display_name'] = $account['full_name'];
			}

		}

		$this->view = 'BuzzCamps./Camps/account_profile';

		return;
	}

	/**
	 * Create or edit camps
	 *
	 * @param int $id Camp record ID if editing, pass null to create a new camp
	 * @param bool $copy Pass true to prepopulate form with values from existing camp
	 * @return void
	 */
	public function account_edit_camp($id = null, $copy = false) {
		$Camp = $this->{$this->modelClass};

		$accountId = $this->Auth->user('id');
		$canCreateCamps = $this->Auth->user('can_create_camps');
		$profile = $Camp->AccountProfile->getAccountProfile($accountId);

		// Account holder must have created a profile before being allowed to
		// create a camp.
		if (empty($profile) || $canCreateCamps === false) {
			throw new ForbiddenException();
		}

		if (!empty($this->request->data)) {

			// Make sure they add the camp to their own profile.
			$this->request->data['Camp']['account_profile_id'] = $profile['AccountProfile']['id'];

			if ($Camp->saveAssociated($this->request->data) !== false) {
				$message = __d('buzz_camps', 'Your camp has been updated');
				$message = $id !== null || (bool)$copy === true ? $message : __d('buzz_camps', 'Your new camp has been created');
				$this->Session->setFlash($message, 'flash_success');
				return $this->redirect(['action' => 'account_index']);
			} else {
				$this->Session->setFlash(__d('buzz_camps', 'There was a problem saving your camp'), 'flash_fail');
			}

		} elseif (!empty($id)) {

			$this->request->data = $Camp->getAccountCamp($id, $profile['AccountProfile']['id']);
			if (empty($this->request->data)) {
				throw new ForbiddenException();
			}

			// If we're making a 'copy' unset data that doesn't want duplicating.
			if ((bool)$copy === true) {
				unset(
					$this->request->data['Camp']['id'],
					$this->request->data['Camp']['date'],
					$this->request->data['Camp']['time']
				);
				$id = null;
			}

		} else {

			$this->request->data['Camp']['is_bookable'] = true;

		}
		$this->loadModel('BuzzSites.Site');
		$sites = $this->Site->find('all');
		$siteList = [];

		$translateUtil = new TranslateUtil();

		foreach ($sites as $site) {
			$siteList[$site['Site']['api_site_id']] = $translateUtil->translate($site, 'Site.name');
		}
		$this->set(compact('id', 'copy', 'siteList'));
		$this->view = 'BuzzCamps./Camps/account_edit_camp';

		return;
	}

}
