<?php

App::uses('BuzzDiaryAccountsAppController', 'BuzzDiaryAccounts.Controller');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');

class CampBookingsController extends BuzzDiaryAccountsAppController implements PurchasableInterface {

/**
 * Helpers
 */
	public $helpers = [
		'BuzzPurchase.Purchase'
	];

/**
 * Components
 *
 * @var array
 */
	public $components = [
		'BuzzPurchase.EcommerceTracking',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

/**
 * Book and pay for a camp.
 *
 * @param int $campId ID of camp being booked
 * @return void
 */
	public function book($campId) {
		$CampBooking = $this->{$this->modelClass};

		$data = $CampBooking->Camp->getBooking($campId);
		if (empty($data)) {
			// We want to redirect here to the camps landing page as the camp
			// may have expired or be temporarily unavailable rather than not
			// exist.
			return $this->redirect(['controller' => 'camps', 'action' => 'index']);
		} elseif ($data['Camp']['remaining_availability'] === 0) {
			$this->Session->setFlash(__d('buzz_camps', 'Sorry, this camp is now fully booked'));
			return $this->redirect(['controller' => 'camps', 'action' => 'index']);
		}

		$this->set(compact('data'));

		if (!empty($this->request->data)) {

			if ($CampBooking->createBooking($campId, $this->request->data) === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_camps', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['AccountCredit']['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
			$this->request->data['AccountCredit']['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();

		}

		$this->loadModel('BuzzSites.Site');
		$site = $this->Site->find('first', ['conditions' => ['api_site_id' => $data['Camp']['site_id']]]);
		$this->set(compact('site'));
		$this->_populateLookups();
		$this->Meta->canonical(['action' => 'book', $campId]);
		$this->view = 'BuzzCamps./CampBookings/book';
	}

/**
 * Stripe payment
 *
 * @return void
 * @throws MethodNotAllowedException
 */
	public function stripe_payment() {
		if (Configure::read('Transactions.gateway') !== 'stripe_checkout') {
			throw new MethodNotAllowedException('This site is not configured to use this action');
		}

		$CampBooking = $this->{$this->modelClass};
		$basketId = $CampBooking->getBasketId();
		$basket = $CampBooking->find(
			'first',
			[
				'contain' => ['AccountCredit.CustomerAddress.Country'],
				'conditions' => ['CampBooking.id' => $basketId]
			]
		);

		$stripeTransaction = $this->_takePayment($basketId, $basket);

		if ($stripeTransaction['result'] === false || empty($stripeTransaction['transaction_id'])) {
			$this->Session->setFlash(
				__d('buzz_camps', 'There was a problem processing your payment, please check your details and try again'),
				'flash_fail'
			);

			return $this->redirect(['action' => 'book']);
		}

		$this->loadModel('Transactions.Transaction');
		$transaction = $this->Transaction->findById(
			$stripeTransaction['transaction_id']
		);

		$stripeKey = Configure::read('Transactions.stripe.live.publishable_key');
		if (Configure::read('db.config') !== 'live') {
			$stripeKey = Configure::read('Transactions.stripe.dev.publishable_key');
		}

		$this->set(compact('transaction', 'stripeKey'));
		// Use the AJAX layout to improve the redirect transition to Stripe.
		$this->layout = 'ajax';
	}

/**
 * Processes the payment and triggers the completion of the payment on success
 *
 * @return void
 */
	protected function _pay() {
		$CampBooking = $this->{$this->modelClass};

		// Get the latest basket.
		$id = $CampBooking->getBasketId();
		$basket = $CampBooking->find(
			'first',
			[
				'contain' => ['AccountCredit.CustomerAddress.Country'],
				'conditions' => ['CampBooking.id' => $id]
			]
		);

		if ($basket['CampBooking']['total_paid'] <= 0) {
			return $this->_completeBooking($id);
		}

		if (Configure::read('Transactions.gateway') === 'stripe_checkout') {
			return $this->redirect(['action' => 'stripe_payment']);
		}

		$payment = [];
		if (Configure::read('BuzzPurchase.onsite') === true) {
			$payment = $this->request->data['Payment'];

			// Split name into first and second names for payment gateways.
			$fullName = trim($payment['card_name']);
			$splitName = explode(' ', $fullName);
			$payment['last_name'] = array_pop($splitName);
			$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
		}

		$transaction = $this->_takePayment($id, $basket, $payment);

		if (!empty($transaction['result'])) {
			// Payment taken successfully, complete purchase.
			return $this->_completeBooking($id);
		}

		$this->Session->setFlash(
			__d('buzz_camps', 'There was a problem processing your payment, please check your details and try again'),
			'flash_fail'
		);
	}

/**
 * Generic payment gateway callback.
 *
 * @param int $basketId Basket ID
 * @return void
 */
	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completeBooking($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_camps', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'index'
			]);
		}
	}

/**
 * Take a payment
 *
 * @param int $basketId Basket ID
 * @param array $basket Basket data
 * @param array $payment Payment data
 * @return mixed The response from Transactions::takePayment()
 */
	protected function _takePayment($basketId, $basket, $payment = []) {
		return $this->Transactions->takePayment(
			// Return URLs
			[
				'return' => Router::url(['action' => 'payment_callback', $basketId], true),
				'cancel' => Router::url(['action' => 'book', $basket['CampBooking']['camp_id']], true)
			],
			// Calling record
			[
				'model' => 'BuzzCamps.CampBooking',
				'model_id' => $basket['CampBooking']['id']
			],
			// Amount to be paid
			$basket['CampBooking']['total_paid'],
			// Items
			[],
			// Extras
			[
				'language' => Configure::read('Config.language'),
				'card' => $payment,
				'address' => $basket['AccountCredit']['CustomerAddress'],
				'user' => ['email' => $basket['AccountCredit']['CustomerAddress']['email']],
				'description' => __d(
					'buzz_camps',
					'%s Camp Booking %d',
					[Configure::read('SiteSetting.site_title'), $basket['CampBooking']['id']]
				)
			],
			0,
			Configure::read('BuzzPurchase.payment_gateway')
		);
	}

/**
 * Complete the credit transaction (after payment received)
 *
 * @param int $basketId Basket ID
 * @return void Redirects to confirmation page
 */
	protected function _completeBooking($basketId) {
		$CampBooking = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$CampBooking->setBasketId($basketId);
		$this->Session->write('CampBooking.hash', $CampBooking->hashBasketId());

		$CampBooking->completeBooking(true);

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

/**
 * Credit confirmation page.
 *
 * @param int|string $id Booking ID
 * @return void
 * @throws ForbiddenException
 */
	public function confirmation($id) {
		$CampBooking = $this->{$this->modelClass};

		if (
			$CampBooking->hashBasketId($id) !== $this->Session->read('CampBooking.hash')
			&& Configure::read('app.enviroment') !== 'DEVELOPMENT' // Allow us to refresh confirmation page on dev environments
		) {
			throw new ForbiddenException();
		}

		$this->Session->delete('CampBooking.hash');

		$booking = $CampBooking->readForView($id);

		if (Configure::check('BuzzCamps.camp_booking_confirmation_page_id') === true) {
			$this->assignPage(Configure::read('BuzzCamps.camp_booking_confirmation_page_id'));
		}

		$this->EcommerceTracking->addItem(
			$booking['Camp']['name'],
			'CA' . $booking['Camp']['id'],
			$booking['CampBooking']['total_paid'],
			1,
			['category' => 'Camp Booking']
		);
		$tax = 0;
		$ecommerceTracking = $this->EcommerceTracking->transaction(
			$booking['CampBooking']['sales_ref'],
			$booking['CampBooking']['total_paid'],
			CakeNumber::defaultCurrency(),
			$tax
		);
		$this->set(compact('ecommerceTracking'));

		$this->set(compact('booking'));
		$this->view = 'BuzzCamps.confirmation';
	}

/**
 * Populate lookups.
 *
 * @return void
 */
	protected function _populateLookups() {
		$CampBooking = $this->{$this->modelClass};

		$countries = $CampBooking->CustomerAddress->Country->translatedList();
		$usStates = $CampBooking->CustomerAddress->UsState->find('list');

		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());

		$this->set(compact('countries', 'usStates'));
		return;
	}

}
