<?php

App::uses('CakeEventListener', 'Event');

class BookingListener implements CakeEventListener {

	public function implementedEvents() {
		return array(
			'Model.Booking.completed' => 'confirmation',
			'Model.Booking.fallout' => 'fallout'
		);
	}

	/**
	 * Send emails on confirmation of a booking.
	 *
	 * @param CakeEvent $Event
	 * @return void
	 */
	public function confirmation(CakeEvent $Event) {
		$purchase = ClassRegistry::init('BuzzBookings.Booking')->getPurchase($Event->data['id']);

		// Send confirmation email
		$this->_sendConfirmation($purchase);

		// Send notification emails
		$this->_sendNotificationEmails($purchase);

		// If the booking is for today we want to send an email informing the
		// site of the booking.
		if (date('dmY', strtotime($purchase['Booking']['booking_date'])) === date('dmY')) {
			$this->_sendSameDayEmail($purchase);
		}

		return;
	}

	/**
	 * Sends a same day booking email.
	 *
	 * @param array $purchase
	 * @return void
	 */
	protected function _sendSameDayEmail($purchase) {
		// Send confirmation email
		$Email = ClassRegistry::init('BuzzEmails.Email');
		$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
			'BuzzBookings.same_day_booking',
			[
				'salesRef' => $purchase['Booking']['sales_ref'],
				'fullName' => $purchase['CustomerAddress']['first_name'] . ' ' . $purchase['CustomerAddress']['last_name'],
				'firstName' => $purchase['CustomerAddress']['first_name'],
				'lastName' => $purchase['CustomerAddress']['last_name'],
				'date' => date('j F Y', strtotime($purchase['Booking']['booking_date'])),
				'time' => date('H:i', strtotime($purchase['Booking']['booking_date']))
			]
		);
		if (! empty($email['Email']['to'])) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				$email['Email']['to'],
				$email['Email']['from']
			);
		}

		return;
	}

	protected function _sendConfirmation($purchase) {
		// Send confirmation email
		$Controller = new Controller();
		$Controller->helpers = ['BuzzTranslate.I18n', 'BuzzPurchase.Purchase'];
		$View = new View($Controller);
		$View->set('data', $purchase);
		$details = $View->render('BuzzBookings./Bookings/email_basket', 'ajax');

		$Email = ClassRegistry::init('BuzzEmails.Email');
		$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
			'BuzzBookings.confirmation_email',
			[
				'salesRef' => $purchase['Booking']['sales_ref'],
				'details' => $details,
				'fullName' => $purchase['CustomerAddress']['first_name'] . ' ' . $purchase['CustomerAddress']['last_name'],
				'firstName' => $purchase['CustomerAddress']['first_name'],
				'lastName' => $purchase['CustomerAddress']['last_name'],
				'checkin' => $purchase['Booking']['check_in_time'],
				'date' => date('j F Y', strtotime($purchase['Booking']['booking_date'])),
				'time' => date('H:i', strtotime($purchase['Booking']['booking_date']))
			]
		);
		$Email->queueEmail(
			$email['Email']['subject'],
			$email['Email']['content'],
			array($purchase['CustomerAddress']['email'] => $purchase['CustomerAddress']['first_name'] . ' ' . $purchase['CustomerAddress']['last_name']),
			$email['Email']['from']
		);

		return;
	}

	protected function _sendNotificationEmails($purchase) {
		// Send notification emails (if any are needed)
		$Controller = new Controller();
		$Controller->helpers = ['BuzzTranslate.I18n', 'BuzzPurchase.Purchase'];
		$View = new View($Controller);
		$View->set('data', $purchase);
		$details = $View->render('BuzzBookings./Bookings/email_basket', 'ajax');

		$emailAddresses = array();

		if (! empty($purchase['BookingItem'])) {
			foreach ($purchase['BookingItem'] as $item) {
				if (! empty($item['ActivityPackage']['email_notifications'])) {
					$emailAddresses += explode(',', $item['ActivityPackage']['email_notifications']);
				}
			}

			$emailAddresses = array_map('trim', $emailAddresses);

			if (! empty($emailAddresses)) {
				$emailAddresses = array_unique($emailAddresses);

				$Email = ClassRegistry::init('BuzzEmails.Email');
				$email = ClassRegistry::init('BuzzEmails.Email')->generateEmailData(
					'BuzzBookings.notification_email',
					[
						'salesRef' => $purchase['Booking']['sales_ref'],
						'details' => $details,
						'fullName' => $purchase['CustomerAddress']['first_name'] . ' ' . $purchase['CustomerAddress']['last_name'],
						'firstName' => $purchase['CustomerAddress']['first_name'],
						'lastName' => $purchase['CustomerAddress']['last_name'],
						'checkin' => $purchase['Booking']['check_in_time'],
						'date' => date('j F Y', strtotime($purchase['Booking']['booking_date'])),
						'time' => date('H:i', strtotime($purchase['Booking']['booking_date']))
					]
				);

				$Email->queueEmail(
					$email['Email']['subject'],
					$email['Email']['content'],
					$emailAddresses,
					$email['Email']['from']
				);
			}
		}

		return;
	}

	public function fallout(CakeEvent $Event) {
		$Booking = ClassRegistry::init('BuzzBookings.Booking');
		$booking = $Booking->getBasket($Event->data['id']);
		// Send confirmation email
		$Controller = new Controller();
		$Controller->helpers = ['BuzzTranslate.I18n', 'BuzzPurchase.Purchase'];
		$View = new View($Controller);
		$View->set('data', $booking);
		$details = $View->render('BuzzBookings./Bookings/email_basket', 'ajax');

		$Email = ClassRegistry::init('BuzzEmails.Email');
		$email = $Email->generateEmailData(
			'BuzzBookings.fallout_email',
			[
				'details' => $details,
				'checkin' => $booking['Booking']['check_in_time'],
				'date' => CakeTime::i18nFormat($booking['Booking']['booking_date'], '%e %B %Y'),
				'time' => CakeTime::i18nFormat($booking['Booking']['booking_date'], '%H:%M'),
				'url' => Router::url([
					'plugin' => 'buzz_bookings',
					'controller' => 'bookings',
					'action' => 'restore',
					$booking['Booking']['id'],
					$Booking->generateSignature($booking['Booking'])
				], true)
			]
		);
		if ($email !== false) {
			$Email->queueEmail(
				$email['Email']['subject'],
				$email['Email']['content'],
				array($booking['Booking']['email'] => $booking['Booking']['email']),
				$email['Email']['from']
			);
		}

		// Tell the API that the customer has abandoned their booking.
		ClassRegistry::init('BuzzBookings.BookingApi')->abandonSale(
			'Booking',
			$Event->data['id'],
			$booking['Booking']['email'],
			$booking['Booking']['sales_ref'],
			Configure::read('Config.language')
		);

		return;
	}

}
