<?php

App::uses('BuzzBookingsAppModel', 'BuzzBookings.Model');

class ActivityPackage extends BuzzBookingsAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzTranslate.Translatable'
	);

	/**
	 * Belongs to associations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'Activity' => array(
			'className' => 'BuzzBookings.Activity'
		)
	);

	/**
	 * HABTM associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'UpsellApplicableActivity' => array(
			'className' => 'BuzzBookings.Activity',
			'joinTable' => 'activity_packages_upsell_applicable_activities'
		),
		'UpsellExclusion' => array(
			'className' => 'BuzzBookings.ActivityPackage',
			'joinTable' => 'activity_packages_upsell_exclusions',
			'foreignKey' => 'activity_package_id',
			'associationForeignKey' => 'upsell_package_id'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'api_reference' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'multi_experience' => array(
				'rule' => 'validateApiReferences',
				'message' => 'Invalid reference codes'
			),
			'max' => array(
				'rule' => array('maxLength', 80),
				'message' => 'No more than 80 characters long',
			)
		),
		'api_additional_notes' => array(
			'max' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more than 250 characters long',
				'allowEmpty' => true
			)
		),
		'api_discount_code' => array(
			'max' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters long',
				'allowEmpty' => true
			)
		),
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters long'
			)
		),
		'activity_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'upsells' => array(
				'rule' => 'validateActivityForUpsells',
				'message' => 'Multi-experience packages cannot have upsells'
			)
		),
		'peak_price' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'decimal' => array(
				'rule' => '|^\d+(\.\d{2})?$|',
				'message' => 'Decimal numbers only'
			),
			'max' => array(
				'rule' => array('comparison', '<', 1000000),
				'message' => 'Value must be less than 1000000.00'
			)
		),
		'off_peak_price' => array(
			'decimal' => array(
				'rule' => '|^\d+(\.\d{2})?$|',
				'message' => 'Decimal numbers only',
				'allowEmpty' => true
			),
			'max' => array(
				'rule' => array('comparison', '<', 1000000),
				'message' => 'Value must be less than 1000000.00'
			)
		),
		'upsell_price' => array(
			'require' => array(
				'rule' => 'validateRequireForUpsells',
				'message' => 'Required'
			),
			'decimal' => array(
				'rule' => '|^\d+(\.\d{2})?$|',
				'message' => 'Decimal numbers only',
				'allowEmpty' => true
			),
			'max' => array(
				'rule' => array('comparison', '<', 1000000),
				'message' => 'Value must be less than 1000000.00'
			)
		),
		'sequence' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('naturalNumber', true)
			)
		)
	);

	/**
	 * Image slots
	 *
	 * @var array
	 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		]
	];

	/**
	 * Read for edit
	 *
	 * @param int $id
	 * @param array $query
	 * @return array
	 */
	public function readForEdit($id, $query = []) {
		$query['contain'][] = 'UpsellApplicableActivity';
		$query['contain'][] = 'UpsellExclusion';
		return parent::readForEdit($id, $query);
	}

	/**
	 * Custom validation method for checking the supplied API references.
	 *
	 * Only multi-experience activities can have a list of reference numbers.
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validateApiReferences(array $data) {
		$check = array_pop($data);

		if (!empty($this->data[$this->alias]['activity_id'])) {

			if ((int)$this->data[$this->alias]['activity_id'] !== $this->Activity->getMultiExperience()) {
				return preg_match('|^\d+$|', $check) === 1;
			} else {
				return preg_match('|^(\d+\[\d+(\.\d{2})?\],?\ ?)+$|', $check) === 1;
			}

		}

		return;
	}

	/**
	 * Custom validation method for checking the activity is appropriate for
	 * upsells.
	 *
	 * Multi-experience activities cannot be used as upsells.
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validateActivityForUpsells(array $data) {
		$check = array_pop($data);

		if ((int)$check === $this->Activity->getMultiExperience()) {
			return empty($this->data['UpsellApplicableActivity']['UpsellApplicableActivity']);
		}

		return true;
	}

	/**
	 * Require field if there are upsells on the package.
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validateRequireForUpsells(array $data) {
		$check = array_pop($data);

		if (!empty($this->data['UpsellApplicableActivity']['UpsellApplicableActivity'])) {
			return !empty($check);
		}

		return true;
	}

	/**
	 * After find logic
	 *
	 * @param array $results
	 * @param bool $primary
	 * @return array
	 */
	public function afterFind($results, $primary = false) {
		foreach ($results as &$result) {
			// For packages with a single price create a price attribute. This
			// is to simplify things elsewhere.
			if (!empty($result[$this->alias]['peak_price'])) {
				$result[$this->alias]['price'] = null;
				if (
					$result[$this->alias]['peak_price'] === $result[$this->alias]['off_peak_price']
					|| empty($result[$this->alias]['off_peak_price'])
					|| $this->Activity->getMultiExperience() === (int)$result[$this->alias]['activity_id']
				) {
					$result[$this->alias]['price'] = $result[$this->alias]['peak_price'];
				}
			}
		}
		return $results;
	}

	/**
	 * Returns all available private hire packages.
	 *
	 * @return array
	 */
	public function getPrivateHirePackages() {
		return $this->find('all', $this->_privateHirePackagesParams());
	}

	/**
	 * Returns all available private hire packages.
	 *
	 * @return bool
	 */
	public function hasPrivateHirePackages() {
		return (bool)$this->find('count', $this->_privateHirePackagesParams());
	}

	/**
	 * Returns an array of parameters for searching for private hire packages.
	 *
	 * @return array
	 */
	protected function _privateHirePackagesParams() {
		return [
			'contain' => [
				'Activity',
				'Image'
			],
			'conditions' => [
				'OR' => [
					'ActivityPackage.api_reference <>' => null,
					'ActivityPackage.api_reference <>' => ''
				],
				// Private hire packages only.
				'ActivityPackage.participants' => 0,
				'ActivityPackage.is_active' => true
			],
			'order' => [
				'ActivityPackage.is_special_offer' => 'DESC',
				'ActivityPackage.sequence' => 'ASC'
			]
		];
	}

	/**
	 * Get bookable upsells for our activity.
	 *
	 * @param int $activityId
	 * @param array $packages Activity package IDs currently in basket
	 * @return array
	 */
	public function getUpsellPackages($activityId, array $packages) {
		return $this->find(
			'all',
			[
				'contain' => [
					'Image'
				],
				'joins' => [
					[
						'table' => 'activity_packages_upsell_applicable_activities',
						'alias' => 'ActivityPackagesUpsellApplicableActivities',
						'conditions' => [
							'ActivityPackage.id = ActivityPackagesUpsellApplicableActivities.activity_package_id',
							'ActivityPackagesUpsellApplicableActivities.activity_id' => $activityId
						]
					],
					[
						'table' => 'activity_packages_upsell_exclusions',
						'alias' => 'ActivityPackagesUpsellExclusions',
						'type' => 'LEFT',
						'conditions' => [
							'ActivityPackage.id = ActivityPackagesUpsellExclusions.activity_package_id',
							'ActivityPackagesUpsellExclusions.upsell_package_id' => $packages
						]
					]
				],
				'conditions' => [
					'ActivityPackage.is_active' => true,
					'ActivityPackagesUpsellExclusions.id' => null
				]
			]
		);
	}

}
