<?php

App::uses('BuzzBookingsAppController', 'BuzzBookings.Controller');
App::uses('CustomerAddress', 'BuzzCustomers.Model');
App::uses('PurchasableInterface', 'BuzzPurchase.Lib/Interface');
App::uses('TimerTrait', 'BuzzBookings.Lib/Trait');

class BookingsController extends BuzzBookingsAppController implements PurchasableInterface {

	use TimerTrait;

/**
 * List of admin actions the controller supports
 *
 * @var array
 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_email_confirmation'
	);

/**
 * Components
 *
 * @var array
 */
	public $components = [
		'BuzzPurchase.FbPixel',
		'BuzzPurchase.GaEcommerce',
		'BuzzPurchase.Purchase',
		'Transactions.Transactions'
	];

/**
 * Helpers
 */
	public $helpers = array(
		'Address.Address',
		'BuzzBookings.Voucher'
	);

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'ajax_dates',
			'ajax_remove_voucher',
			'ajax_times',
			'ajax_voucher',
			'basket',
			'checkout',
			'confirmation',
			'date',
			'extras',
			'insurance',
			'index',
			'packages',
			'payment_callback',
			'private_hire',
			'quit',
			'remove_vouchers',
			'restore',
			'timeout'
		));
	}

/**
 * Bookings landing page - will mostly get bypassed by homepage/activity pages
 *
 * @param int|bool $redeemVouchers Flag if the booking has vouchers
 * @return void
 */
	public function index($redeemVouchers = false) {
		$Booking = $this->{$this->modelClass};
		if (!empty($this->request->data)) {
			// Used to redirect to the relevant step from the 'Book Now' widget on the homepage.
			if (!empty($this->request->data['Booking']['activity_id'])) {
				// Forward to the correct packages step.
				return $this->redirect([
					'action' => 'packages',
					$this->request->data['Booking']['activity_id'],
					0,
					$redeemVouchers
				]);
			} else {
				$this->Session->setFlash(
					__d('buzz_bookings', 'Please select an activity'),
					'flash_fail'
				);
			}
		}
		$activities = $Booking->Activity->getListed();
		if (count($activities) === 1) {
			return $this->redirect([
				'action' => 'packages'
			]);
		} elseif (empty($activities)) {
			// No activities have been setup or created so throw an exception
			throw new NotFoundException();
		}
		$this->set(compact('activities', 'redeemVouchers'));
		$this->set(
			'hasPrivateHirePackages',
			$Booking->Activity->ActivityPackage->hasPrivateHirePackages()
		);
		if ($redeemVouchers) {
			$this->Meta->canonical(['action' => 'index', $redeemVouchers]);
		} else {
			$this->Meta->canonical(['action' => 'index']);
		}
		$this->view = 'BuzzBookings./Bookings/index';
		return;
	}

/**
 * Packages step - add packages and vouchers to a booking
 *
 * To jump to the packages step for a specific activity the activity ID can be
 * passed; otherwise the activity ID is set on the booking.
 *
 * To preselect a specific package a package ID needs to be passed.
 *
 * @param int $activityId Activity ID
 * @param int $packageId Optional activity package ID
 * @param int|bool $redeemVouchers Flag if the booking has vouchers
 * @return void
 */
	public function packages($activityId = null, $packageId = null, $redeemVouchers = false) {
		$Booking = $this->{$this->modelClass};

		if ((int)$activityId === -1) {
			// Need to redirect to the private hire booking process.
			return $this->redirect(['action' => 'private_hire']);
		}

		// Get the basket.
		$basket = $Booking->getBasket();

		if (empty($activityId)) {
			if (!empty($basket['Booking']['activity_id'])) {
				// Use the basket's activity.
				$activityId = $basket['Booking']['activity_id'];
			} else {
				// Check if the site is configured to only have one listed
				// bookable activity. This is needed to prevent a redirect
				// loop with this and the previous step.
				$activityId = $Booking->Activity->onlyBookableActivity();
			}
		}

		// Process form data.
		if (!empty($this->request->data) && !empty($activityId)) {
			$result = $Booking->createBasket(
				$activityId,
				!empty($this->request->data['BookingItem']) ? $this->request->data['BookingItem'] : []
			);
			if ($result === true) {
				return $this->redirect(['action' => 'date']);
			} else {
				$this->Session->setFlash(
					__d('buzz_bookings', 'Please select a package or add a voucher'),
					'flash_fail'
				);
			}
		}

		// If the activity ID hasn't been set then redirect to the previous
		// step.
		if (empty($activityId)) {
			return $this->redirect([
				'action' => 'index'
			]);
		} else {
			// Set the booking's activity ID.
			$this->request->data['Booking']['activity_id'] = $activityId;
		}

		// Get the packages and activity details for generating the form.
		$data = $Booking->Activity->getBookableActivity($activityId);

		if (empty($data)) {
			throw new NotFoundException();
		}

		if (
			!empty($basket) && (int)$basket['Booking']['activity_id'] !== (int)$activityId
		) {
			// Clear the basket as we're either changing the activity being booked so
			// existing items are no longer valid.
			$Booking->clearBasket(false);
			// We don't need to pass the basket to the view anymore as we're
			// essentially starting a fresh basket.
			$basket = [];
		} elseif (empty($this->request->data['BookingItem']) && !empty($basket['BookingItem'])) {
			// Populate the form with existing booking items.
			$this->request->data['BookingItem'] = Hash::combine($basket['BookingItem'], '{n}.activity_package_id', '{n}');
		}

		// If someone's followed a link to directly book a specific package
		// then set the quantity to 1.
		if (!empty($packageId) && empty($this->request->data['BookingItem'])) {
			$this->request->data['BookingItem'][$packageId]['quantity'] = 1;
		}

		$previousStep = $this->_packagesPreviousStep($basket, $data, (bool)$redeemVouchers);

		// Get the available vouchers.
		$this->_populateVouchers($activityId, $data['Activity']['api_reference'], true);
		$this->set(
			'hasPrivateHirePackages',
			$Booking->Activity->ActivityPackage->hasPrivateHirePackages()
		);
		$this->set(
			'isMultiExperience',
			$Booking->Activity->getMultiExperience() === (int)$data['Activity']['id']
		);
		$this->set('redeemVouchers', (bool)$redeemVouchers);
		$this->set(compact('data', 'basket', 'previousStep'));
		$this->view = 'BuzzBookings./Bookings/packages';

		return;
	}

/**
 * Get the previous step for packages
 * @param array $basket The booking data
 * @param array $data The activity data
 * @param bool $redeemVouchers True if redeeming vouchers
 * @return array Router array for previous step
 */
	protected function _packagesPreviousStep($basket, $data, $redeemVouchers) {
		$Booking = $this->{$this->modelClass};
		$activities = $Booking->Activity->getListed();
		$previousStep = ['action' => 'index'];
		// Check if this is a hidden activity (i.e. we will need to return the user to the activity
		// page rather than allow them to select a different activity). We'll also deal with sites
		// with only one bookable activity here too.
		if (
			!empty($basket['Activity']['is_hidden_activity'])
			|| (bool)$data['Activity']['is_hidden_activity'] === true
			|| count($activities) === 1
		) {
			$previousStep = ['controller' => 'activities', 'action' => 'view', $data['Activity']['id']];
		} elseif ($redeemVouchers === true) {
			// This is a redeem voucher booking.
			$previousStep[] = 1;
		}
		return $previousStep;
	}

/**
 * Private Hire step - add package or voucher to a private hire booking.
 *
 * @param int $packageId Optional activity package ID
 * @return void
 */
	public function private_hire($packageId = null, $redeemVouchers = false) {
		$Booking = $this->{$this->modelClass};

		// If someone's followed a link to directly book a specific package
		// then we want to automatically submit the data for creating the
		// basket.
		if (!empty($packageId) && $packageId > 0 && empty($this->request->data['BookingItem'])) {
			$this->request->data['BookingItem'][$packageId]['activity_package_id'] = $packageId;
			$this->request->data['BookingItem'][$packageId]['quantity'] = 1;
		}

		// Process form data.
		if (!empty($this->request->data)) {
			$this->request->data = Hash::insert($this->request->data, 'BookingItem.{n}.quantity', 1);
			$result = $Booking->createBasket(
				!empty($this->request->data['Booking']['activity_id']) ? $this->request->data['Booking']['activity_id'] : -1,
				!empty($this->request->data['BookingItem']) ? $this->request->data['BookingItem'] : [],
				true
			);
			if ($result === true) {
				return $this->redirect(['action' => 'date']);

			} else {
				$this->Session->setFlash(
					__d('buzz_bookings', 'Please select a package or add a voucher'),
					'flash_fail'
				);

			}

		}

		// Get the basket.
		$basket = $Booking->getBasket();

		// Get the packages and activity details for generating the form.
		$data = $Booking->Activity->ActivityPackage->getPrivateHirePackages();

		if (empty($data)) {
			throw new NotFoundException();
		}

		if (
			!empty($basket)
		) {
			// Clear the basket as we're either changing the activity being booked so
			// existing items are no longer valid.
			$Booking->clearBasket(false);
			// We don't need to pass the basket to the view anymore as we're
			// essentially starting a fresh basket.
			$basket = [];
		}

		// Get the available vouchers.
		$this->_populateVouchers(-1, -1, true);
		$this->set('redeemVouchers', (bool)$redeemVouchers);
		$this->set(compact('data', 'basket'));
		$this->view = 'BuzzBookings./Bookings/private_hire';

		return;
	}

/**
 * Removes all vouchers from a booking. To be used by the multi-experience
 * voucher form.
 *
 * @param bool $isPrivateHire
 * @return void
 */
	public function remove_vouchers($isPrivateHire = false) {
		$Booking = $this->{$this->modelClass};
		$Booking->removeVouchers();
		if ((bool)$isPrivateHire === true) {
			$redirectUrl = ['action' => 'private_hire'];
		} else {
			$redirectUrl = ['action' => 'packages', $Booking->Activity->getMultiExperience()];
		}
		$this->redirect($redirectUrl);
		return;
	}

/**
 * Date step - book a time and date for the booking.
 *
 * @return void
 */
	public function date() {
		$Booking = $this->{$this->modelClass};

		$this->Session->delete('BuzzBookings.extrasAvailable');

		// Get the basket.
		$basket = $Booking->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'index']);
		} elseif (empty($basket['BookingItem'])) {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Please select a package or add a voucher'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'packages', $basket['Booking']['activity_id']]);
		}

		if ($Booking->clearTemporaryBookings() === false) {
			throw new InternalErrorException();
		}

		if (!empty($this->request->data)) {

			$data = $this->Session->read('BuzzBookings.availableDates.' . $this->request->data['Booking']['date'] . '.' . $this->request->data['Booking']['time']);

			if ($Booking->addTemporaryBooking($basket, $this->request->data['Booking']['date'], $data ?: []) === true) {
				// Get rid of the available dates from the session (these will
				// need regenerating if the customer returns to this step).
				$this->Session->delete('BuzzBookings.availableDates');
				// Start the booking timer.
				$this->_startTimer();
				// Redirect to the next step.
				return $this->redirect(['action' => 'extras']);
			} else {
				// The booking failed, so we want to ask the customer to pick a
				// new date/time.
				$this->Session->setFlash(
					__d('buzz_bookings', 'Your selected booking is no longer available, please try again'),
					'flash_fail'
				);
				return $this->redirect(['action' => 'date']);
			}

		}

		if (!empty($basket['Booking']['is_private_hire'])) {
			$previousStep = ['action' => 'private_hire'];
		} else {
			$previousStep = ['action' => 'packages', $basket['Booking']['activity_id']];
		}

		$this->set(compact('basket', 'previousStep'));
		$this->view = 'BuzzBookings./Bookings/date';

		return;
	}

/**
 * Extras step - add upsells to a booking.
 *
 * This step gets skipped for private hire bookings (check is inside this
 * action) as private hire bookings do not have upsells.
 *
 * @return void
 */
	public function extras() {
		$Booking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $Booking->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'index']);
		} elseif (empty($basket['Booking']['booking_date'])) {
			return $this->redirect(['action' => 'date']);
		}

		if (!empty($this->request->data)) {
			$result = $Booking->addUpsells(
				!empty($this->request->data['BookingExtra']) ? $this->request->data['BookingExtra'] : [],
				!empty($this->request->data['BookingItem']) ? $this->request->data['BookingItem'] : [],
				$this->Session->read('BuzzBookings.bookableUpsells')
			);
			if ($result === true) {
				return $this->redirect(['action' => 'insurance']);
			}
			$this->Session->setFlash(
				__d('buzz_bookings', 'There was a problem adding your extras to your booking'),
				'flash_fail'
			);
		} else {
			// We need to clear the extras from the basket. They will get
			// re-added when the user proceeds to the next step, but we need
			// to make sure that any bookable items can still be booked before
			// proceeding.
			$Booking->clearExtras();
			// Set form data.
			$this->request->data['BookingExtra'] = Hash::combine($basket['BookingExtra'], '{n}.extra_id', '{n}');
			// Get the updated basket for the View.
			$basket = $Booking->getBasket();
		}

		$participants = $Booking->getParticipants();
		if (empty($participants)) {
			// If we don't know the number of participants for the booking (i.e.
			// a private hire or group booking we skip this step).
			return $this->redirect(['action' => 'insurance']);
		}
		$this->set('participants', array_combine(range(0, $participants), range(0, $participants)));

		$bookableUpsells = $Booking->getUpsells($basket);
		// Get the extras for all non-private hire bookings (we do not upsell
		// for private hire bookings as the activity type is not always set,
		// i.e. when using a private hire voucher).
		$extras = $basket['Booking']['is_private_hire'] !== true ? $Booking->getExtras($basket) : null;
		$this->Session->write(
			'BuzzBookings.extrasAvailable',
			!empty($bookableUpsells) || !empty($extras)
		);
		$this->Session->write(
			'BuzzBookings.bookableUpsells',
			Hash::combine($bookableUpsells, '{n}.ActivityPackage.id', '{n}.Booking')
		);
		if (empty($bookableUpsells) && empty($extras)) {
			return $this->redirect(['action' => 'insurance']);
		}

		$previousStep = ['action' => 'date'];

		$this->set(compact('basket', 'bookableUpsells', 'extras', 'previousStep', '_checkTimeRemaining'));
		$this->view = 'BuzzBookings./Bookings/extras';

		return;
	}

/**
 * Insurance step
 *
 * @return void
 */
	public function insurance() {
		$Booking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $Booking->getBasket();

		// Check we're ready for this step.
		if (empty($basket['Booking']['booking_date'])) {
			return $this->redirect(['action' => 'date']);
		}

		if (!empty($this->request->data)) {
			$result = $Booking->updateBookingInsurance($this->request->data['Booking']['is_insured']);
			if ($result === true) {
				return $this->redirect(['action' => 'basket']);
			} else {
				$this->Session->setFlash(__d('buzz_bookings', 'Please select one of the options below', 'flash_fail'));
			}
		} else {
			// Remove the insurance when first landing on the page. We need to
			// reload the basket to make sure we have the correct basket total.
			$Booking->updateBookingInsurance(false);
			$basket = $Booking->getBasket();
		}

		// Get the insurance cost for the basket.
		$insuranceCost = (float)$Booking->getInsuranceCost($basket);
		if (empty($insuranceCost)) {
			return $this->redirect(['action' => 'basket']);
		}

		$previousStep = $this->Session->read('BuzzBookings.extrasAvailable') ? ['action' => 'extras'] : ['action' => 'date'];

		// Determine the step number based on previous steps.
		$step = 4;
		if ($this->Session->read('BuzzBookings.extrasAvailable') === false) {
			--$step;
		}

		$this->set(compact('basket', 'insuranceCost', 'previousStep', 'step'));
		$this->view = 'BuzzBookings.insurance';

		return;
	}

/**
 * Basket step
 *
 * @return void
 */
	public function basket() {
		$Booking = $this->{$this->modelClass};


		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $Booking->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'index']);
		} elseif (empty($basket['Booking']['booking_date'])) {
			return $this->redirect(['action' => 'date']);
		}

		if (! empty($basket['Booking']['discount_code'])) {
			$Booking->applyDiscountCode($basket['Booking']['id'], $basket['Booking']['discount_code']);
			$basket = $Booking->getBasket();
		}

		if (!empty($this->request->data)) {
			if (! empty($this->request->data['Booking']['discount_code'])) {
				if (
					$Booking->applyDiscountCode($this->request->data['Booking']['id'],
					$this->request->data['Booking']['discount_code'])) {

					$this->Session->setFlash(
						__d('buzz_bookings', 'Your discount has been applied to your basket.'),
						'flash_success'
					);
				} else {
					$this->Session->setFlash(
						__d('buzz_bookings', 'The discount code entered was not valid for any of the items in your basket.'),
						'flash_fail'
					);
				}

				return $this->redirect($this->here);
			}

			return $this->redirect(['action' => 'checkout']);
		} else {
			// We set the basket ID for the form just so that there's some data
			// to submit in the form to handle redirecting to the next step.
			// This value should *not* be used directly, always retrieve the
			// basket ID from the session!
			$this->request->data['Booking']['id'] = $Booking->getBasketId();
		}

		$previousStep = ['action' => 'date'];
		if ((float)$basket['Booking']['insurance_cost'] > 0) {
			$previousStep = ['action' => 'insurance'];
		} elseif ($this->Session->read('BuzzBookings.extrasAvailable')) {
			$previousStep = ['action' => 'extras'];
		}

		// Determine the step number based on previous steps.
		$step = 5;
		if ((float)$basket['Booking']['insurance_cost'] < 0.01) {
			--$step;
		}
		if ($this->Session->read('BuzzBookings.extrasAvailable') === false) {
			--$step;
		}

		$this->set(compact('basket', 'previousStep', 'step'));
		$this->view = 'BuzzBookings.basket';

		$this->set('nextStep', array(
			'admin' => false,
			'plugin' => 'buzz_bookings',
			'controller' => 'bookings',
			'action' => 'checkout'
		));

		if (Configure::read('BuzzDiscount.disableDiscountForm') == false) {
			$this->set('showDiscountForm', true);
		}

		return;
	}

/**
 * Allow customers to quit the booking and have booking details emailed to
 * them.
 *
 * @return void
 */
	public function quit() {
		$Booking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $Booking->getBasket();

		if (!empty($this->request->data)) {
			if ($Booking->fallout($this->request->data['Booking']['email']) === true) {
				$this->Ga->triggerEvent('Checkout', 'Send Email Summary');
				$this->Session->setFlash(
					__d('buzz_bookings', 'We\'ve emailed details of your booking to you'),
					'flash_success'
				);
				return $this->redirect('/');
			} else {
				$this->Session->setFlash(
					__d('buzz_bookings', 'An invalid email was supplied'),
					'flash_fail'
				);
				return $this->redirect(['action' => 'basket']);
			}
		}

		$this->view = 'BuzzBookings.quit';
		$this->layout = 'ajax';

		return;
	}

/**
 * Restores a quit booking from a fallout email
 *
 * @param int $id
 * @param string $signature Hashed email address and booking ID
 * @return void
 */
	public function restore($id, $signature) {
		$Booking = $this->{$this->modelClass};

		$data = $Booking->restoreBooking($id, $signature);

		if (!empty($data)) {
			// The basket has been restored.
			if (empty($data['Booking']['booking_date'])) {
				$this->Session->setFlash(
					__d('buzz_bookings', 'Sorry, your booking is no longer available'),
					'flash_fail'
				);
				return $this->redirect(['action' => 'date']);
			}
			// Start the booking timer.
			$this->_startTimer();
			// Continue to where the customer left off.
			$this->Session->setFlash(
				__d('buzz_bookings', 'Your booking has been restored'),
				'flash_success'
			);
			return $this->redirect(['action' => 'basket']);
		} else {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Sorry, we were unable to find this booking'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'index']);
		}
	}

/**
 * AJAX call for generating voucher fields.
 *
 * @param int $activityId Activity ID
 * @return void
 */
	public function ajax_voucher($activityId) {
		$Booking = $this->{$this->modelClass};

		// Check if we're dealing with a multi-experience booking.
		$isMultiExperience = $Booking->Activity->getMultiExperience() === (int)$activityId;
		// Is this a private hire booking (these use a modified booking process).
		$isPrivateHire = $activityId < 0;

		$result = false;
		if (!empty($this->request->query['data'])) {
			$result = $Booking->addVoucher($activityId, $data = $this->request->query['data']['BookingItemVoucher']);
			if ($result !== false) {
				$this->set('voucher', $result);

				// For multi-experience and private hire bookings we restrict
				// the booking to just one voucher.
				if ($isMultiExperience === true || $isPrivateHire === true) {
					$this->set('hideButton', true);
				}

				$this->set(compact('activityId', 'isMultiExperience', 'isPrivateHire'));
				$this->layout = 'ajax';
				$this->view = 'BuzzBookings.ajax_voucher_added';

				return;
			} else {
				// Set form data.
				$this->request->data = $this->request->query['data'];
				// Invalidate fields
				$Booking->BookingItem->BookingItemVoucher->invalidate('voucher_code');
				$Booking->BookingItem->BookingItemVoucher->invalidate('expiry_date');
			}
		}

		$this->_populateVouchers($activityId);

		$this->set(compact('activityId', 'isMultiExperience', 'isPrivateHire'));
		$this->layout = 'ajax';
		$this->view = 'BuzzBookings.ajax_voucher';

		return;
	}

/**
 * AJAX call for removing a voucher from a booking.
 *
 * @param int $bookingItemId Booking item record ID
 * @return void
 */
	public function ajax_remove_voucher($bookingItemId) {
		$Booking = $this->{$this->modelClass};

		$this->autoRender = false;

		// Remove the voucher/booking item (this checks it belongs to the user
		// too).
		$result = $Booking->removeVoucher($bookingItemId);

		return json_encode($result);
	}

/**
 * AJAX call for generating the available times and dates for the current
 * booking.
 *
 * @return void
 */
	public function ajax_dates() {
		$Booking = $this->{$this->modelClass};

		// Get the available times and dates.
		$months = (int)Configure::read('BuzzBookings.diary_months');
		$months = $months ?: 3;
		$fromDate = date('Y-m-d');
		$toDate = date('Y-m-t', strtotime("+$months months"));
		$data = $Booking->getAvailableDates($fromDate, $toDate);

		if (!empty($data)) {
			// Work out the unavailable dates within our date range.
			$unavailableDates = $Booking->calculateUnavailableDates($data, $fromDate, $toDate);

			// Store the available dates/times in the session.
			$this->Session->write('BuzzBookings.availableDates', Hash::combine($data, '{n}.date', '{n}.times'));

			$this->set(compact('unavailableDates', 'fromDate', 'toDate'));
			$view = 'ajax_dates';
		} else {
			$view = 'ajax_dates_unavailable';
		}

		$this->layout = 'ajax';
		$this->view = 'BuzzBookings./Bookings/' . $view;

		return;
	}

/**
 * AJAX call for displaying available times.
 *
 * @param string $date
 * @return void
 */
	public function ajax_times($date) {
		$Booking = $this->{$this->modelClass};

		$data = $this->Session->read('BuzzBookings.availableDates.' . $date);

		// If no data is returned we don't want to return anything. This is to
		// take into account the date picker loading with the current date and
		// no times being available.
		if (empty($data)) {
			$this->autoRender = false;
			return;
		}

		$this->set(compact('data', 'date'));
		$this->layout = 'ajax';
		$this->view = 'BuzzBookings./Bookings/ajax_times';

		return;
	}

/**
 * Checkout step
 *
 * @return void
 */
	public function checkout() {
		$Booking = $this->{$this->modelClass};

		$this->_checkTimeRemaining();

		// Get the basket.
		$basket = $Booking->getBasket();
		if (empty($basket)) {
			$this->Session->setFlash(
				__d('buzz_bookings', 'Your session has expired'),
				'flash_fail'
			);
			return $this->redirect(['action' => 'index']);
		} elseif (empty($basket['Booking']['booking_date'])) {
			return $this->redirect(['action' => 'date']);
		}

		if (!empty($this->request->data)) {

			$result = $Booking->addBillingDetails($this->request->data['CustomerAddress']);
			$this->loadModel('BuzzPurchase.Payment');
			$this->Payment->set($this->request->data);
			$paymentValidates = $this->Payment->validates();
			if ($result === true && $paymentValidates === true) {
				// Take payment.
				$this->_pay();
			} else {
				$this->Session->setFlash(
					__d('buzz_bookings', 'Please correct the errors below'),
					'flash_fail'
				);
			}

		} else {

			// Set the address field defaults (these are configured by the
			// CustomerAddress plugin to make it easier to override for each site).
			$this->request->data['CustomerAddress']['country_id'] = CustomerAddress::getDefaultCountry();
			$this->request->data['CustomerAddress']['us_state_id'] = CustomerAddress::getDefaultUsState();

		}

		// Get the basket.
		$basket = $Booking->getBasket();

		// Get gift voucher purchase conditions
		$this->loadModel('BuzzConditions.Condition');
		// Prefix conditions with a default arrival time condition.
		$conditions = [
			- 1 => [
				'Condition' => [
					'id' => 0,
					'rendered_content' => __d(
						'buzz_bookings',
						'I understand that I need to arrive by %s in order to take part in my first activity',
						$basket['Booking']['check_in_time']
					)
				]
			]
		];
		$conditions += $this->Condition->getConditions('Booking');

		$previousStep = ['action' => 'basket'];

		$this->set(compact('basket', 'conditions', 'previousStep'));
		$this->set('cardTypes', $this->Purchase->getCardTypes());
		$this->set('months', $this->Purchase->getMonths());
		$this->set('years', $this->Purchase->getYears());
		$this->set('pastYears', $this->Purchase->getPastYears());
		$this->_populateLookups();
		$this->view = 'BuzzBookings.checkout';

		return;
	}

/**
 * Processes the payment and triggers the completion of the payment on success
 *
 * @return void
 */
	protected function _pay() {
		$Booking = $this->{$this->modelClass};

		// Get the latest basket.
		$basketId = $Booking->getBasketId();
		$basket = $Booking->getBasket();

		if ($basket['Booking']['total_cost'] > 0) {

			if (Configure::read('BuzzPurchase.onsite') === true) {
				$payment = $this->request->data['Payment'];

				// Split name into first and second names for payment gateways.
				$fullName = trim($payment['card_name']);
				$splitName = explode(' ', $fullName);
				$payment['last_name'] = array_pop($splitName);
				$payment['first_name'] = count($splitName) ? implode(' ', $splitName) : '';
			} else {
				$payment = [];
			}

			$transaction = $this->Transactions->takePayment(
				// Return URLs
				array(
					'return' => Router::url(['action' => 'payment_callback', $basketId], true),
					'cancel' => Router::url(['action' => 'checkout'], true)
				),
				// Calling record
				array(
					'model' => 'BuzzBookings.Booking',
					'model_id' => $basket['Booking']['id']
				),
				// Amount to be paid
				$basket['Booking']['total_cost'],
				// Items
				array(),
				// Extras
				array(
					'language' => Configure::read('Config.language'),
					'card' => $payment,
					'address' => $basket['CustomerAddress'],
					'user' => array('email' => $basket['CustomerAddress']['email']),
					'description' => __d(
						'buzz_bookings',
						'%s Booking %d',
						[Configure::read('SiteSetting.site_title'), $basket['Booking']['id']]
					)
				),
				0,
				Configure::read('Transactions.gateway')
			);

			if (!empty($transaction['result'])) {
				// Payment taken successfully, complete purchase.
				return $this->_completePurchase($basketId);
			} else {
				$this->Session->setFlash(
					__d(
						'buzz_bookings',
						'There was a problem processing your payment, please check your details and try again'
					),
					'flash_fail'
				);
			}

		} else {
			// Nothing to pay, complete purchase.
			return $this->_completePurchase($basketId);
		}

		return;
	}

/**
 * Complete the purchase (after payment received)
 *
 * @param int $basketId Basket ID
 * @return void Redirects to confirmation page
 */
	protected function _completePurchase($basketId) {
		$Booking = $this->{$this->modelClass};
		if (empty($basketId)) {
			throw new ForbiddenException();
		}
		// The original session may have expired if the customer has sat on an off-site payment
		// window for a long time so we need to restart the session.
		$Booking->setBasketId($basketId);
		$Booking->completePurchase(true);

		// The hash *must* be set after completing the purchase as
		// completePurchase() will delete the session.
		$this->Session->write('Booking.hash', $Booking->hashBasketId($basketId));

		return $this->redirect([
			'action' => 'confirmation',
			$basketId
		]);
	}

/**
 * Confirmation page
 *
 * @param int $id Basket ID
 * @return void
 */
	public function confirmation($id) {
		$Booking = $this->{$this->modelClass};

		if ($Booking->hashBasketId($id) === $this->Session->read('Booking.hash') || Configure::read('app.enviroment') == 'DEVELOPMENT') {

			$this->Session->delete('Booking.hash');

			$basket = $Booking->getPurchase($id);

			if (Configure::check('BuzzBookings.confirmation_page_id') === true) {
				$this->assignPage(Configure::read('BuzzBookings.confirmation_page_id'));
			}

			// Set up the GA ecommerce tracking variables.
			$gaTransaction = $this->GaEcommerce->transaction(
				$basket['Booking']['sales_ref'],
				$basket['Booking']['total_cost']
			);
			foreach ($basket['BookingItem'] as $item) {
				if (!empty($item['ActivityPackage']['id'])) {
					$this->GaEcommerce->addItem(
						$basket['Booking']['sales_ref'],
						$item['ActivityPackage']['name'],
						'B' . $item['ActivityPackage']['id'],
						$item['item_unit_cost'],
						$item['quantity'],
						'Booking'
					);
					$this->FbPixel->addItem('B' . $item['ActivityPackage']['id']);
				}
			}
			foreach ($basket['BookingExtra'] as $item) {
				if (!empty($item['Extra']['id'])) {
					$this->GaEcommerce->addItem(
						$basket['Booking']['sales_ref'],
						$item['Extra']['name'],
						'X' . $item['Extra']['id'],
						$item['extra_unit_cost'],
						$item['quantity'],
						'Booking'
					);
					$this->FbPixel->addItem('X' . $item['Extra']['id']);
				}
			}
			if ($basket['Booking']['is_insured'] === true) {
				$this->GaEcommerce->addItem(
					$basket['Booking']['sales_ref'],
					'Insurance',
					'INSURANCE',
					$basket['Booking']['insurance_cost'],
					1,
					'Booking'
				);
				$this->FbPixel->addItem('INSURANCE');
			}
			$gaItems = $this->GaEcommerce->items();
			// Set up the Facebook Pixel tracking variables.
			$fbPixelEvent = $this->FbPixel->transaction(
				$basket['Booking']['total_cost']
			);
			$this->set(compact('fbPixelEvent', 'gaTransaction', 'gaItems'));

			$this->set(compact('basket'));
			$this->view = 'BuzzBookings.confirmation';

		} else {

			throw new ForbiddenException();

		}

		return;
	}

/**
 * Populate lookups for address fields.
 *
 * @return void
 */
	protected function _populateLookups() {
		$Booking = $this->{$this->modelClass};

		$countries = ClassRegistry::init('BuzzCustomers.Country')->translatedList();
		$usStates = $Booking->CustomerAddress->UsState->find('list');

		$this->set(compact('countries', 'usStates'));

		return;
	}

/**
 * Populate voucher details
 *
 * @param int $activityId
 * @param int $apiRef
 * @param bool $forceUpdate Pass true to make sure vouchers are fetched from the API not session
 * @return void
 */
	protected function _populateVouchers($activityId, $apiRef = null, $forceUpdate = false) {
		$Booking = $this->{$this->modelClass};

		if ($this->Session->check('BuzzBookings.vouchers') === false || $forceUpdate === true) {

			if (empty($apiRef)) {
				$activity = $Booking->Activity->findById($activityId);
				$apiRef = $activity['Activity']['api_reference'];
			}

			$data = $Booking->getVouchers($apiRef);

		} else {

			$data = $this->Session->read('BuzzBookings.vouchers');

		}

		// Build a simple array of voucher providers for convenience in the View.
		if (!empty($data)) {
			$apiVoucherProvider = $Booking->getApiVoucherProvider();
			$voucherProviders = Hash::combine(
				$data,
				'{n}.VoucherProvider.id',
				'{n}.VoucherProvider.name'
			);

			// Sort voucher providers alphabetically with the main provider first.
			asort($voucherProviders);
			if (!empty($voucherProviders[$apiVoucherProvider])) {
				$alternativeNames = $Booking->getApiAlternativeVoucherProviderNames();
				if (!empty($alternativeNames)) {
					$alternativeNames = array_merge(
						(array)$voucherProviders[$apiVoucherProvider],
						$alternativeNames
					);
					$voucherProviders = [$apiVoucherProvider => $alternativeNames] + $voucherProviders;
				} else {
					$voucherProviders = [$apiVoucherProvider => $voucherProviders[$apiVoucherProvider]] + $voucherProviders;

				}
			}

			$vouchers = [];
			unset($data[$Booking->getApiVoucherProvider()]);
			foreach ($data as $voucherProviderId => $voucherProvider) {
				$vouchers[$voucherProviderId] = Hash::combine(
					$voucherProvider,
					'VoucherProvider.Voucher.{n}.id',
					'VoucherProvider.Voucher.{n}.name'
				);
			}
		} else {
			$voucherProviders = [];
			$vouchers = [];
		}

		$this->set(compact('voucherProviders', 'vouchers'));

		return;
	}

/**
 * Generic payment gateway callback.
 *
 * @param int $basketId Basket ID
 * @return void
 */
	public function payment_callback($basketId) {
		$result = $this->Transactions->checkPayment();

		if ($result === true) {
			return $this->_completePurchase($basketId);
		} else {
			$this->Session->setFlash(
				__d('buzz_bookings', 'There was a problem processing your payment, please try again'),
				'flash_fail'
			);
			return $this->redirect([
				'action' => 'checkout'
			]);
		}
	}

/**
 * Admin index paginate
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'CustomerAddress',
				'BookingState'
			)
		);

		return $paginate;
	}

/**
 * Admin index columns
 *
 * @return array
 */
	protected function _adminIndexColumns() {
		$Booking = $this->{$this->modelClass};

		$columns = parent::_adminIndexColumns();

		// Remove created/modified columns.
		unset($columns['Booking.created']);
		unset($columns['Booking.modified']);

		$columns[$Booking->alias . '.total_cost']['type'] = 'currency';

		$newColumns = array(
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_bookings', 'Customer'),
				'type' => 'string'
			),
			'BookingState.name' => array(
				'label' => __d('buzz_bookings', 'Status'),
				'type' => 'string'
			)
		);

		return ArrayUtil::addAfter($columns, 'Booking.sales_ref', $newColumns);
	}

/**
 * Admin columns whitelist
 *
 * @return array
 */
	protected function _adminIndexColumnsWhitelist() {
		$Booking = $this->{$this->modelClass};

		$whitelist = parent::_adminIndexColumnsWhitelist();
		$whitelist[] = $Booking->alias . '.sales_ref';
		$whitelist[] = $Booking->alias . '.total_cost';
		$whitelist[] = $Booking->alias . '.completed_date';

		return $whitelist;
	}

/**
 * Filters
 *
 * @return array
 */
	protected function _adminFilterFields() {
		$filters = parent::_adminFilterFields();

		unset($filters['Booking.name']);
		unset($filters['Booking.created']);
		unset($filters['Booking.modified']);

		$newFilters = array(
			'Booking.sales_ref' => array(
				'label' => __d('buzz_bookings', 'Sales Ref'),
				'type' => 'string',
				'compare' => array('Booking.sales_ref' => "%s")
			),
			'CustomerAddress.full_name' => array(
				'label' => __d('buzz_bookings', 'Customer'),
				'type' => 'string',
				'compare' => array('CONCAT(CustomerAddress.first_name, " ", CustomerAddress.last_name) LIKE' => "%%%s%%")
			),
			'Booking.booking_state_id' => array(
				'label' => 'Status',
				'type' => 'select',
				'default' => BookingState::COMPLETE,
				'compare' => array('Booking.booking_state_id' => '%s')
			),
			'Booking.completed_date' => array(
				'label' => __d('buzz_bookings', 'Completed Date'),
				'type' => 'date',
				'compare' => array('Booking.completed_date' => '%s')
			)
		);

		return ArrayUtil::addAfter($filters, 'Booking.sales_ref', $newFilters);
	}

/**
 * Used to populate form drop down selects
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$this->set('bookingStates', $this->Booking->BookingState->find('list'));
		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);

		if ((int)$this->Auth->user('UserGroup') === 1) {
			$this->loadModel('BuzzSource.ApiLog');
			$this->set('apiCalls', $this->ApiLog->getEntries('Booking', $id));

			$bookingItemIds = Hash::extract($this->request->data['BookingItem'], '{n}.id');

			$this->set('itemApiCalls', $this->ApiLog->find('all', array(
				'conditions' => array(
					'model' => 'BookingItem',
					'foreign_id' => $bookingItemIds
				)
			)));
		}

		// We're overriding the scaffolding template as we want to customise
		// the tabs.
		$this->view = 'BuzzBookings.admin_form';

		return;
	}

	protected function _adminIndexToolbar($id = null) {
		return [];
	}

	protected function _adminFormToolbar($id = null) {
		$actions = parent::_adminFormToolbar($id);
		unset($actions['Add New']);

		$actions['Resend Email'] = array(
			'url' => array('action' => 'email_confirmation', $id),
			'icon' => 'envelope'
		);

		return $actions;
	}

/**
 * Re-sends the confirmation email to the customer and redirects to the
 * admin edit form.
 *
 * @param int $id Purchase ID
 */
	public function admin_email_confirmation($id) {
		$Booking = $this->{$this->modelClass}->alias;

		$Event = new CakeEvent('Model.Booking.completed', $this, array(
			'id' => $id
		));
		$this->getEventManager()->dispatch($Event);

		$this->Session->setFlash(
			array(
				'title' => __d('buzz_bookings', '%s confirmation sent', [InflectorExt::humanize($this->$Booking->displayName)]),
				'description' => __d('buzz_bookings', 'The confirmation email has been resent to the customer!')
			),
			'flash_success'
		);

		return $this->redirect(['action' => 'edit', $id]);
	}

}
