<?php

App::uses('BuzzBookingsAppController', 'BuzzBookings.Controller');

class ActivitiesController extends BuzzBookingsAppController {

	public $components = array(
		'BuzzAdverts.Adverts',
		'BuzzTranslate.Translation' => array(
			'fields' => array(
				'name',
				'content' => array(
					'type' => 'html'
				),
				'secondary_content' => array(
					'type' => 'html'
				)
			),
			'menu' => true,
			'meta' => true
		)
	);

	public function __construct($request = null, $response = null) {
		$this->adminActions[] = 'admin_adverts';
		// If we're using EvTinyMceImage we want to load the plugin's component.
		if (CakePlugin::loaded('EvTinyMceImage') === true) {
			$this->components['EvTinyMceImage.TinyMCE'] = [
				'model_alias' => 'Activity'
			];
		}
		parent::__construct($request, $response);
	}

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view'
		));
	}

/**
 * Activities landing page
 *
 * @return void
 */
	public function index() {
		$activities = $this->Activity->getListed(false);

		if (count($activities) === 1 && !empty($activities[0]['Activity']['id'])) {
			return $this->redirect([
				'controller' => 'activities',
				'action' => 'view',
				$activities[0]['Activity']['id']
			]);
		}

		if (Configure::check('BuzzBookings.activities_page_id') === true) {
			$this->assignPage(Configure::read('BuzzBookings.activities_page_id'));
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($this->viewVars['data']['Page']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->Page->Gallery->readForView($this->viewVars['data']['Page']['gallery_id'])
			);
		}

		$this->Meta->canonical(['action' => 'index']);
		$this->set(compact('activities'));
		$this->view = 'BuzzBookings./Activities/index';

		return;
	}

/**
 * Activity page
 *
 * @param int $id Activity ID
 * @return void
 */
	public function view($id, $urlHeader = null) {
		$data = $this->Activity->readForView($id);
		if ($urlHeader) {
			$data['Activity']['page_title'] = str_replace('-', ' ', $urlHeader);
		}

		// For BuzzGallery enabled sites set the gallery for the View.
		if (CakePlugin::loaded('BuzzGallery') === true && !empty($data['Activity']['gallery_id'])) {
			$this->set(
				'gallery',
				$this->Activity->Gallery->readForView($data['Activity']['gallery_id'])
			);
		}

		$activities = $this->Activity->getListed(false);

		$this->Meta->set($data);
		$this->Meta->canonical(['action' => 'view', $id]);
		$this->set(compact('activities', 'data'));
		$this->view = 'BuzzBookings./Activities/view';

		// Check if we're overriding the view template.
		if (!empty($data['Activity']['view_template'])) {

			$this->view = $data['Activity']['view_template'];

			// Check if we need to grab any extra data for this template.
			$template = Inflector::camelize($data['Activity']['view_template']);
			if (method_exists($this, '_get' . $template . 'Extras')) {
				$method = '_get' . $template . 'Extras';
				$this->$method();
			}
		}

		return;
	}

	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->Adverts->setupAdvertAdminFields();
		$this->Translation->adminTranslationFormFields();
		$this->view = 'admin_form';
		return;
	}

/**
 * Admin form fields
 *
 * @return array
 */
	protected function _adminFormFields() {
		$Activity = $this->{$this->modelClass};
		$fields = parent::_adminFormFields();

		$fields['Activity.name']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'The name of the activity used for the booking process.') . '</p>';

		$fields['Activity.page_title']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'This is the title that appears at the top of the activity\'s page. Leave it blank to use the Activity name instead.') . '</p>';

		$fields['Activity.content']['type'] = 'html';
		$fields['Activity.secondary_content']['type'] = 'html';

		$fields['Activity.listed']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'Listed activities will be shown on the Activities landing page and the homepage.') . '</p>';

		$fields['Activity.is_hidden_activity']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'Hidden activities are not shown on the Activities landing page or homepage and can only be booked by going directly to the activity\'s URL.') . '</p>';

		$fields['Activity.video_url']['label'] = __d('buzz_bookings', 'Video URL');

		$fields['Activity.api_reference']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'Used for displaying the correct voucher list for redemption.') . '</p>';

		if (Configure::read('BuzzBookings.slider_text') === true) {
			$fields['Activity.slider_text']['after'] = '<p class="input__help">' . __d('buzz_bookings', 'Text to be used on the homepage slider.') . '</p>';
		} else {
			unset($fields['Activity.slider_text']);
		}

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$galleryFields = array(
				'Activity.gallery_id' => array(
					'type' => 'select',
					'label' => __d('buzz_bookings', 'Gallery')
				)
			);
			$fields = ArrayUtil::addAfter($fields, 'Activity.theme_id', $galleryFields);
		}

		// Only Evoluted admin users should be able to see the view_template
		// field.
		if ($this->Auth->user('UserGroup.id') > 1) {
			unset($fields[$Activity->alias . '.view_template']);
		}

		return $fields;
	}

	/**
	 * Populate admin lookups
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$Activity = $this->{$this->modelClass};

		$this->set('themes', $Activity->Theme->find('list'));

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->set('galleries', $Activity->Gallery->find('list'));
		}

		return;
	}

	protected function _adminFormToolbar($id = null) {
		$toolbar = parent::_adminFormToolbar($id);

		if (!empty($id)) {
			$toolbar['Adverts'] = [
				'url' => ['action' => 'adverts', $id],
				'icon' => 'bars'
			];
			$toolbar['Preview'] = [
				'url' => ['action' => 'view', $id, 'admin' => false],
				'icon' => 'search',
				'target' => '_blank'
			];
		}

		return $toolbar;
	}

}
