<?php

App::uses('BuzzBookingsAppModel', 'BuzzBookings.Model');

class Activity extends BuzzBookingsAppModel {

/**
 * Behaviors
 *
 * @var array
 */
	public $actsAs = array(
		'BuzzAdverts.AdvertHost',
		'BuzzTranslate.Translatable',
		'EvCore.Tabbed' => array(
			'max_tabs' => 3
		),
		'MetaData.Meta',
		'Navigation.Navigatable',
	);

/**
 * Belongs to associations
 *
 * @var array
 */
	public $belongsTo = array(
		'Theme' => array(
			'className' => 'BuzzTheme.Theme'
		)
	);

/**
 * Has many associations
 *
 * @var array
 */
	public $hasMany = array(
		'ActivityPackage' => array(
			'className' => 'BuzzBookings.ActivityPackage'
		)
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'validate.maxLength150'
			)
		),
		'video_url' => array(
			'required' => array(
				'rule' => array('url', true),
				'message' => 'validate.video_url',
				'allowEmpty' => true
			)
		),
		'sequence' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			),
			'max' => array(
				'rule' => array('naturalNumber', true),
				'message' => 'validate.integer'
			)
		),
		'theme_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'validate.required'
			)
		)
	);

/**
 * Image slots
 *
 * @var array
 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		],
		'listing' => [
			'slots' => 1,
			'fields' => []
		],
	];

/**
 * Define the number of associated files/documents.
 *
 * @var int|array
 */
	public $documentSlots = -1;

/**
 * Define the multi-experience activity
 *
 * @var int
 */
	protected $_multiExperience = null;

/**
 * Define the multi-experience activity API reference
 *
 * @var int
 */
	protected $_multiExperienceApiRef = null;

/**
 * Constructor
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		if (Configure::check('BuzzBookings.routes.activities') === true) {
			$this->addBehaviour('Routable.Routable', Configure::read('BuzzBookings.routes.activities'));
		}

		if (Configure::check('app.primary_nav') === true) {
			$this->parentMenu = Configure::read('app.primary_nav');
		}

		// If BuzzGallery loaded associate activities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}

		// Set the multi-experience activity.
		$this->_multiExperience = Configure::read('BuzzBookings.multi_experience');

		// Set the multi-experience API reference.
		$this->_multiExperienceApiRef = Configure::read('BuzzBookings.multi_experience_api_ref');
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);

		foreach ($results as &$result) {
			// If this is a hidden activity hide the private hire option.
			if (
				!empty($result[$this->alias])
				&& array_key_exists('is_hidden_activity', $result[$this->alias])
				&& $result[$this->alias]['is_hidden_activity'] === true
			) {
				$result[$this->alias]['show_private_hire'] = false;
			}
		}

		return $results;
	}

/**
 * Returns the multi-experience activity ID
 *
 * @return int
 */
	public function getMultiExperience() {
		return $this->_multiExperience;
	}

/**
 * Returns the multi-experience activity API reference
 *
 * @return int
 */
	public function getMultiExperienceApiRef() {
		return $this->_multiExperienceApiRef;
	}

/**
 * Get all visible listed facilities with bookable packages
 *
 * @param bool $packagesOnly Pass false to show activities regardless of whether they have packages or not
 * @return array
 */
	public function getListed($packagesOnly = true) {
		$params = array(
			'contain' => array(
				'ListingImage'
			),
			'conditions' => array(
				'Activity.api_reference IS NOT NULL',
				'Activity.api_reference <>' => '',
				'Activity.listed' => true,
				'Activity.is_hidden_activity' => false,
				'Activity.is_active' => true,
				'Activity.is_removed' => false
			),
			'group' => array(
				'Activity.id'
			),
			'order' => array(
				'Activity.sequence' => 'ASC',
				'Activity.created' => 'ASC'
			)
		);
		if ($packagesOnly === true) {
			$params['joins'][] = [
				'table' => 'activity_packages',
				'alias' => 'ActivityPackage',
				'type' => 'LEFT',
				'conditions' => 'ActivityPackage.activity_id = Activity.id'
			];
			$params['conditions']['ActivityPackage.is_active'] = true;
		}
		return $this->find('all', $params);
	}

/**
 * Read for edit
 *
 * @param int $id
 * @param array $query
 * @return array
 */
	public function readForEdit($id, $query = []) {
		$query['contain']['Advert'] = array(
			'AdvertRelation',
			'AdvertTemplate',
			'Image',
			'order' => array('AdvertRelation.sequence' => 'ASC')
		);
		return parent::readForEdit($id, $query);
	}

/**
 * Read for view
 *
 * @param int $id Activity ID
 * @param array $params Additional query parameters
 * @return array
 */
	public function readForView($id, $params = []) {
		$defaults['contain']['Advert']['conditions']['Advert.is_active'] = true;
		$defaults['contain']['ActivityPackage'] = array(
			'Image',
			'conditions' => array(
				'ActivityPackage.is_active' => true,
				'ActivityPackage.is_removed' => false,
				'ActivityPackage.upsell_only' => false
			),
			'order' => array(
				'ActivityPackage.is_special_offer' => 'DESC',
				'ActivityPackage.sequence' => 'ASC',
				'ActivityPackage.created' => 'ASC'
			)
		);
		$params = Hash::merge($defaults, $params);
		$data = parent::readForView($id, $params);

		// Remove hidden adverts, these shouldn't show.
		if (!empty($data['Advert'])) {
			foreach ($data['Advert'] as $key => $advert) {
				if ($advert['is_active'] === false) {
					unset($data['Advert'][$key]);
				}
			}
		}

		if (!empty($data['Advert']) && Configure::check('BuzzAdverts.redeem_vouchers_id') === true) {
			// Link Redeem Vouchers ad to the correct activity.
			$data['Advert'] = Hash::insert(
				$data['Advert'],
				'{n}[id=' . Configure::read('BuzzAdverts.redeem_vouchers_id') . ']',
				[
					'url' => ['plugin' => 'buzz_bookings', 'controller' => 'bookings', 'action' => 'packages', $id]
				]
			);
		}

		return $data;
	}

/**
 * Checks if there is only one available listed activity for the site, if
 * there's only one activity it will return the activity ID, otherwise it
 * returns false
 *
 * @return bool|int
 */
	public function onlyBookableActivity() {
		// Get the listed activities
		$data = $this->getListed();
		return count($data) === 1 && !empty($data[0]['Activity']['id']) ? $data[0]['Activity']['id'] : false;
	}

/**
 * Returns a bookable activity and its packages
 *
 * @param int $activityId Activity ID
 * @return array|bool
 */
	public function getBookableActivity($activityId) {
		// Want to make sure all returned packages have an API reference set.
		$params = array(
			'contain' => array(
				'ActivityPackage' => array(
					'conditions' => array(
						'OR' => array(
							'ActivityPackage.api_reference <>' => null,
							'ActivityPackage.api_reference <>' => ''
						),
						// Exclude private hire packages.
						'ActivityPackage.participants >' => 0
					)
				)
			),
			'conditions' => array(
				'OR' => array(
					'Activity.api_reference <>' => null,
					'Activity.api_reference <>' => ''
				)
			)
		);
		// Get the activity and its packages.
		$data = $this->readForView($activityId, $params);
		return !empty($data) && !empty($data['ActivityPackage']) ? $data : false;
	}

/**
 * Returns a list of non-multi-experience activities.
 *
 * @param array
 */
	public function listNonMultiExperienceActivities() {
		$params = [];
		if ($this->getMultiExperience()) {
			$params['conditions'][$this->alias . '.id <>'] = $this->getMultiExperience();
		}
		return $this->find('list', $params);
	}

}
