<?php

App::uses('BuzzSourceAppModel', 'BuzzSource.Model');

class BookingApi extends BuzzSourceAppModel {

	/**
	 * Get vouchers for an activity type at a specific site
	 *
	 * @param int $siteId Location/Site ID
	 * @param int $activityType ID of activity type
	 * @return array
	 */
	public function getVouchersBySite($siteId, $activityType) {
		$result = $this->find('all', array(
			'method' => 'GetVouchersBySite',
			'conditions' => array(
				'siteId' => $siteId,
				'activityType' => $activityType
			)
		));

		return !empty($result['BookingApi']['d']) ? $this->_processVouchers($result['BookingApi']['d']) : false;
	}

	/**
	 * Get multi-experience vouchers at a specific site
	 *
	 * @param int $siteId Location/Site ID
	 * @return array
	 */
	public function getMultiActivityVouchersBySite($siteId) {
		$result = $this->find('all', array(
			'method' => 'GetMultiActivityVouchersBySite',
			'conditions' => array(
				'siteId' => $siteId
			)
		));

		return !empty($result['BookingApi']['d']) ? $this->_processVouchers($result['BookingApi']['d']) : false;
	}

	/**
	 * Get multi-experience vouchers at a specific site
	 *
	 * @param int $siteId Location/Site ID
	 * @return array
	 */
	public function getPrivateHireVouchersBySite($siteId) {
		$result = $this->find('all', array(
			'method' => 'GetPrivateHireVouchersBySite',
			'conditions' => array(
				'siteId' => $siteId
			)
		));

		return !empty($result['BookingApi']['d']) ? $this->_processVouchers($result['BookingApi']['d']) : false;
	}

	/**
	 * Process vouchers data from the API
	 *
	 * @param array $data Response from API
	 * @return array
	 */
	protected function _processVouchers($data) {
		$result = [];

		foreach ($data as $voucherProvider) {
			$vouchers = [];
			if (isset($voucherProvider['Vouchers'])) {
				foreach ($voucherProvider['Vouchers'] as $voucher) {
					$vouchers[$voucher['VoucherId']] = array(
						'id' => $voucher['VoucherId'],
						'name' => $voucher['VoucherName'],
						'description' => isset($voucher['VoucherDesription']) ? $voucher['VoucherDesription'] : '',
						'participants' => $voucher['Participants'],
						'code_type' => trim($voucher['NumberType']),
						'code_location' => trim($voucher['NumberLocation']),
						'requires_validation' => $voucher['RequiresValidation'],
						'requires_expiry_date' => $voucher['RequiresExpiryDate'] ? true : false,
						'activity_api_id' => $voucher['BookingTypeId'],
						'is_off_peak_only' => $voucher['IsOffPeakOnly'] ? true : false
					);
				}
			}

			$result[$voucherProvider['VoucherProviderId']] = array(
				'VoucherProvider' => array(
					'id' => $voucherProvider['VoucherProviderId'],
					'name' => $voucherProvider['VoucherProviderName'],
					'logo_url' => $voucherProvider['VoucherProviderLogoUrl'],
					'Voucher' => $vouchers
				)
			);

		}

		return $result;
	}

	/**
	 * Check a 'Bodyflight' voucher code
	 *
	 * @param string $code Voucher code
	 * @param string $expiryDate Voucher expiry date
	 * @return array|bool
	 */
	public function checkVoucherCode($code, $expiryDate) {
		$result = $this->find('all', array(
			'method' => 'CheckBodyFlightVoucherCode',
			'conditions' => array(
				'voucherCode' => $code,
				'expiryDate' => $expiryDate
			)
		));

		return !empty($result['BookingApi']['d']['Activities']) ? $this->_processCheckVoucherCode($result['BookingApi']['d']) : false;
	}

	/**
	 * Process the response from checkVoucherCode.
	 *
	 * @param array $data
	 * @return array
	 */
	protected function _processCheckVoucherCode(array $data) {
		$result = false;

		if (
			!empty($data['Status'])
			&& strtolower($data['Status']) === 'ok'
			&& !empty($data['Activities'])
		) {
			foreach ($data['Activities'] as $activity) {
				$activities[] = array(
					'api_activity_reference' => $activity['ActivityId'],
					'api_package_reference' => $activity['BookingTypeId'],
					'quantity' => $activity['RequiredQuantity'],
				);
			}
			$result = array(
				'Voucher' => array(
					'participants' => $data['Participants'],
					'description' => $data['VoucherDescription'],
					'Activity' => $activities
				)
			);
		}

		return $result;
	}

	/**
	 * Check a third-party voucher code
	 *
	 * @param string $code Voucher code
	 * @param string $expiryDate Voucher expiry date
	 * @return array|bool
	 */
	public function checkThirdPartyVoucherCode($voucherId, $code, $expiryDate) {
		$result = $this->find('all', array(
			'method' => 'CheckVoucherCode',
			'conditions' => array(
				'voucherId' => $voucherId,
				'voucherCode' => $code,
				'expiryDate' => $expiryDate
			)
		));

		$validates = !empty($result['BookingApi']['d']['Activities'])
			|| strtolower($result['BookingApi']['d']['Status']) !== 'error';

		return $validates === true ? $this->_processCheckVoucherCode($result['BookingApi']['d']) : false;
	}

	/**
	 * Checks that the voucher has not already been used.
	 *
	 * @param int $activityId
	 * @param int $voucherProviderId
	 * @param string $voucherCode
	 * @return bool
	 */
	public function checkVoucherAvailable($activityId, $voucherProviderId, $voucherCode) {
		$result = $this->find('all', array(
			'method' => 'IsVoucherUsed',
			'conditions' => array(
				'activityId' => $activityId,
				'vouchersupplierId' => $voucherProviderId,
				'voucherno' => $voucherCode
			)
		));

		return empty($result['BookingApi']['d']);
	}

	/**
	 * Get available dates for bookings based on current booking items.
	 *
	 * @param int $siteId
	 * @param string $fromDate
	 * @param string $toDate
	 * @param array $items Booking items
	 * @param bool $concurrent Set to false if the booking items take place in series rather than at the same time
	 * @param bool $offPeakOnly
	 * @return array
	 */
	public function getDates($siteId, $fromDate, $toDate, array $items = [], $concurrent = true, $offPeakOnly = false) {
		// We Need to build up an array of booking items and their
		// quantities to pass to the API.
		$params = [];
		$i = 0;
		foreach ($items as $bookingTypeId => $quantity) {
			++$i;
			$params['bookingtype' . $i] = $bookingTypeId;
			$params['quantity' . $i] = $quantity;
		}

		$result = $this->find('all', array(
			'method' => 'GetAvailableDates',
			'conditions' => array(
				'siteId' => $siteId,
				'fromDate' => $fromDate,
				'toDate' => $toDate,
				'concurrent' => $concurrent === true ? 'true' : 'false',
				'uniquetimes' => 'true'
			) + $params,
			'debug' => 2
		));

		return !empty($result['BookingApi']['d']) ? $this->_processDates($result['BookingApi']['d']) : false;
	}

	/**
	 * Gets bookable extras.
	 *
	 * @param string  $sessionId
	 * @param int $bookingTypeId
	 * @param int $quantity Required quantity
	 * @return array
	 */
	public function getExtras($sessionId, $bookingTypeId, $quantity) {
		$result = $this->find('all', array(
			'method' => 'GetExtras',
			'conditions' => array(
				'sessionId' => $sessionId,
				'bookingTypeId' => $bookingTypeId,
				'quantity' => $quantity
			),
			'debug' => 2
		));

		return !empty($result['BookingApi']['d']) ? $this->_processDates($result['BookingApi']['d']) : false;
	}

	/**
	 * Process dates returned by the API
	 *
	 * @param array $data
	 * @param bool $offPeakOnly
	 * @return array
	 */
	protected function _processDates($data, $offPeakOnly = false) {
		$result = [];

		// Set some date/time variables ready for use later so that we don't
		// need to keep calculating these.
		$today = date('d/m/Y');
		$twoHoursFromNow = date('H:i', strtotime('+2 hours'));

		foreach ($data as $date) {
			$times = [];

			foreach ($date['Itineraries'] as $itinerary) {
				$time = null;
				$checkIn = null;
				$peak = false;
				$activities = [];

				foreach ($itinerary as $activity) {
					$activities[] = array(
						'activity_api_id' => $activity['BookingTypeId'],
						'description' => $activity['Description'],
						'check_in' => $activity['CheckInMins'],
						'from_time' => $activity['FromTime'],
						'to_time' => $activity['ToTime'],
						'peak' => (bool)$activity['IsPeak'],
						'experience_id' => $activity['ExperienceId']
					);

					if ($time === null || $activity['FromTime'] < $time) {
						$time = $activity['FromTime'];
					}

					if ($checkIn === null || $activity['CheckInMins'] > $checkIn) {
						$checkIn = $activity['CheckInMins'];
					}

					$peak = $activity['IsPeak'] === true ? true : $peak;

				}

				// We only want to keep times 2 hours in the future from now.
				if (
					((string)$date['Date'] !== $today || $time >= $twoHoursFromNow)
					&& ($offPeakOnly === false || $peak === false)
				) {
					$times[] = array(
						'time' => $time,
						'check_in' => $checkIn,
						'activities' => $activities,
						'peak' => $peak
					);
				}

			}

			if (!empty($times)) {
				// Sort the times.
				usort($times, function ($a, $b) {
					return $a['time'] > $b['time'];
				});
				// Store in the results array.
				$result[] = array(
					'date' => self::dateToSqlFormat($date['Date']),
					'times' => $times
				);
			}

		}

		return $result;
	}

	/**
	 * Creates a temporary booking (with sale).
	 *
	 * @param int $siteId
	 * @param int $bookingTypeId
	 * @param int $experince
	 * @param int $quantity
	 * @param string $bookingDate
	 * @param int $sessionId
	 * @param string $salesOrderId
	 * @param float $unitCost
	 * @param string $discountCode
	 * @param string $ip IP address
	 * @return array
	 */
	public function createTemporaryBookingWithSale($siteId, $bookingTypeId, $experince, $quantity, $bookingDate, $sessionId, $salesOrderId, $unitCost, $discountCode = null, $ip = null) {
		$result = $this->find('all', array(
			'method' => 'CreateTemporaryBookingWithSale',
			'conditions' => array(
				'siteId' => $siteId,
				'bookingtype' => (int)$bookingTypeId,
				'experince' => (int)$experince,
				'quantity' => (int)$quantity,
				'bookingdate' => date('Y-m-d\TH:i', strtotime($bookingDate)),
				'sessionId' => $sessionId,
				'ipaddress' => $ip,
				'salesorderId' => $salesOrderId ?: 0,
				'unitcost' => $unitCost,
				'discountcode' => $discountCode
			)
		));

		return !empty($result['BookingApi']['d']) ? $this->_processTemporaryBooking($result['BookingApi']['d']) : false;
	}

	/**
	 * Creates a temporary booking (with sale).
	 *
	 * @param int $siteId
	 * @param int $bookingTypeId
	 * @param int $experinceId
	 * @param int $quantity
	 * @param string $bookingDate
	 * @param array $vouchers
	 * @param int $sessionId
	 * @param string $salesOrderId
	 * @param string $ip IP address
	 * @return array
	 */
	public function createTemporaryBookingWithVouchers($siteId, $bookingTypeId, $experinceId, $quantity, $bookingDate, array $vouchers, $sessionId, $salesOrderId, $ip = null) {
		$result = $this->find('all', array(
			'method' => 'CreateTemporaryBooking',
			'conditions' => array(
				'siteId' => $siteId,
				'bookingtype' => (int)$bookingTypeId,
				'experince' => (int)$experinceId,
				'quantity' => (int)$quantity,
				'bookingdate' => date('Y-m-d\TH:i', strtotime($bookingDate)),
				'sessionId' => $sessionId,
				'ipaddress' => $ip,
				'salesorderId' => $salesOrderId ?: 0,
				'vouchers' => json_encode($vouchers)
			)
		));

		return !empty($result['BookingApi']['d']) ? $this->_processTemporaryBooking($result['BookingApi']['d']) : false;
	}

	/**
	 * Process a temporary bookings response.
	 *
	 * @param array $data
	 * @return array|bool
	 */
	protected function _processTemporaryBooking($data) {
		$result = false;

		if (!empty($data['BookingId'])) {
			$result = array(
				'booking_ref' => $data['BookingId'],
				'sales_ref' => !empty($data['SalesOrderId']) ? $data['SalesOrderId'] : null
			);
		}

		return $result;
	}

	/**
	 * Delete a temporary booking item.
	 *
	 * @param int $bookingId
	 * @return bool
	 */
	public function deleteTempoararyBooking($bookingId) {
		$result = $this->find('all', array(
			'method' => 'DeleteTemporaryBooking',
			'conditions' => array(
				'bookingId' => $bookingId
			)
		));
		return !empty($result['BookingApi']['d']) && strtoupper($result['BookingApi']['d']) == 'OK';
	}

	/**
	 * Delete temporary booking items for the whole booking.
	 *
	 * @param int $sessionId
	 * @return bool
	 */
	public function deleteTempoararyBookings($sessionId) {
		$result = $this->find('all', array(
			'method' => 'DeleteTemporaryBookings',
			'conditions' => array(
				'sessionId' => $sessionId
			)
		));
		return !empty($result['BookingApi']['d']) && strtoupper($result['BookingApi']['d']) == 'OK';
	}

	/**
	 * Add an item to a sales order
	 *
	 * @param int $siteId
	 * @param int $salesRef
	 * @param string $code
	 * @param int $quantity
	 * @param float $price
	 * @return array|bool
	 */
	public function addItemToSalesOrder($siteId, $salesRef, $code, $quantity, $price) {
		$result = $this->find('all', array(
			'method' => 'AddItemToSalesOrder',
			'conditions' => array(
				'salesorderId' => $salesRef,
				'code' => $code,
				'quantity' => (int)$quantity,
				'siteId' => (int)$siteId,
				'eachcost' => $price
			)
		));
		return !empty($result['BookingApi']['d']) ? $this->_processAddItemToSalesOrder($result['BookingApi']['d']) : false;
	}

	/**
	 * Process add item to sales order response.
	 *
	 * @param array $data
	 * @return array|bool
	 */
	protected function _processAddItemToSalesOrder($data) {
		$result = false;

		if (!empty($data['SalesOrderItemId'])) {
			$result = array(
				'api_ref' => $data['SalesOrderItemId'],
				'sales_ref' => !empty($data['SalesOrderId']) ? $data['SalesOrderId'] : null
			);
		}

		return $result;
	}

	/**
	 * Completes a booking
	 *
	 * @param int $salesRef
	 * @param string $firstName
	 * @param string $lastName
	 * @param string $email
	 * @param string $telephone
	 * @param array $address
	 * @param float $totalPaid
	 * @param array $paymentDetails
	 * @param string $notes
	 * @return bool
	 */
	public function completeSale($salesRef, $firstName, $lastName, $email, $telephone, array $address, $totalPaid = 0, $paymentDetails = null, $notes = null) {
		$result = $this->find('all', array(
			'method' => 'CompleteSale',
			'conditions' => array(
				'salesOrderId' => $salesRef,
				'firstname' => $firstName,
				'lastname' => $lastName,
				'contacttel' => $telephone,
				'email' => $email,
				'addressline1' => $address['address_line_1'],
				'town' => $address['address_line_3'],
				'postcode' => $address['postcode'],
				'paymentamount' => $totalPaid,
				'vendortxcode' => $paymentDetails['vendorTxCode'],
				'vpstxId' => $paymentDetails['vpstxId'],
				'status' => $paymentDetails['status'],
				'txauthno' => $paymentDetails['txauthno'],
				'notes' => $notes
			)
		));
		return !empty($result['BookingApi']['d']) && strtolower($result['BookingApi']['d']) === 'ok';
	}

	/**
	 * Completes a voucher booking
	 *
	 * @param int $sessionId
	 * @param int $salesRef
	 * @param string $firstName
	 * @param string $lastName
	 * @param string $email
	 * @param string $telephone
	 * @param string $notes
	 * @return bool
	 */
	public function completeTemporaryBooking($sessionId, $salesRef, $firstName, $lastName, $email, $telephone, $notes = null) {
		$result = $this->find('all', array(
			'method' => 'CompleteTemporaryBookings',
			'conditions' => array(
				'sessionId' => $sessionId,
				'salesOrderId' => $salesRef,
				'contactname' => $firstName . ' ' . $lastName,
				'contacttel' => $telephone,
				'email' => $email,
				'notes' => $notes
			)
		));
		return !empty($result['BookingApi']['d']) && strtolower($result['BookingApi']['d']) === 'ok';
	}

	/**
	 * Abandon a booking
	 *
	 * @param string $email
	 * @param string $salesRef
	 * @param string $lang ISO 639-1 language code
	 * @param string $notes
	 * @return bool
	 */
	public function abandonSale($email, $salesRef, $lang, $notes = null) {
		$result = $this->find('all', array(
			'method' => 'AbandonSale',
			'conditions' => array(
				'email' => $email,
				'salesorderid' => $salesRef,
				'lang' => $lang,
				'notes' => $notes
			)
		));
		return !empty($result['BookingApi']['d']) && strtolower($result['BookingApi']['d']) === 'ok';
	}

}
