<?php

App::uses('BuzzBookingsAppModel', 'BuzzBookings.Model');
App::uses('BookingState', 'BuzzBookings.Model');

class Booking extends BuzzBookingsAppModel {

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzPurchase.Payable'
	);

	/**
	 * Default order
	 *
	 * @var array|string
	 */
	public $order = 'Booking.id DESC';

	/**
	 * @var array Belongs to associations
	 */
	public $belongsTo = array(
		'Activity' => array(
			'className' => 'BuzzBookings.Activity'
		),
		'BookingState' => array(
			'className' => 'BuzzBookings.BookingState'
		),
		'CustomerAddress' => array(
			'className' => 'BuzzCustomers.CustomerAddress'
		)
	);

	/**
	 * @var array Has many associations
	 */
	public $hasMany = array(
		'BookingExtra' => array(
			'className' => 'BuzzBookings.BookingExtra'
		),
		'BookingItem' => array(
			'className' => 'BuzzBookings.BookingItem'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'email' => array(
			'email' => array(
				'rule' => 'email',
				'message' => 'Not a valid email address'
			),
			'maxLength' => array(
				'rule' => array('maxLength', 254),
				'message' => 'No more than 254 characters long'
			)
		)
	);

	/**
	 * Read for edit
	 *
	 * @param int $id Booking ID
	 * @param array $params
	 * @return array
	 */
	public function readForEdit($id, $params = []) {
		$defaults = array(
			'contain' => array(
				'Activity',
				'BookingExtra' => array(
					'Extra'
				),
				'BookingItem' => array(
					'ActivityPackage',
					'BookingItemPlace',
					'BookingItemVoucher'
				),
				'CustomerAddress' => array(
					'Country'
				)
			)
		);
		$params = Hash::merge($defaults, $params);
		$results = parent::readForEdit($id, $params);

		// Set the participants against the booking items.
		if (!empty($results['BookingItem'])) {
			foreach ($results['BookingItem'] as &$item) {
				if (!empty($item['BookingItemVoucher'])) {
					$item['participants'] = $item['BookingItemVoucher']['participants'];
				} else {
					$item['participants'] = $item['ActivityPackage']['participants'];
				}
			}
		}

		return $results;
	}

	/**
	 * Clears the basket of existing items
	 *
	 * @param bool $validate Pass as true to check it is safe to clear the basket
	 * @param bool $keepVouchers Pass as true to preserve vouchers in the basket
	 * @return bool
	 */
	public function clearBasket($validate = true, $keepVouchers = false) {
		$bookingId = $this->getBasketId();

		$this->id = $bookingId;
		if ((int)$this->field('booking_state_id') !== BookingState::UNPAID) {
			// This booking has been completed so cannot be cleared!
			return false;
		}

		$params = array(
			'BookingItem.booking_id' => $bookingId
		);

		if ($keepVouchers === true) {
			$voucherBookingItems = $this->BookingItem->BookingItemVoucher->find(
				'list',
				[
					'fields' => [
						'BookingItemVoucher.id',
						'BookingItemVoucher.booking_item_id'
					],
					'contain' => [
						'BookingItem'
					],
					'conditions' => [
						'BookingItem.booking_id' => $bookingId
					]
				]
			);
			if (!empty($voucherBookingItems)) {
				$params['NOT']['BookingItem.id'] = $voucherBookingItems;
			}
		}

		// Reset booking time.
		$this->save([
			'Booking' => [
				'id' => $bookingId,
				'check_in_time' => null,
				'booking_date' => null
			]
		]);

		return $this->BookingItem->deleteAll($params, true);
	}

	/**
	 * Get the current basket
	 *
	 * @param int $bookingId
	 * @return array
	 */
	public function getBasket($bookingId = null) {
		$bookingId = $bookingId ?: $this->getBasketId();
		if (empty($bookingId)) {
			return [];
		}
		$params = array(
			'conditions' => array(
				'Booking.booking_state_id' => BookingState::UNPAID
			)
		);
		$data = $this->readForView($bookingId, $params);

		if (!empty($data)) {
			// Check if the basket has any vouchers.
			$data['Booking']['has_vouchers'] = (bool)count(
				Hash::extract($data, 'BookingItem.{n}.BookingItemVoucher.id')
			);

			// Sum the total items in the basket.
			$data['Booking']['total_items'] = array_sum(Hash::extract($data, 'BookingItem.{n}.quantity'));
			$data['Booking']['total_items'] += array_sum(Hash::extract($data, 'BookingExtra.{n}.quantity'));
		}

		return $data;
	}

	/**
	 * Get a completed booking to display to the customer.
	 *
	 * @param int $id Purchase ID
	 * @return array
	 */
	public function getPurchase($id) {
		$params = array(
			'conditions' => array(
				'Booking.booking_state_id' => array(
					BookingState::COMPLETE,
					BookingState::API_FAILED
				)
			)
		);
		return $this->readForView($id, $params);
	}

	/**
	 * Update the current basket total.
	 *
	 * @return bool
	 */
	protected function _updateBasketTotal() {
		$basketId = $this->getBasketId();

		$basket = $this->getBasket();

		// Calculate the total cost of the booking items.
		$total = array_sum(Hash::extract($basket, 'BookingItem.{n}.total'));

		// Add the cost of booking extras to the total.
		$total += array_sum(Hash::extract($basket, 'BookingExtra.{n}.total'));

		// Save the update total.
		$data = array(
			'id' => $basketId,
			'total_cost' => $total
		);

		return $this->save($data) !== false;
	}

	/**
	 * Creates a new basket (or recreates an existing basket).
	 *
	 * @param int $activityId ID of activity being booked
	 * @param array $bookingItems Packages being booked
	 * @param bool $isPrivateHire
	 * @return bool
	 */
	public function createBasket($activityId, $bookingItems = [], $isPrivateHire = false) {
		$bookingId = $this->getBasketId();

		if (!empty($bookingId)) {
			// We're editing an existing basket so need to clear it first. If
			// the basket has been completed we need to create a new one. If
			// the activity hasn't changed we want to keep the vouchers.
			$booking = $this->findById($bookingId);
			if (
				$this->clearBasket(
					true,
					(int)$booking['Booking']['activity_id'] === (int)$activityId
				) === false
			) {
				$bookingId = null;
			}
		}

		// Make sure we don't save items with zero quantity to the database.
		foreach ($bookingItems as $key => $val) {
			if ((int)$val['quantity'] === 0) {
				unset($bookingItems[$key]);
			}
		}

		// Get all the packages being booked from the database so that we can
		// create the booking item places.
		$packages = $this->Activity->ActivityPackage->find(
			'all',
			array(
				'conditions' => array(
					'ActivityPackage.id' => Hash::extract($bookingItems, '{n}.activity_package_id')
				)
			)
		);
		$packages = Hash::combine($packages, '{n}.ActivityPackage.id', '{n}.ActivityPackage');
		foreach ($bookingItems as &$bookingItem) {
			$bookingItem['BookingItemPlace'] = $this->_generateBookingItemPlace($packages, $bookingItem);
		}

		$data = array(
			'Booking' => array(
				'id' => $bookingId,
				'activity_id' => $activityId,
				'total_cost' => 0,
				'booking_state_id' => BookingState::UNPAID,
				'is_private_hire' => $isPrivateHire
			),
			'BookingItem' => $bookingItems
		);

		$result = $this->saveAssociated($data, ['deep' => true]) !== false;

		// Update the session.
		if ($result === true) {
			$this->setBasketId($this->id);
		} else {
			$this->clearBasketId();
		}

		return $result;
	}

	/**
	 * Generates a booking item place to attach to a booking item
	 *
	 * @param array $packages
	 * @param array $bookingItem
	 * @return array
	 */
	protected function _generateBookingItemPlace(array $packages, array $bookingItem) {
		$package = $packages[$bookingItem['activity_package_id']];
		$data = [];

		if ((int)$package['activity_id'] === $this->Activity->getMultiExperience()) {
			$apiRefs = explode(',', $package['api_reference']);
			$apiRefs = array_map('trim', $apiRefs);

			foreach ($apiRefs as $apiRef) {
				preg_match('|^(\d+)\[([\d\.]+)\]$|', $apiRef, $matches);
				$data[] = array(
					'api_reference' => $matches[1],
					'quantity' => $bookingItem['quantity'],
					'name' => $package['name'],
					'item_unit_cost' => $matches[2]
				);
			}

		} else {
			$data[] = array(
				'api_reference' => $package['api_reference'],
				'quantity' => $bookingItem['quantity'],
				'name' => $package['name'],
				'item_unit_cost' => null
			);

		}

		return $data;
	}

	/**
	 * Adds a voucher to the booking.
	 *
	 * @param int $activityId Activity ID
	 * @param array $voucher BookingItemVoucher data to save
	 * @return bool|array Returns false if the voucher fails validation or the voucher data
	 */
	public function addVoucher($activityId, array $voucher) {
		$bookingId = $this->getBasketId();

		// Validate the voucher.
		$validatedVoucher = $this->_validateVoucher($activityId, $voucher);
		if ($validatedVoucher === false) {
			return false;
		}

		// Build booking item places for the voucher.
		$bookingItemPlaces = [];
		foreach ($validatedVoucher['Voucher']['Activity'] as $activity) {
			$bookingItemPlaces[] = array(
				'api_reference' => $activity['api_package_reference'],
				'quantity' => 1,
				'name' => $validatedVoucher['Voucher']['description'],
				'item_unit_cost' => 0
			);
		}

		// Save the voucher to the booking.
		$data = array(
			'Booking' => array(
				'id' => $bookingId,
				'activity_id' => $activityId,
				'total_cost' => 0,
				'booking_state_id' => BookingState::UNPAID,
				'is_private_hire' => (int)$activityId === -1
			),
			'BookingItem' => array(
				array(
					'id' => null,
					'activity_package_id' => null,
					'item_unit_cost' => 0,
					'quantity' => 1,
					'total' => 0,
					'BookingItemPlace' => $bookingItemPlaces,
					'BookingItemVoucher' => $voucher + array(
						'voucher_provider' => CakeSession::read('BuzzBookings.vouchers.' . $voucher['voucher_provider_id'] . '.VoucherProvider.name'),
						'name' => $validatedVoucher['Voucher']['description'],
						'participants' => $validatedVoucher['Voucher']['participants']
					)
				)
			)
		);
		$result = $this->saveAssociated($data, ['deep' => true]) !== false;

		// Set the basket ID if not already set.
		if ($bookingId === null && $result !== false) {
			$this->setBasketId($this->id);
		}

		if ($result !== false) {
			$data['BookingItem'][0]['BookingItemVoucher']['booking_item_id'] = $this->BookingItem->id;
		}

		return $result !== false ? $data['BookingItem'][0]['BookingItemVoucher'] : false;
	}

	/**
	 * Removes all vouchers from the booking.
	 *
	 * @return bool Returns true on success
	 */
	public function removeVouchers() {
		$bookingItems = $this->BookingItem->find(
			'all',
			[
				'contain' => [
					'BookingItemVoucher'
				],
				'conditions' => [
					'BookingItem.booking_id' => $this->getBasketId(),
					'BookingItemVoucher.id <>' => null
				]
			]
		);
		return $this->BookingItem->deleteAll(
			[
				'BookingItem.id' => Hash::extract($bookingItems, '{n}.BookingItem.id'),
				'BookingItem.booking_id' => $this->getBasketId()
			]
		);
	}

	/**
	 * Removes a voucher from the booking.
	 *
	 * @param int $bookingItemId Booking item ID
	 * @return bool Returns true on success
	 */
	public function removeVoucher($bookingItemId) {
		return $this->BookingItem->deleteAll(
			[
				'BookingItem.id' => $bookingItemId,
				'BookingItem.booking_id' => $this->getBasketId()
			]
		);
	}

	/**
	 * Check that a voucher is valid
	 *
	 * @param int $activityId
	 * @param array $data
	 * @return array
	 */
	protected function _validateVoucher($activityId, array $data) {
		$result = false;

		// First check that the data validates against app validation rules.
		$this->BookingItem->BookingItemVoucher->set($data);
		if ($this->BookingItem->BookingItemVoucher->validates() === false) {
			return false;
		}

		if ((int)$data['voucher_provider_id'] === $this->getApiVoucherProvider()) {
			$result = $this->_checkVoucherCode(
				$activityId,
				$data['voucher_code'],
				$data['expiry_date']
			);
		} else {
			$result = $this->_checkThirdPartyVoucherCode(
				$activityId,
				$data['voucher_provider_id'],
				$data['api_voucher_id'],
				$data['voucher_code'],
				$data['expiry_date']
			);
		}

		if ($result !== false) {
			// We need to make sure that this voucher hasn't already been added
			// to this booking.
			$isUnique = $this->_checkUniqueVoucher(
				(int)$data['voucher_provider_id'],
				$data['voucher_code'],
				$data['expiry_date']
			);
			if ($isUnique === false) {
				$result = false;
			}
		}

		if ($result !== false) {
			// If the voucher is valid then we need to do one final check that
			// it hasn't already been used before accepting it.
			$this->Activity->id = $activityId;
			$apiRef = $this->Activity->field('api_reference');
			$isAvailable = $this->_checkVoucherAvailable(
				$apiRef,
				(int)$data['voucher_provider_id'],
				$data['voucher_code']
			);
			if ($isAvailable === false) {
				return false;
			}
		}

		return $result;
	}

	/**
	 * Check a first party voucher code.
	 *
	 * @param int $activityId
	 * @param string $voucherCode
	 * @param string $expiryDate Expiry date in SQL format
	 * @return array|bool Returns voucher data or false if not valid
	 */
	protected function _checkVoucherCode($activityId, $voucherCode, $expiryDate) {
		$result = $this->_checkVoucherCodeApiCall(
			$voucherCode,
			$expiryDate
		);

		if ($result === false) {
			return false;
		}

		if (
			(int)$activityId !== $this->Activity->getMultiExperience()
			&& (int)$activityId !== -1
		) {
			// Make sure the voucher is for the current activity (and not
			// multi-experience).
			$this->Activity->id = $activityId;
			$apiRef = (int)$this->Activity->field('api_reference');
			if (
				count($result['Voucher']['Activity']) > 1
				|| (int)$result['Voucher']['Activity'][0]['api_activity_reference'] !== $apiRef
			) {
				return false;
			}
		}

		return $result;
	}

	/**
	 * Check a third party voucher code.
	 *
	 * Checks against the stored session data (retrieved from the API) and against
	 * the API.
	 *
	 * @param int $activityId
	 * @param int $voucherProviderId
	 * @param int $apiVoucherId
	 * @param string $voucherCode
	 * @param string $expiryDate Expiry date in SQL format
	 * @return array|bool Returns voucher data or false if not valid
	 */
	protected function _checkThirdPartyVoucherCode($activityId, $voucherProviderId, $apiVoucherId, $voucherCode, $expiryDate) {
		// Check if the voucher being checked exists in our session stored
		// vouchers data.
		$voucher = $this->_getVoucherData($activityId, $voucherProviderId, $apiVoucherId);

		// Fail validation if the voucher data cannot be found. If we have data
		// and it doesn't require validation make sure it hasn't expired before
		// returning the voucher details.
		if (empty($voucher)) {
			return false;
		} elseif ($voucher['requires_validation'] === false && $expiryDate >= date('Y-m-d')) {
			return [
				'Voucher' => [
					'participants' => $voucher['participants'],
					'description' => $voucher['description'],
					'Activity' => [
						[
							'api_package_reference' => $voucher['activity_api_id'],
							'quantity' => 1
						]
					]
				]
			];
		}

		return $this->_checkThirdPartyVoucherCodeApiCall(
			$apiVoucherId,
			$voucherCode,
			$expiryDate
		);
	}

	/**
	 * API call to check vouchers (to enable unit test mocking).
	 *
	 * @param string $voucherCode
	 * @param string $expiryDate Expiry date in SQL format
	 * @return array
	 */
	protected function _checkVoucherCodeApiCall($voucherCode, $expiryDate) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->checkVoucherCode(
			$voucherCode,
			$expiryDate
		);
	}

	/**
	 * API call to check third-party vouchers (to enable unit test mocking).
	 *
	 * @param int $voucherId
	 * @param string $voucherCode
	 * @param string $expiryDate Expiry date in SQL format
	 * @return array
	 */
	protected function _checkThirdPartyVoucherCodeApiCall($voucherId, $voucherCode, $expiryDate) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->checkThirdPartyVoucherCode(
			$voucherId,
			$voucherCode,
			$expiryDate
		);
	}

	/**
	 * API call (to enable unit test mocking).
	 *
	 * @param int $activityId
	 * @param int $voucherProviderId
	 * @param string $voucherCode
	 * @return bool
	 */
	protected function _checkVoucherAvailable($activityId, $voucherProviderId, $voucherCode) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->checkVoucherAvailable(
			$activityId,
			$voucherProviderId,
			$voucherCode
		);
	}

	/**
	 * Check that a voucher is unique to a booking.
	 *
	 * @param int $voucherProviderId
	 * @param string $voucherCode
	 * @param string $expiryDate
	 * @return bool
	 */
	protected function _checkUniqueVoucher($voucherProviderId, $voucherCode, $expiryDate) {
		$result = $this->BookingItem->BookingItemVoucher->find(
			'count',
			[
				'contain' => [
					'BookingItem'
				],
				'conditions' => [
					'BookingItem.booking_id' => $this->getBasketId(),
					'BookingItemVoucher.voucher_provider_id' => (int)$voucherProviderId,
					'BookingItemVoucher.voucher_code' => $voucherCode,
					'BookingItemVoucher.expiry_date' => $expiryDate
				]
			]
		);
		// There shouldn't be any existing vouchers matching the supplied
		// details in the database already.
		return $result < 1;
	}

	/**
	 * Returns matching voucher data retrieved from the API for a given voucher.
	 *
	 * @return array|bool Voucher data or false if not found
	 */
	protected function _getVoucherData($activityId, $voucherProviderId, $apiVoucherId) {
		return CakeSession::read('BuzzBookings.vouchers.' . $voucherProviderId . '.VoucherProvider.Voucher.' . $apiVoucherId);
	}

	/**
	 * Returns the API's voucher provider.
	 *
	 * @return int
	 */
	public function getApiVoucherProvider() {
		return (int)Configure::read('BuzzBookings.voucher_provider_id');
	}

	/**
	 * Returns voucher data.
	 *
	 * @param int $activityId API Activity reference code
	 * @return array
	 */
	public function getVouchers($activityId) {
		// Get the vouchers (get private hire vouchers if the api reference is
		// negative).
		if ($activityId < 0) {
			$data = ClassRegistry::init('BuzzBookings.BookingApi')->getPrivateHireVouchersBySite(
				Configure::read('api.site_id')
			);
		} elseif ($activityId === $this->Activity->getMultiExperienceApiRef()) {
			$data = ClassRegistry::init('BuzzBookings.BookingApi')->getMultiActivityVouchersBySite(
				Configure::read('api.site_id')
			);
		} else {
			$data = ClassRegistry::init('BuzzBookings.BookingApi')->getVouchersBySite(
				Configure::read('api.site_id'),
				$activityId
			);
		}
		// Save the voucher data to the session for checking against when
		// adding a voucher to the booking.
		CakeSession::write('BuzzBookings.vouchers', $data);
		// Return the voucher data.
		return $data;
	}

	/**
	 * Get the available dates for the booking.
	 *
	 * @param string $fromDate From date
	 * @param string $toDate To date which the method will modify according to the booking
	 * @return array
	 */
	public function getAvailableDates($fromDate, $toDate) {
		// Get the current basket.
		$basket = $this->getBasket();

		// Modify the toDate if any vouchers expire before the toDate passed
		// to the method.
		$expiryDates = Hash::extract($basket['BookingItem'], '{n}.BookingItemVoucher.{n}.expiry_date');
		if (!empty($expiryDates)) {
			sort($expiryDates);
			$toDate = $expiryDates[0] < $toDate ? $expiryDates[0] : $toDate;
		}

		// Get the times and dates available for the booking.
		$data = ClassRegistry::init('BuzzBookings.BookingApi')->getDates(
			Configure::read('api.site_id'),
			$fromDate,
			$toDate,
			$this->_getBookingActivitiesList($basket['Booking']['id']),
			(int)$basket['Booking']['activity_id'] !== $this->Activity->getMultiExperience()
		);

		return $data;
	}

	/**
	 * Returns an array of unavailables dates from the specified date range.
	 *
	 * @param array $dates
	 * @param string $fromDate
	 * @param string $toDate
	 * @return array
	 */
	public function calculateUnavailableDates(array $dates, $fromDate, $toDate) {
		$availableDates = [];
		$unavailableDates = [];

		foreach ($dates as $date) {
			$availableDates[$date['date']] = true;
		}

		$fromDate = strtotime($fromDate);
		$toDate = strtotime($toDate);
		for ($date = $fromDate; $date <= $toDate; $date += 86400) {
			$dateKey = date('Y-m-d', $date);
			if (isset($availableDates[$dateKey]) === false) {
				$unavailableDates[$dateKey] = true;
			}
		}

		return $unavailableDates;
	}

	/**
	 * Returns a list of API activities and quantities.
	 *
	 * @param int $bookingId Booking/Basket ID
	 * @return array
	 */
	protected function _getBookingActivitiesList($bookingId) {
		$result = $this->BookingItem->BookingItemPlace->find(
			'all',
			array(
				'joins' => array(
					array(
						'table' => 'booking_items',
						'alias' => 'BookingItem',
						'type' => 'INNER',
						'conditions' => array(
							'BookingItem.id = BookingItemPlace.booking_item_id',
							'BookingItem.booking_id' => $bookingId
						)
					)
				)
			)
		);

		$data = [];

		$items = Hash::combine(
			$result,
			'{n}.BookingItemPlace.booking_item_id',
			'{n}.BookingItemPlace.quantity',
			'{n}.BookingItemPlace.api_reference'
		);
		foreach ($items as $apiRef => $item) {
			$data[$apiRef] = array_sum($item);
		}

		return $data;
	}

	/**
	 * @param array $basket
	 * @param string $date
	 * @param array $booking
	 * @return bool
	 */
	public function addTemporaryBooking(array $basket, $date, array $booking) {
		if (empty($booking)) {
			return false;
		}

		// Generate the booking date/time
		$bookingDate = $date . ' ' . $booking['time'] . ':00';

		// Calculate the check-in time.
		$checkInTime = strtotime($booking['time']) - $booking['check_in'] * 60;

		$data = array(
			'Booking' => array(
				'id' => $this->getBasketId(),
				'booking_date' => $bookingDate,
				'check_in_time' => date('H:i', $checkInTime)
			)
		);

		$activitiesInBasket = $this->_getBasketActivities();

		foreach ($booking['activities'] as $activity) {

			if (!empty($activitiesInBasket[$activity['activity_api_id']])) {

				// Generate the booking date/time.
				$bookingDate = $date . ' ' . $activity['from_time'] . ':00';
				// Calculate the check-in time for the activity.
				$checkInTime = strtotime($activity['from_time']) - $activity['check_in'] * 60;
				$checkInTime = date('H:i', $checkInTime);

				foreach ($activitiesInBasket[$activity['activity_api_id']] as $bookingItemPlace) {

					$price = 0;
					if (!empty($bookingItemPlace['ActivityPackage']['price'])) {
						$price = $bookingItemPlace['ActivityPackage']['price'];
					} elseif (!empty($bookingItemPlace['ActivityPackage']['peak_price'])) {
						$price = $booking['peak'] === true ? $bookingItemPlace['ActivityPackage']['peak_price'] : $bookingItemPlace['ActivityPackage']['off_peak_price'];
					}

					$data['BookingItem'][$bookingItemPlace['BookingItem']['id']]['id'] = $bookingItemPlace['BookingItem']['id'];
					$data['BookingItem'][$bookingItemPlace['BookingItem']['id']]['item_unit_cost'] = $price;
					$data['BookingItem'][$bookingItemPlace['BookingItem']['id']]['total'] = $price * $bookingItemPlace['BookingItem']['quantity'];
					$data['BookingItem'][$bookingItemPlace['BookingItem']['id']]['BookingItemPlace'][] = array(
						'id' => $bookingItemPlace['BookingItemPlace']['id'],
						'name' => $activity['description'],
						'date_time' => $bookingDate,
						'check_in_time' => $checkInTime,
						'api_experience' => $activity['experience_id']
					);

				}

			}

		}

		if ($this->saveAssociated($data, ['deep' => true]) !== false) {

			// We need to update the basket total now as we have updated the
			// booking item prices based on the times booked.
			$this->_updateBasketTotal();

			return $this->_createTemporaryBooking();
		}

		return false;
	}

	/**
	 * Create a temporary booking.
	 *
	 * @param int $type 0 for all activities, -1 for upsells and 1 for main activities
	 * @return bool
	 */
	protected function _createTemporaryBooking($type = 1) {
		$id = $this->getBasketId();

		$activities = $this->_getBasketActivities(false, $type);

		$salesRef = null;

		foreach ($activities as $activity) {

			$quantity = array_sum(Hash::extract($activity, '{n}.BookingItem.quantity'));

			// Create the temporary booking
			if (!empty($activity[0]['BookingItemVoucher']['id'])) {
				// Voucher booking

				$vouchers = [];
				foreach ($activity as $voucher) {
					$vouchers[] = [
						'SupplierId' => $voucher['BookingItemVoucher']['voucher_provider_id'],
						'VoucherNo' => $voucher['BookingItemVoucher']['voucher_code'],
						'ExpiryDate' => $voucher['BookingItemVoucher']['expiry_date']
					];
				}

				$result = ClassRegistry::init('BuzzBookings.BookingApi')->createTemporaryBookingWithVouchers(
					Configure::read('api.site_id'),
					$activity[0]['BookingItemPlace']['api_reference'],
					$activity[0]['BookingItemPlace']['api_experience'],
					$quantity,
					$activity[0]['BookingItemPlace']['date_time'],
					$vouchers,
					$id,
					$salesRef
				);

			} else {
				// Sale

				$result = ClassRegistry::init('BuzzBookings.BookingApi')->createTemporaryBookingWithSale(
					Configure::read('api.site_id'),
					$activity[0]['BookingItemPlace']['api_reference'],
					$activity[0]['BookingItemPlace']['api_experience'],
					$quantity,
					$activity[0]['BookingItemPlace']['date_time'],
					$id,
					$salesRef,
					$this->_calculateAveragePrice($activity),
					// We pass the first activities discount code, making the assumption
					// that these have been mixed.
					$activity[0]['ActivityPackage']['api_discount_code']
				);
			}

			if ($result !== false) {

				$salesRef = $result['sales_ref'];

			} else {

				return false;

			}

		}

		if (!empty($salesRef)) {
			$data = array(
				'id' => $id,
				'sales_ref' => $salesRef
			);
			return $this->save($data) !== false;
		}

		return false;
	}

	/**
	 * Remove a temporary booking from the basket.
	 *
	 * @param int $apiRef
	 * @return bool
	 */
	public function deleteTemporaryBooking($apiRef) {
		$result = ClassRegistry::init('BuzzBookings.BookingApi')->deleteTempoararyBooking($apiRef);
		if ($result === false) {
			throw new InternalErrorException();
		}

		return true;
	}

	/**
	 * Remove temporary bookings from the basket.
	 *
	 * @return bool
	 */
	public function clearTemporaryBookings() {
		$id = $this->getBasketId();

		if (!empty($id)) {

			$result = ClassRegistry::init('BuzzBookings.BookingApi')->deleteTempoararyBookings($id);
			if ($result === false) {
				throw new InternalErrorException();
			}

			return $this->save(
				array(
					'id' => $id,
					'sales_ref' => null,
					'booking_date' => null,
					'check_in_time' => null
				),
				false
			) !== false;

		}

		return false;
	}

	/**
	 * Returns the average price of booking items.
	 *
	 * @param array $bookingItems
	 * @return float
	 */
	protected function _calculateAveragePrice(array $bookingItems) {
		$total = 0;

		foreach ($bookingItems as $item) {
			// If the booking item place has been given a price (for
			// multi-experiences) then we add that to the total; otherwise we
			// just use the price of the booking item.
			$total += $item['BookingItemPlace']['item_unit_cost'] ?: $item['BookingItem']['item_unit_cost'];
		}

		$average = $total > 0 ? $total / count($bookingItems) : 0;

		return round($average, 2);
	}

	/**
	 * Returns activities in the basket.
	 *
	 * @param bool $combineVouchersAndSales
	 * @param int $type 0 for all activities, -1 for upsells and 1 for main activities
	 * @return array
	 */
	protected function _getBasketActivities($combineVouchersAndSales = true, $type = 1) {
		$id = $this->getBasketId();

		$params = array(
			'conditions' => array(
				'BookingItem.booking_id' => $id
			),
			'contain' => array(
				'ActivityPackage',
				'BookingItemPlace',
				'BookingItemVoucher'
			)
		);

		if (!empty($type)) {
			$params['conditions']['BookingItem.is_upsell'] = $type < 0;
		}

		$items = $this->BookingItem->find('all', $params);

		$data = [];

		foreach ($items as $item) {
			foreach ($item['BookingItemPlace'] as $place) {

				$bookingItemPlace = $item;
				unset($bookingItemPlace['BookingItemPlace']);
				$bookingItemPlace['BookingItemPlace'] = $place;

				// Make sure we've got an API reference for the booking item
				// place.
				if (
					$place['api_reference'] === null
					&& !empty($item['ActivityPackage']['api_reference'])
				) {
					$bookingItemPlace['BookingItemPlace']['api_reference'] = $item['ActivityPackage']['api_reference'];
				}

				// Build the key for the $data array. If we're not combining
				// the vouchers and sales for an activity then we add a 'V' or
				// 'S' prefix respectivily to keep them separate. This is
				// primarily used when creating the temporary bookings as these
				// need separate calls to the API.
				$key = $bookingItemPlace['BookingItemPlace']['api_reference'];
				if ($combineVouchersAndSales === false) {
					$key = (!empty($item['BookingItemVoucher']['id']) ? 'V' : 'S') . $key;
				}

				$data[$key][] = $bookingItemPlace;

			}
		}

		return $data;
	}

	/**
	 * Get the total number of participants on the booking.
	 *
	 * @return int
	 */
	public function getParticipants() {
		$results = $this->BookingItem->find(
			'all',
			[
				'contain' => ['ActivityPackage', 'BookingItemVoucher'],
				'conditions' => [
					'BookingItem.booking_id' => $this->getBasketId(),
					'BookingItem.is_upsell' => false
				]
			]
		);

		// Need to sum the total participants from both CMS defined activity
		// packages and API data for vouchers.
		$participants = 0;
		foreach ($results as $result) {
			if (!empty($result['ActivityPackage']['participants'])) {
				$participants += $result['ActivityPackage']['participants'] * $result['BookingItem']['quantity'];
			} elseif (!empty($result['BookingItemVoucher']['participants'])) {
				$participants += $result['BookingItemVoucher']['participants'] * $result['BookingItem']['quantity'];
			}
		}

		return $participants;
	}

	/**
	 * Returns non-bookable extras.
	 *
	 * @param array $basket Current basket
	 * @param int $limit Maximum number of extras to return
	 * @return array
	 */
	public function getExtras(array $basket, $limit = 3) {
		$packages = Hash::extract($basket['BookingItem'], '{n}.activity_package_id');
		$result = $this->BookingExtra->Extra->getAvailableExtras(
			$basket['Booking']['activity_id'],
			$packages,
			$limit
		);
		return $result;
	}

	/**
	 * Returns bookable upsells.
	 *
	 * @param array $basket Current basket
	 * @param int $limit Maximum number of extras to return
	 * @return array
	 */
	public function getUpsells(array $basket, $limit = 3) {
		$packages = Hash::extract($basket['BookingItem'], '{n}.activity_package_id');
		$results = $this->Activity->ActivityPackage->getUpsellPackages(
			$basket['Booking']['activity_id'],
			$packages
		);

		$data = [];
		$count = 0;
		foreach ($results as $result) {

			if ($count < $limit) {

				$booking = ClassRegistry::init('BuzzBookings.BookingApi')->getExtras(
					$basket['Booking']['id'],
					$result['ActivityPackage']['api_reference'],
					$this->getParticipants()
				);

				if ($booking[0]['times'][0]) {

					$result['ActivityPackage']['price'];

					$data[] = $result + [
						'Booking' => $booking[0]['times'][0]
					];

					$count++;

				}

			}

		}

		return $data;
	}

	/**
	 * Add all upsells to the basket.
	 *
	 * @param array $bookingExtras Non-bookable extras being added to basket
	 * @param array $bookingItems Items being booked
	 * @param array $bookableUpsells Available bookable upsells (for lookup)
	 * @return bool
	 */
	public function addUpsells(array $bookingExtras, array $bookingItems, $bookableUpsells = []) {
		$bookingId = $this->getBasketId();

		// Add upsells
		$result = $this->_addExtras($bookingId, $bookingExtras);
		if ($result === true) {
			// Add booking items
			$result = $this->_addBookableUpsells($bookingId, $bookingItems, $bookableUpsells);
		}

		// Update the basket total.
		$this->_updateBasketTotal();

		return $result;
	}

	/**
	 * Clears extras from the basket.
	 *
	 * @return bool
	 */
	public function clearExtras() {
		$bookingId = $this->getBasketId();

		$this->id = $bookingId;
		if ((int)$this->field('booking_state_id') !== BookingState::UNPAID) {
			// This booking has been completed so cannot be cleared!
			return false;
		}

		// Delete the non-bookable extras first.
		$params = array(
			'BookingExtra.booking_id' => $bookingId
		);
		if ($this->BookingExtra->deleteAll($params, true) === false) {
			throw new InternalErrorException();
		}

		// Get all bookable extras currently in the basket.
		$bookableExtras = $this->BookingItem->find(
			'all',
			array(
				'contain' => array(
					'BookingItemPlace'
				),
				'conditions' => array(
					'BookingItem.booking_id' => $bookingId,
					'BookingItem.is_upsell' => true
				)
			)
		);
		// Delete the temporary bookings for the extras via the API.
		foreach ($bookableExtras as $bookableExtra) {
			foreach ($bookableExtra['BookingItemPlace'] as $extra) {
				$this->deleteTemporaryBooking($extra['api_reference']);
			}
		}

		// Delete the bookable extras
		$params = array(
			'BookingItem.id' => Hash::extract($bookableExtras, '{n}.BookingItem.id'),
			'BookingItem.is_upsell' => true
		);
		return $this->BookingItem->deleteAll($params, true);
	}

	/**
	 * Add extras to the basket.
	 *
	 * @param int $bookingId
	 * @param array $bookingExtras
	 * @return bool
	 */
	protected function _addExtras($bookingId, array $bookingExtras) {
		// Make sure we don't save items with zero quantity to the database.
		foreach ($bookingExtras as $key => $val) {
			if (empty($val['quantity']) || (int)$val['quantity'] === 0) {
				unset($bookingExtras[$key]);
			}
		}

		// Get all the extras being purchased from the database so that we can
		// calculate the costs.
		$extras = $this->BookingExtra->Extra->find(
			'all',
			array(
				'conditions' => array(
					'Extra.id' => Hash::extract($bookingExtras, '{n}.extra_id')
				)
			)
		);
		$extras = Hash::combine($extras, '{n}.Extra.id', '{n}.Extra');
		foreach ($bookingExtras as &$bookingExtra) {
			$bookingExtra = $this->_generateBookingExtra($bookingId, $extras, $bookingExtra);
		}

		$data = array(
			'Booking' => array(
				'id' => $bookingId,
			),
			'BookingExtra' => $bookingExtras
		);

		return $this->saveAssociated($data, ['deep' => true]) !== false;
	}

	/**
	 * Add bookable extras to the basket.
	 *
	 * @param int $bookingId
	 * @param array $bookingItems
	 * @param array $bookableUpsells
	 * @return bool
	 */
	protected function _addBookableUpsells($bookingId, array $bookingItems, array $bookableUpsells) {
		// Make sure we don't save items with zero quantity to the database.
		foreach ($bookingItems as $key => $val) {
			if ((int)$val['quantity'] === 0) {
				unset($bookingItems[$key]);
			}
		}

		if (empty($bookingItems)) {
			return true;
		}

		$this->id = $bookingId;
		$bookingDate = substr($this->field('booking_date'), 0, 10);

		// (Important) Mark all booking items as upsells before adding to the
		// basket.
		$bookingItems = Hash::insert($bookingItems, '{n}.is_upsell', true);

		// Get all the packages being booked from the database so that we can
		// create the booking item places.
		$packages = $this->Activity->ActivityPackage->find(
			'all',
			array(
				'conditions' => array(
					'ActivityPackage.id' => Hash::extract($bookingItems, '{n}.activity_package_id')
				)
			)
		);
		$packages = Hash::combine($packages, '{n}.ActivityPackage.id', '{n}.ActivityPackage');
		foreach ($bookingItems as &$bookingItem) {
			$bookableUpsell = $bookableUpsells[$bookingItem['activity_package_id']];

			// Set costs.
			$bookingItem['item_unit_cost'] = $packages[$bookingItem['activity_package_id']]['upsell_price'];
			$bookingItem['total'] = $bookingItem['item_unit_cost'] * $bookingItem['quantity'];

			// Generate the booking item place.
			$bookingItem['BookingItemPlace'] = $this->_generateBookingItemPlace($packages, $bookingItem);
			// Generate the booking date/time.
			$bookingItem['BookingItemPlace'][0]['date_time'] = $bookingDate . ' ' . $bookableUpsell['time'] . ':00';
			// Calculate the check-in time for the activity.
			$checkInTime = strtotime($bookableUpsell['time']) - $bookableUpsell['check_in'] * 60;
			$bookingItem['BookingItemPlace'][0]['check_in_time'] = date('H:i', $checkInTime);
		}

		// Save the upsell items to the basket.
		$data = array(
			'Booking' => array(
				'id' => $bookingId
			),
			'BookingItem' => $bookingItems
		);
		if ($this->saveAssociated($data, ['deep' => true]) !== false) {

			// We need to update the basket total now as we have updated the
			// booking item prices based on the times booked.
			$this->_updateBasketTotal();

			// Make temporary bookings for the upsells (only).
			return $this->_createTemporaryBooking(-1);

		}

		return false;
	}

	/**
	 * Generates a booking extra to attach to a booking
	 *
	 * @param int $bookingId
	 * @param array $extras
	 * @param array $bookingExtra
	 * @return array
	 */
	protected function _generateBookingExtra($bookingId, array $extras, array $bookingExtra) {
		$extra = $extras[$bookingExtra['extra_id']];

		return array(
			'id' => $bookingExtra['id'],
			'booking_id' => $bookingId,
			'extra_id' => $extra['id'],
			'api_reference' => $extra['api_reference'],
			'extra_unit_cost' => $extra['price'],
			'quantity' => $bookingExtra['quantity'],
			'total' => $bookingExtra['quantity'] * $extra['price'],
		);
	}

	/**
	 * Saves an email address to a booking when a customer decides to quit the
	 * booking process.
	 *
	 * @param string $email Email address to save to the booking
	 * @return bool
	 */
	public function fallout($email) {
		$data = array(
			'id' => $this->getBasketId(),
			'email' => $email
		);

		if ($this->save($data) !== false) {
			$Event = new CakeEvent('Model.Booking.fallout', $this, array(
				'id' => $this->getBasketId(),
				'email' => $email
			));
			$this->getEventManager()->dispatch($Event);

			$this->clearBasketId();

			return true;
		}

		return false;
	}

	/**
	 * Add the customer billing details to the booking
	 *
	 * @param array $customer
	 * @return bool
	 */
	public function addBillingDetails(array $customer) {
		$data = array(
			'Booking' => array(
				'id' => $this->getBasketId()
			),
			'CustomerAddress' => $customer
		);
		return $this->saveAssociated($data) !== false;
	}

	/**
	 * Complete the purchase
	 *
	 * @param bool $paid True if payment successfully made
	 * @return bool
	 */
	public function completePurchase($paid) {
		$id = $this->getBasketId();

		$basket = $this->getBasket($id);

		$response = true;

		if ((int)$basket['Booking']['booking_state_id'] === BookingState::UNPAID) {
			if ($paid === true) {

				// @TODO api stuff
				$result = $this->_completeExtras($basket);
				if ($result === true) {

					// Complete the booking.
					$result = $this->_completeBooking(
						$basket['Booking']['sales_ref'],
						$basket['CustomerAddress']['first_name'],
						$basket['CustomerAddress']['last_name'],
						$basket['CustomerAddress']['email'],
						$basket['CustomerAddress']['telephone'],
						CustomerAddress::generateAddressArray($basket['CustomerAddress']),
						$basket['Booking']['total_cost'],
						$this->getPaymentDetails($id, 'BuzzBookings.Booking')
					);

				}

				$data = array(
					'id' => $id,
					'booking_state_id' => $result === true ? BookingState::COMPLETE : BookingState::API_FAILED,
					'completed_date' => gmdate('Y-m-d H:i:s')
				);

				$response = $this->save($data);
			} else {
				// Mark as payment failed.
				$data = array(
					'id' => $id,
					'booking_state_id' => BookingState::PAYMENT_FAILED
				);
				$response = $this->save($data);
			}

			// Clear the basket session.
			$this->clearBasketId();

			// Raise an event when purchase is complete. This will be used for
			// triggering the sending of confirmation emails (etc.).
			$Event = new CakeEvent('Model.Booking.completed', $this, array(
				'id' => $this->id
			));
			$this->getEventManager()->dispatch($Event);
		}

		return $response;
	}

	/**
	 * Add any non-bookable extras to the booking after payment has been taken.
	 *
	 * @param array $basket
	 * @return bool
	 */
	protected function _completeExtras(array $basket) {
		if (empty($basket['BookingExtra'])) {
			// Nothing to add.
			return true;
		}

		$data = [];

		foreach ($basket['BookingExtra'] as $extra) {
			$result = $this->_addItemToBooking(
				$basket['Booking']['sales_ref'],
				$extra['Extra']['api_reference'],
				$extra['quantity'],
				$extra['extra_unit_cost']
			);

			if ($result === false) {
				// API failure.
				return false;
			} else {
				$data[] = array(
					'id' => $extra['id'],
					'api_reference' => $result['api_ref']
				);
			}
		}

		return $this->saveMany($data) !== false;
	}

	/**
	 * Add an item to the booking via the API.
	 *
	 * @param int $salesRef
	 * @param string $apiRef
	 * @param int $quantity
	 * @param float $price
	 * @return array|bool
	 */
	protected function _addItemToBooking($salesRef, $apiRef, $quantity, $price) {
		return ClassRegistry::init('BuzzBookings.BookingApi')->addItemToSalesOrder(
			Configure::read('api.site_id'),
			$salesRef,
			$apiRef,
			$quantity,
			$price
		);
	}

	/**
	 * Complete a booking via the API.
	 *
	 * @param int $salesRef,
	 * @param string $firstName,
	 * @param string $lastName,
	 * @param string $email,
	 * @param string $telephone,
	 * @param array $address,
	 * @param float $totalPaid,
	 * @param array $paymentDetails,
	 * @param string $notes
	 * @return bool
	 */
	protected function _completeBooking($salesRef, $firstName, $lastName, $email, $telephone, array $address, $totalPaid, array $paymentDetails, $notes = null) {
		// Append the site to the notes to pass to the API (this is perhaps a
		// bit redundant now that Flowhouse has closed).
		$notes .= (!empty($notes) ? ' ' : '') . '(Site: ' . $_SERVER['HTTP_HOST'] . ')';

		if ($totalPaid > 0) {
			return ClassRegistry::init('BuzzBookings.BookingApi')->completeSale(
				$salesRef,
				$firstName,
				$lastName,
				$email,
				$telephone,
				$address,
				$totalPaid,
				$paymentDetails,
				$notes
			);
		} else {
			// Complete a voucher booking
			return ClassRegistry::init('BuzzBookings.BookingApi')->completeTemporaryBooking(
				$this->getBasketId(),
				$salesRef,
				$firstName,
				$lastName,
				$email,
				$telephone,
				$notes
			);
		}
	}

	/**
	 * Check if the basket contains vouchers.
	 *
	 * @param array $basket
	 * @return bool
	 */
	public function hasVouchers(array $basket) {
		foreach ($basket['BookingItem'] as $bookingItem) {
			if (!empty($bookingItem['BookingItemVoucher'])) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Restores a booking
	 *
	 * @param int $id Booking ID
	 * @param string $signature
	 * @return array|bool
	 */
	public function restoreBooking($id, $signature) {
		// Get basket.
		$data = $this->getBasket($id);
		if (!empty($data['Booking']) && $signature === $this->generateSignature($data['Booking'])) {
			// Set the basket ID.
			$this->setBasketId($id);

			$bookingDate = $data['Booking']['booking_date'];

			// Clear the temporary bookings ready for generating new ones to
			// check availability.
			$this->clearTemporaryBookings();

			// We need to attempt to create the temporary bookings for the
			// basket.
			$result = $this->_createTemporaryBooking();
			if ($result !== false) {
				// Temporary bookings recreated. Restore the booking_date
				// field.
				$this->id = $id;
				$this->saveField('booking_date', $bookingDate);
			}
			return $data;
		}
		return false;
	}

	/**
	 * Generates a booking signature. Used when returning to an abandoned booking
	 * to confirm user is who they say they are.
	 *
	 * @param array $data
	 * @return string
	 */
	public function generateSignature(array $data) {
		return md5($data['id'] . $data['email']);
	}

}
