<?php

class BookingTest extends CakeTestCase {

	public $fixtures = array(
		'plugin.buzz_bookings.Activity',
		'plugin.buzz_bookings.ActivityPackage',
		'plugin.buzz_bookings.Booking',
		'plugin.buzz_bookings.BookingExtra',
		'plugin.buzz_bookings.BookingItem',
		'plugin.buzz_bookings.BookingItemPlace',
		'plugin.buzz_bookings.BookingItemVoucher',
		'plugin.buzz_bookings.Extra',
		'plugin.buzz_theme.Theme',
		'plugin.buzz_customers.CustomerAddress',
		'plugin.buzz_customers.Country',
		'plugin.buzz_customers.UsState',
		'plugin.buzz_translate.Translation'
	);

	public function setUp() {
		parent::setUp();
		$this->Booking = ClassRegistry::init('BuzzBookings.Booking');
	}

	/**
	 * Make protected methods accessible for testing
	 *
	 * @param string $name Method name
	 * @return object
	 */
	public static function getMethod($name) {
		$Booking = new ReflectionClass('Booking');
		$method = $Booking->getMethod($name);
		$method->setAccessible(true);
		return $method;
	}

	public function testGetBasket() {
		// Test that we can retrieve a basket.
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(1));
		$result = $Booking->getBasket();
		$this->assertEquals(1, $result['Activity']['id']);
		return;
	}

	public function testGetBasketFails() {
		// Test that we can retrieve a basket.
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(3));
		$result = $Booking->getBasket();
		$this->assertEquals([], $result);
		return;
	}

	public function testCreateBasket() {
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(null));
		$result = $Booking->createBasket(
			1,
			array(
				array(
					'activity_package_id' => 1,
					'quantity' => 2
				),
				array(
					'activity_package_id' => 2,
					'quantity' => 1
				)
			)
		);
		$this->assertTrue($result);
		// Make sure the booking items have been created.
		$result = $this->Booking->BookingItem->find(
			'all',
			array(
				'contain' => array(
					'BookingItemPlace'
				),
				'conditions' => array(
					'BookingItem.booking_id' => $Booking->id
				)
			)
		);
		$this->assertEquals(2, count($result));
		// Check booking item place has been generated.
		$this->assertEquals('1', $result[0]['BookingItemPlace'][0]['api_activity_reference']);
		return;
	}

	/**
	 * Test the creation of a multi-experience basket
	 *
	 * @return void
	 */
	public function testCreateMultiExperienceBasket() {
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(null));
		$Booking->Activity = $this->getMockForModel('Activity', ['getMultiExperience']);
		$Booking->Activity->expects($this->any())->method('getMultiExperience')->will($this->returnValue(8));
		// Check the booking has been created.
		$result = $Booking->createBasket(
			1,
			array(
				array(
					'activity_package_id' => 6,
					'quantity' => 1
				)
			)
		);
		$this->assertTrue($result);
		// Now check that the correct number of booking item places have been
		// generated for the multi-expirience.
		$result = $this->Booking->BookingItem->find(
			'all',
			array(
				'contain' => array(
					'BookingItemPlace'
				),
				'conditions' => array(
					'BookingItem.booking_id' => $Booking->id
				)
			)
		);
		$this->assertEquals(2, count($result[0]['BookingItemPlace']));
		return;
	}

	/**
	 * Test _validateVoucher for first-party vouchers
	 *
	 * @return void
	 */
	public function testValidateVoucher() {
		// We need to mock a fair few methods to be able to test this as it
		// calls the API a couple of times (which we can't do for our tests).
		$Booking = $this->getMockForModel(
			'Booking',
			[
				'getBasketId',
				'getApiVoucherProvider',
				'_checkVoucherCodeApiCall',
				'_checkVoucherAvailable'
			]
		);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(null));
		$Booking->expects($this->any())->method('getApiVoucherProvider')->will($this->returnValue(8));
		$Booking->expects($this->any())->method('_checkVoucherCodeApiCall')->will($this->returnValue(array(
			'Voucher' => array(
				'participants' => 2,
				'description' => 'Fly With A Friend',
				'Activity' => array(
					0 => array(
						'api_activity_reference' => 101,
						'quantity' => 1
					)
				)
			)
		)));

		$Booking->expects($this->any())->method('_checkVoucherAvailable')->will($this->returnValue(true));
		$Booking->Activity = $this->getMockForModel('Activity', ['getMultiExperience']);
		$Booking->Activity->expects($this->any())->method('getMultiExperience')->will($this->returnValue(8));
		$validateVoucher = self::getMethod('_validateVoucher');
		$result = $validateVoucher->invokeArgs(
			$Booking,
			array(
				1,
				array(
					'voucher_provider_id' => 8,
					'voucher_code' => 2527106,
					'expiry_date' => '2016-06-01'
				)
			)
		);
		$this->assertInternalType('array', $result);

		return;
	}

	/**
	 * Test _validateVoucher for third-party vouchers
	 *
	 * @return void
	 */
	public function testValidateThirdPartyVoucher() {
		// We need to mock a fair few methods to be able to test this as it
		// calls the API a couple of times (which we can't do for our tests).
		$Booking = $this->getMockForModel(
			'Booking',
			[
				'getBasketId',
				'getApiVoucherProvider',
				'_checkThirdPartyVoucherCode',
				'_checkVoucherAvailable'
			]
		);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(null));
		$Booking->expects($this->any())->method('getApiVoucherProvider')->will($this->returnValue(8));
		$Booking->expects($this->any())->method('_checkThirdPartyVoucherCode')->will($this->returnValue(array(
			'Voucher' => array(
				'participants' => 2,
				'description' => 'Fly With A Friend',
				'Activity' => array(
					0 => array(
						'api_activity_reference' => 0,
						'api_package_reference' => 10,
						'quantity' => 1
					)
				)
			)
		)));
		$Booking->expects($this->any())->method('_checkVoucherAvailable')->will($this->returnValue(true));
		$validateVoucher = self::getMethod('_validateVoucher');
		$result = $validateVoucher->invokeArgs(
			$Booking,
			array(
				1,
				array(
					'voucher_provider_id' => 1,
					'voucher_code' => 'EXAMPLE',
					'expiry_date' => '2016-06-01',
					'api_voucher_id' => 1
				)
			)
		);
		$this->assertInternalType('array', $result);

		return;
	}

	/**
	 * Test adding a voucher to a basket
	 *
	 * @return void
	 */
	public function testAddVoucher() {
		$this->Booking = $this->getMockForModel('Booking', ['getBasketId', '_validateVoucher']);
		$this->Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(null));
		$this->Booking->expects($this->any())->method('_validateVoucher')->will($this->returnValue(array(
			'Voucher' => array(
				'participants' => 2,
				'name' => 'Fly With A Friend',
				'value' => 10,
				'Activity' => array(
					0 => array(
						'api_activity_reference' => 0,
						'api_package_reference' => 10,
						'quantity' => 1
					)
				)
			)
		)));
		$result = $this->Booking->addVoucher(
			1,
			array(
				'voucher_provider_id' => 8,
				'voucher_code' => 2527106,
				'expiry_date' => '2016-06-01',
				'value' => 10
			)
		);
		$this->assertInternalType('array', $result);

		return;
	}

	/**
	 * Test calculateUnavailableDates
	 *
	 * @return void
	 */
	public function testCalculateUnavailableDates() {
		$dates = array(
			array('date' => '2015-05-01'),
			array('date' => '2015-05-02'),
			array('date' => '2015-05-04'),
			array('date' => '2015-05-05'),
			array('date' => '2015-05-07'),
			array('date' => '2015-05-08'),
			array('date' => '2015-05-10')
		);
		$result = $this->Booking->calculateUnavailableDates($dates, '2015-05-01', '2015-05-10');
		$expected = array(
			'2015-05-03' => true,
			'2015-05-06' => true,
			'2015-05-09' => true
		);
		$this->assertEquals($expected, $result);

		return;
	}

	/**
	 * Test _getBookingActivitiesList
	 *
	 * @return void
	 */
	public function testGetBookingActivitiesList() {
		$getBookingActivitiesList = self::getMethod('_getBookingActivitiesList');
		$result = $getBookingActivitiesList->invokeArgs($this->Booking, [2]);
		$this->assertEquals(2, $result[1]);
		$this->assertEquals(1, $result[2]);

		return;
	}

	public function testAddExtras() {
		$addExtras = self::getMethod('_addExtras');
		$result = $addExtras->invokeArgs(
			$this->Booking,
			array(
				1,
				array(
					array(
						'id' => null,
						'extra_id' => 2,
						'quantity' => 1
					),
					array(
						'id' => null,
						'extra_id' => 3,
						'quantity' => 2
					)
				)
			)
		);
		$this->assertTrue($result);
		// Make sure the booking extras have been created.
		$result = $this->Booking->BookingExtra->find(
			'all',
			array(
				'contain' => array(
					'Extra'
				),
				'conditions' => array(
					'BookingExtra.booking_id' => 1
				)
			)
		);
		$this->assertCount(2, $result);
		// Check booking extra costs calculated correctly.
		$this->assertEquals(20, $result[1]['BookingExtra']['total']);
		return;
	}

	/**
	 * Test the update basket total method.
	 *
	 * @return void
	 */
	public function testUpdateBasketTotal() {
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(2));
		$updateBasketTotal = self::getMethod('_updateBasketTotal');
		$result = $updateBasketTotal->invokeArgs($Booking, []);
		$this->assertTrue($result);

		$result = $Booking->findById(2);
		$this->assertEquals(15, $result['Booking']['total_cost']);

		return;
	}

	public function testRemoveVouchers() {
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(4));
		$Booking->removeVouchers();

		$result = $Booking->BookingItem->find('count', [
			'conditions' => [
				'BookingItem.booking_id' => $Booking->getBasketId()
			]
		]);
		$this->assertEquals(1, 1);
		return;
	}

	public function testGenerateApiNotes() {
		$Booking = $this->getMockForModel('Booking', ['getBasketId']);
		$Booking->expects($this->any())->method('getBasketId')->will($this->returnValue(2));
		$basket = $Booking->getBasket();
		$result = $Booking->generateApiNotes($basket);
		$expected = 'Foo bar; Lorem ipsum';
		$this->assertEquals($expected, $result);
		return;
	}

}
